import os
import shutil
import zipfile
import hashlib
from xml.dom import minidom
import json


class PackageManager(object):

    def __init__(self, addon, download, graphql, logger, rpc, xbmc):
        self.downloader = download
        self.addon = addon
        self.logger = logger
        self.graphql = graphql
        self.xbmc = xbmc
        self.rpc = rpc

        self.load()
        self.addons_list = []
        self.installed_addons = []
        self.depset = set()
        self.buildin_addons = []
        self.downloaded_addons = []
        self.unpacked_addons = {}

        self.kodi_addons_path = xbmc.translatePath('special://home/addons/')

    def load(self):
        self.downloaded_addons = []
        self.unpacked_addons = {}

    def refresh_installed_addons(self):
        self.addons_list = []
        self.installed_addons = []
        self.depset = set()
        self.buildin_addons = []

        addons_list_full = self.rpc.get_installed_addons()
        self.logger.debug("Refresh installed addons cache")

        # Silently drop addons which are already uninstalled
        # by checking if their path is still present
        for addon in addons_list_full['result']['addons']:
            if os.path.isdir(addon["path"]):
                self.addons_list.append(addon)

        # Create a set with addons that might be installed as a pure dependency
        for addon in self.addons_list:
            if 'dependencies' in addon:
                for dependency in addon['dependencies']:
                    self.depset.add(dependency['addonid'])

        # Second loop
        for addon in self.addons_list:
            if not addon["path"].startswith(self.kodi_addons_path):
                addon['isBuildin'] = "true"
            else:
                addon['isBuildin'] = "false"

            if addon['addonid'] in self.depset:
                addon['isDep'] = "true"
            else:
                addon['isDep'] = "false"

            if addon['isBuildin'] == "true":
                self.buildin_addons.append(addon)
            else:
                self.installed_addons.append(addon)

        self.get_kodi_buildins()

    def get_kodi_buildins(self):
        # Kodi does not report all buildins. So we'll get them ourselves
        buildins_addons_path = os.path.dirname(self.buildin_addons[0]['path'])
        self.buildin_addons = []
        for root, _, files in os.walk(buildins_addons_path):
            for filename in files:
                if filename.endswith("addon.xml"):
                    path = os.path.join(root, filename)

                    xmldoc = minidom.parse(path)
                    itemlist = xmldoc.getElementsByTagName('addon')
                    for item in itemlist:
                        addon = {
                            "addonid": item.attributes['id'].value,
                            "version": item.attributes['version'].value}
                        self.buildin_addons.append(addon)

    def list_installed_addons(self):
        return self.installed_addons

    def list_buildin_addons(self):
        return self.buildin_addons

    # generate a combined hash off all available addons, used for change detection
    def generate_combined_hash(self):
        return hashlib.md5(json.dumps(self.rpc.get_installed_addons(), sort_keys=True)).hexdigest()

    def generate_buildin_hash(self):
        return hashlib.md5(json.dumps(self.buildin_addons, sort_keys=True)).hexdigest()

    def uninstall_addons(self, uninstall_list):
        if not uninstall_list:
            self.logger.error(
                "Called uninstall_addons() without uninstall list")
            return

        uninstalled = []
        try:
            for addon in uninstall_list:
                path = os.path.join(
                    self.kodi_addons_path, addon['a'])

                # NOTICE: source seems to always return an 'a' key/value lately.
                # if not addon['a']:
                #     self.logger.debug("Didn't receive an AID. Skip it.")
                #     continue

                if addon['a'] == self.addon.id:
                    self.logger.error(
                        "Can't uninstall myself. How could this happen?!")
                    continue

                if not any(y['addonid'] == addon['a'] for y in self.installed_addons):
                    self.logger.debug(
                        "Could not uninstall %s as it is buildin or already uninstalled."
                        % addon['a'])

                    uninstalled.append({"aid": addon['a']})
                    continue

                if os.path.isdir(path):
                    try:
                        self.rpc.set_addon_state(addon['a'], "false")
                        shutil.rmtree(path)
                        uninstalled.append({"aid": addon['a']})

                        if any(y['addonid'] == addon['a'] for y in self.buildin_addons):
                            self.logger.debug(
                                "Uninstalled buildin addon %s override. " % addon['a'])
                        else:
                            self.logger.debug("Uninstalled %s" % addon['a'])

                        continue

                    except OSError:
                        self.logger.error(
                            "OSError while uninstalling %s" % addon['a'])
                    except:
                        self.logger.error(
                            "Unknown exception while uninstalling %s" % addon['a'])
                        raise

                    continue

                self.logger.warning(
                    "Unexpected result when trying to uninstall %s. Path does not exist."
                    % addon['a'])

            self.logger.notice("Finished uninstalling addons")
        except Exception:
            self.logger.error("Exception raised while uninstalling addons")
            raise
        finally:
            self.logger.debug("Add uninstalled addons to payload")
            self.graphql.add_uninstalled_addons_to_payload(uninstalled)

    def create_dir(self, path):
        try:
            os.makedirs(path)
        except OSError as e:
            if not os.path.isdir(path):
                self.logger.error(
                    "Directory does not exist and could not be created: " + path)
                self.logger.error(e)
                raise

    def unpack(self, source_path, destination_path='.'):
        self.logger.debug("Unpack: " + source_path)
        try:
            with zipfile.ZipFile(source_path, 'r') as fh:
                parent_dir = os.path.join(
                    destination_path, os.path.commonprefix(fh.namelist()))
                self.create_dir(parent_dir)

                for element in fh.infolist():

                    filename = unicode(element.filename, "cp437")
                    is_directory = filename.endswith("/")

                    abs_path = os.path.abspath(
                        os.path.join(destination_path, filename))
                    abs_parent_path = os.path.dirname(abs_path)

                    # only allow files within destination_path
                    if not abs_path.startswith(os.path.abspath(destination_path)):
                        raise "File traversal in zip detected: " + source_path

                    if is_directory:
                        self.create_dir(abs_path)
                    else:
                        self.create_dir(abs_parent_path)

                        try:
                            outputfile = open(abs_path, "wb")
                            shutil.copyfileobj(
                                fh.open(element.filename), outputfile)
                        except:
                            raise

        except:
            self.logger.error("Failed to unpack: " + source_path)
            raise
        else:
            self.logger.debug("Unpacked %s to %s\n-----" %
                              (source_path, destination_path))
            return parent_dir

    def remove_unpack_dir(self, path):
        try:
            self.logger.debug("Attempt to remove temp directory: %s" % path)
            shutil.rmtree(path)
        except Exception as e:
            self.logger.debug(
                "Failed to remove temp directory: %s. Error: %s" % (path, e))
        else:
            self.logger.debug("Removed temp directory: %s" % path)

    def verify_checksum(self, source_path, checksum):
        self.logger.debug(
            "Verify checksum of downloaded file: %s" % (source_path))

        sha256_hash = hashlib.sha256()

        try:
            with open(source_path, "rb") as fh:
                # Read and update hash string value in blocks of 4K
                for byte_block in iter(lambda: fh.read(4096), b""):
                    sha256_hash.update(byte_block)

                if sha256_hash.hexdigest() == checksum.lower():
                    self.logger.debug(
                        'Checksum of file %s matches checksum %s' % (source_path, checksum))
                    return checksum

                self.logger.warning(
                    'Checksum of file %s does not match checksum %s' % (source_path, checksum))
                self.logger.debug("Remote Checksum: %s" % (checksum))
                self.logger.debug("File Checksum: %s" %
                                  (sha256_hash.hexdigest()))

                raise Exception(
                    'Checksum of file %s does not match checksum %s' % (source_path, checksum))
        except Exception as e:
            self.logger.error(
                "An error occured while verifying checksum: %s" % e)
            raise

    def prepare_install(self, addonid, checksum, source_path, destination_path):
        try:
            self.verify_checksum(source_path, checksum)
            extracted_folder = self.unpack(source_path, destination_path)
            path = os.path.join(destination_path, extracted_folder)
            self.unpacked_addons[addonid] = dict(path=path, addonid=addonid)
        except Exception as e:
            self.logger.warning(e)
            raise

    def rollback(self):
        self.logger.warning("Rollback initiated.")

        unpacked_addons_copy = dict(self.unpacked_addons)
        downloaded_addons_copy = list(self.downloaded_addons)

        try:
            for addonid, info in unpacked_addons_copy.iteritems():
                self.remove_unpack_dir(info['path'])
                del self.unpacked_addons[addonid]
            for path in downloaded_addons_copy:
                os.remove(path)
                self.downloaded_addons.remove(path)
            self.logger.warn("Rollback completed.")
        except Exception as e:
            self.logger.error(
                str(e))
            self.logger.error(
                "Rollback failed. This should never happen.")
            raise
        finally:
            self.logger.debug("Clear payload")
            self.graphql.clear_payload()

    def install(self, addon_dict):
        source_path = addon_dict['path']
        destination_path = self.kodi_addons_path

        try:
            if os.path.exists(destination_path + addon_dict['addonid']):
                shutil.rmtree(destination_path + addon_dict['addonid'])

            shutil.move(source_path, os.path.join(
                destination_path + addon_dict['addonid']))
            del self.unpacked_addons[addon_dict['addonid']]
        except Exception as e:
            self.logger.error("An error occured while installing %s. Error: %s" % (
                addon_dict['addonid'], e))
            raise
        else:
            self.logger.notice("Successfully installed %s." %
                               addon_dict['addonid'])

    def install_packages(self, packages):
        if not packages:
            return

        self.logger.debug("Install packages")
        installed = []

        try:
            for package in packages:
                if package['i'] != "buildin" and package['a'] != "service.addonscenter.sync":
                    self.logger.debug("Download addon %s" %
                                      package['a']['a'])
                    filename = '{0}-{1}.zip'.format(
                        package['a']['a'], package['v'])
                    path = '{0}/{1}'.format(package['a']['a'], filename)
                    url = '{0}/{1}'.format(package['s']['u'], path)
                    checksum = package['c']
                    dest = os.path.join(self.addon.temp_path, filename)
                    count = 0

                    try:
                        downloaded_file = self.downloader.download(
                            url, dest, False)
                        self.logger.debug(
                            "Prepare installation of addon %s" % package['a']['a'])
                        destination_path = self.addon.temp_path
                        self.prepare_install(
                            package['a']['a'],
                            checksum,
                            downloaded_file,
                            destination_path
                        )
                    except Exception:
                        self.logger.debug("Get alternative download locations for %s (checksum: %s)"
                                          % (package['a']['a'], checksum))
                        alternatives = self.graphql.get_alternative_locations(
                            checksum)
                        self.logger.debug("Alternatives %s" % alternatives)
                        downloaded_file = ''

                        found_ok_alternative = False
                        while count < len(alternatives):
                            try:
                                data_url = alternatives[count]
                                if data_url:
                                    url = '{0}/{1}'.format(data_url, path)
                                    downloaded_file = self.downloader.download(
                                        url, dest, False)
                                    self.logger.debug(
                                        "Prepare installation of addon %s" % package['a']['a'])
                                    destination_path = self.addon.temp_path

                                    try:
                                        self.prepare_install(
                                            package['a']['a'],
                                            checksum,
                                            downloaded_file,
                                            destination_path
                                        )
                                    except Exception:
                                        raise
                                    else:
                                        found_ok_alternative = True
                            except Exception:
                                count += 1
                            else:
                                break

                        if not found_ok_alternative:
                            self.logger.warning(
                                "No alternatives available (or none passed integrity test)")
                            raise "No alternatives available (or none passed integrity test)"

                    self.downloaded_addons.append(downloaded_file)

                else:
                    self.logger.debug("Buildin addon: %s" %
                                      package['a']['a'])
        except Exception:
            self.logger.warning(
                "Encountered an issue while preparing the installation.")
        else:
            self.logger.debug(
                "Addons have been checked; ready to go. Installation initiated.")
            try:
                for package in packages:
                    if package['i'] != "buildin":

                        try:
                            self.install(
                                self.unpacked_addons[package['a']['a']])
                        except:
                            self.logger.debug(
                                "An error occured while installing %s" % package['a']['a'])
                            raise

                    try:
                        install = {
                            "aid": package['a']['a'],
                            "version": package['v'],
                            "isDep": package['d']}

                        installed.append(install)

                    except Exception:
                        self.logger.error(
                            "An error occured while adding install to payload")

            except Exception:
                self.logger.warning(
                    "An error occured while installing the addons after succesfull initiation.")
            else:
                # clean up
                self.logger.debug("All requested addons have been installed")
                try:
                    for path in self.downloaded_addons:
                        if os.path.exists(path):
                            os.remove(path)
                except Exception:
                    pass

                # reset load values
                self.load()

                self.graphql.add_installed_addons_to_payload(installed)

            for package in packages:
                try:
                    self.rpc.set_addon_state(package['a']['a'], "true")
                except:
                    self.logger.debug("RPC call failed for %s." %
                                      package['a']['a'])
                    raise
