# -*- coding: utf-8 -*-
# Module: default
# Author: The One
# Created on: 22.11.2017
# License: GPL v.3 https://www.gnu.org/copyleft/gpl.html

from datetime import datetime

import os
import time


class Main(object):

    def __init__(
            self, addon, config, graphql, interval, logger, monitor,
            notify, pkg_man, rpc, state, sync_check, xbmc, xbmcaddon):

        self.addon = addon
        self.config = config
        self.state = state
        self.logger = logger
        self.notify = notify
        self.graphql = graphql
        self.monitor = monitor
        self.pkg_man = pkg_man
        self.interval = interval
        self.rpc = rpc
        self.sync_check = sync_check
        self.xbmc = xbmc
        self.xbmcaddon = xbmcaddon

        self.first_run = True
        self.sync_to_web_succeeded = False
        self.loop_interval = self.config.get("LOOP_INTERVAL") or 1

        if not os.path.exists(self.addon.user_data_path):
            self.create_userdata_path()

        if not self.state.device_token:
            self.graphql.pair_device()

    def create_userdata_path(self):
        try:
            os.makedirs(self.addon.user_data_path)
        except:
            self.logger.fatal(
                "Could not start as the User Data Path could not be created")
            raise

    def loop_should_stop(self):
        return self.monitor.abortRequested()

    def should_run(self):
        """a first run, a forced run or one that exceeds both the run- and connect-limit"""

        run_exceeded = (self.state.timestamp_last_run +
                        (self.state.addon_run_interval * 60)) < time.time()

        connect_exceeded = (self.state.timestamp_last_connect_attempt +
                            (self.state.addon_connect_interval * 60)) < time.time()

        return self.first_run \
            or self.state.force_sync \
            or (run_exceeded and connect_exceeded)

    # this is a quick loop, to act on changed environment variables
    def loop(self):
        while not self.loop_should_stop():
            try:
                # on top so a changed interval can be used immediately
                self.interval.set_run_interval()
                self.logger.load()
                with self.logger.block():
                    if self.should_run():
                        self.logger.debug("==Execute Run==")
                        self.single_run()
                    else:
                        self.logger.debug("==Skip Run==")
                        self.interval.log()
            except Exception as e:
                self.logger.error(
                    "We encountered a problem while synchronizing "
                    "with Addons.Center. See below for a hint:")
                for error in e:
                    try:
                        # iterable
                        for nested_error in error:
                            self.logger.error(
                                nested_error.get("message", nested_error))
                    except (TypeError, AttributeError):
                        try:
                            # non iterable
                            self.logger.error(
                                error.get("message", error))
                        except (TypeError, AttributeError):
                            # non dict
                            self.logger.error(e.message)
                    except Exception:
                        self.logger.fatal(e)
            finally:
                if self.state.force_sync:
                    self.state.set_force_sync("false")

            if self.monitor.waitForAbort(self.loop_interval * 60):
                break

        self.logger.notice("==Stopped %s==" % self.addon.name)

    def should_perform_full_sync(self):
        return (not self.sync_to_web_succeeded or
                self.state.installed_addons_hash != self.pkg_man.generate_combined_hash())

    def single_run(self):

        self.interval.set_timestamp_last_run()
        # execute full sync only on first run OR when the installed_addons_hash differs
        if self.should_perform_full_sync():
            try:
                self.full_sync()
                # return early when first run did not succeed
                if not self.sync_to_web_succeeded:
                    return
            except Exception:
                self.interval.double_connect_interval()
                raise

        # check if we are allowed to sync
        try:
            self.interval.set_timestamp_last_connect_attempt()
            self.state.is_sync_unlocked = self.sync_check.is_unlocked()
            # return early when not allowed to sync
            if not self.state.is_sync_unlocked:
                return
        except Exception:
            self.interval.double_connect_interval()
            raise

        self.sync_to_local()

        # first run completed
        self.first_run = False

    def full_sync(self):
        try:
            if not self.sync_to_web_succeeded:
                self.xbmc.executebuiltin('UpdateLocalAddons')
            self.state.installed_addons_hash = self.pkg_man.generate_combined_hash()
            self.logger.notice("Local -> Web: Start...")
            self.interval.set_timestamp_last_connect_attempt()
            self.pkg_man.refresh_installed_addons()
            self.sync_to_web()
        except Exception:
            self.logger.error("Local -> Web: Failed.")
            self.interval.double_connect_interval()
            raise
        else:
            self.logger.notice("Local -> Web: Success.")
            self.interval.reset_connect_interval()
            self.sync_to_web_succeeded = True
            return True

    def sync_to_local(self):
        self.logger.notice("Web -> Local: Start...")
        self.interval.set_timestamp_last_connect_attempt()

        response = self.graphql.get_web_addons()

        # return early when no response is received
        if not response:
            return

        # reset connect interval as connection was successful
        self.interval.reset_connect_interval()

        # return early when nothing to do.
        if not response["d"] and not response["i"] and not response["u"]:
            self.logger.notice(
                "Web -> Local: No pending changes.")
            self.first_run = False
            return

        # first uninstall
        if response["d"]:
            self.logger.notice(
                "Web -> Local: We need to uninstall some addons.")
            try:
                self.pkg_man.uninstall_addons(response["d"])
            except Exception:
                self.logger.error("Uninstall failed")
            else:
                self.graphql.send_payload()
                self.graphql.clear_payload()
            finally:
                self.xbmc.executebuiltin('UpdateLocalAddons')
                self.pkg_man.refresh_installed_addons()
                self.state.installed_addons_hash = self.pkg_man.generate_combined_hash()

        # followed by new installs and updates (installs may be dependencies of updated addons)
        if response["i"] or response["u"]:
            self.logger.notice(
                "Web -> Local: We need to install and/or update some addons.")
            try:
                self.pkg_man.install_packages(response["i"])
                self.pkg_man.install_packages(response["u"])
            except Exception as e:
                self.logger.error("Install or Updates failed")
                self.logger.error(e)
                self.pkg_man.rollback()
            else:
                self.graphql.send_payload()
                self.graphql.clear_payload()
            finally:
                self.xbmc.executebuiltin('UpdateLocalAddons')
                self.pkg_man.refresh_installed_addons()
                self.state.installed_addons_hash = self.pkg_man.generate_combined_hash()

    def sync_to_web(self):
        try:
            buildins_hash = self.pkg_man.generate_buildin_hash()
        except:
            self.logger.error("Failed to generate hash for buildin addons")
            raise

        self.state.timestamp_last_connect_attempt = time.time()
        if self.graphql.send_buildins_hash(buildins_hash) != "hash_unknown":
            self.graphql.sync_to_web(
                [x for x in self.pkg_man.list_installed_addons() if x['isBuildin'] == "false"])
        else:
            self.graphql.sync_buildins_to_web(
                self.pkg_man.list_buildin_addons(),
                buildins_hash)
            self.graphql.send_buildins_hash(buildins_hash)

        self.state.set_force_sync('false')
