import logging
from logging.handlers import RotatingFileHandler
from contextlib import contextmanager


# Control the maximum size and the maximum amount of log files
MAX_BYTES = 5000000
BACKUP_COUNT = 3

# Set same log levels as Kodi uses for compatibility
LOGDEBUG = 10
LOGINFO = 20
LOGNOTICE = 25
LOGWARNING = 30
LOGSEVERE = 35
LOGERROR = 40
LOGFATAL = 50

logging.addLevelName(LOGDEBUG, 'DEBUG  ')
logging.addLevelName(LOGINFO, 'INFO   ')
logging.addLevelName(LOGNOTICE, 'NOTICE ')
logging.addLevelName(LOGWARNING, 'WARNING')
logging.addLevelName(LOGSEVERE, 'SEVERE ')
logging.addLevelName(LOGERROR, 'ERROR  ')
logging.addLevelName(LOGFATAL, 'FATAL  ')

# Specify which format to use
LOGFORMAT = logging.Formatter(
    '%(asctime)s - %(name)s - %(levelname)s - %(message)s', "%Y-%m-%d %H:%M:%S")


class Logger(object):

    def __init__(self, addon, notify, state, xbmc):
        self.state = state
        self.xbmc = xbmc
        self.notify = notify

        self.log_prefix = addon.name
        self.log_path = addon.log_path
        self.logger = None
        self.log_verbose = False

        self.in_loop_mode = True
        self.in_supression_mode = False
        self.block_buffer = []
        self.line_index = 0
        self.repeat_count = 0

        self.logger = logging.getLogger(self.log_prefix)

        # setup default handler
        default_handler = RotatingFileHandler(
            self.log_path,
            maxBytes=MAX_BYTES,
            backupCount=BACKUP_COUNT,
            encoding='utf-8')
        default_handler.setFormatter(LOGFORMAT)
        self.logger.addHandler(default_handler)

        self.load()

    def load(self):

        if self.state.log_verbose:
            self.logger.setLevel(LOGDEBUG)
        else:
            self.logger.setLevel(LOGNOTICE)

        # print log setup to kodi.log and own log
        if self.log_verbose != self.state.log_verbose \
                or self.state.timestamp_last_run is None:

            self.log_verbose = self.state.log_verbose

            self.xbmc.log(
                self.log_prefix + ': Verbose logging: %s'
                % self.log_verbose, level=self.xbmc.LOGNOTICE)
            self.xbmc.log(
                self.log_prefix + ': Log Path: %s'
                % self.log_path, level=self.xbmc.LOGNOTICE)

            self.notice('Verbose logging: %s' % self.log_verbose)
            self.notice('Log Path: %s' % self.log_path)

            if self.log_verbose:
                self.notify.toast('Verbose logging', 'Log Path: %s' % self.log_path)

    @contextmanager
    def block(self):
        try:
            self.line_index = 0
            yield True
        except:
            raise
        else:
            if self.in_loop_mode:
                self.repeat_count += 1
            else:
                self.repeat_count = 0

            if self.in_loop_mode and not self.in_supression_mode:
                self.write_msg("The %s lines above were being repeated. Suppressing duplicates." %
                               str(self.line_index), LOGNOTICE)
                self.in_supression_mode = True

    def log(self, msg, level=None):

        if not level:
            level = LOGNOTICE

        if level == LOGDEBUG and not self.log_verbose:
            return

        self.line_index += 1

        if msg in self.block_buffer:

            # message in same index position in buffer
            if self.block_buffer.index(msg) == self.line_index - 1:
                self.in_loop_mode = True
                return

            # equal to the last string (repeating itself)
            elif self.block_buffer[-1] == msg:
                self.repeat_count += 1

                if self.in_supression_mode:
                    return

                self.write_msg(
                    "The line above was being repeated. Suppressing duplicates.", LOGNOTICE)
                self.in_loop_mode = True
                self.in_supression_mode = True
                return

            # in buffer but not at the same place and not the same as previous message
            else:
                real_line_index = self.block_buffer.index(msg)
                self.block_buffer = self.block_buffer[real_line_index:]
                self.line_index = 1
                self.in_loop_mode = True
                return

        if self.in_supression_mode and self.repeat_count > 0:
            #self.write_msg("CASE4: " + msg)
            self.write_msg("Suppressed %s duplicates\n " % self.repeat_count)
            self.block_buffer = []

        #self.write_msg("CASE5: " + msg)
        self.block_buffer.append(msg)
        #self.write_msg("CASE5 buffer after: " + str(self.block_buffer ))
        self.write_msg(msg, level)
        self.in_loop_mode = False
        self.in_supression_mode = False

    def write_msg(self, msg, level=None):

        if not level:
            level = LOGNOTICE

        try:
            msg = msg.replace(self.state.device_token, "[OBFUSCATED]")
        except:
            pass

        self.logger.log(level, str(msg).rstrip('\r\n'))

    def debug(self, msg):
        self.log(msg, LOGDEBUG)

    def info(self, msg):
        self.log(msg, LOGINFO)

    def notice(self, msg):
        self.log(msg, LOGNOTICE)

    def warning(self, msg):
        self.log(msg, LOGWARNING)

    def severe(self, msg):
        self.log(msg, LOGSEVERE)

    def error(self, msg):
        self.log(msg, LOGERROR)

    def fatal(self, msg):
        self.log(msg, LOGFATAL)
