import httplib
import urllib2
import json
import contextlib
import time
import ssl
import errno

import acsync.exceptions as exceptions


class Graphql(object):
    """A connection to the Graphql endpoint """

    def __init__(self, addon, config, interval, logger, notify, rpc, state, xbmc, xbmcgui):
        """Return an Graphql object"""
        self.config = config
        self.state = state
        self.logger = logger
        self.notify = notify
        self.addon = addon
        self.interval = interval
        self.rpc = rpc
        self.xbmcgui = xbmcgui
        self.xbmc = xbmc
        self.user_agent = xbmc.getUserAgent() or xbmc.getInfoLabel(
            "System.BuildVersion") + " (Windows NT 6.1;WOW64;Win64;x64;)"
        self.payload = ""

    def clear_payload(self):
        self.payload = ""

    def add_to_payload(self, query, variables=None):
        payload = {'query': query}

        if variables:
            payload['variables'] = variables

        self.payload += self.clean_payload(json.dumps(payload)).encode('utf-8')

    def clean_payload(self, payload):
        return (""
                .join(payload.replace("\\n", "").split())
                .replace('"query":"mutation', '"query":"mutation ')
                .replace('"query":"query', '"query":"query '))

    def send_payload(self, url=None):
        if not self.payload:
            return

        try:
            response = self.graphql_query(self.payload, url)
            return response
        except Exception as e:
            self.logger.warning(e)
            raise
        finally:
            self.clear_payload()

    def pair_device(self):
        pair_token = self.get_pair_token()
        self.logger.debug("Token is: %s" % pair_token)

        dialog = self.xbmcgui.Dialog()
        do_continue = dialog.yesno(
            self.addon.name,
            'Please enter the following code at manage.addons.center/pair:', '', pair_token,
            "Cancel", "Continue")

        if do_continue:
            self.state.device_token = self.get_device_token(pair_token)

            fh = open(self.addon.token_path, "w+")
            fh.write(self.state.device_token)
            fh.close()
        else:
            self.logger.notice("Canceled pairing. Stop addon and reload screen.")
            self.rpc.set_addon_state(self.addon.id, 'false')
            self.xbmc.executebuiltin('reloadskin')
            exit(0)

    def send_buildins_hash(self, buildins_hash):
        query = """
            mutation ($input:UpdateDeviceBuildinAddonsHashInput!){
                updateDeviceBuildinAddonsHash(input: $input) {
                    result,
                    device {
                        inventoryBuildinsHash
                    }
                }
            }
            """

        variables = {"input": {"device": {
            "token": self.state.device_token, "hash": buildins_hash}}}

        try:
            self.add_to_payload(query, variables)
            response = self.send_payload()

            try:
                if response["data"]["updateDeviceBuildinAddonsHash"]["result"] == "updated":
                    self.logger.debug("Buildin hash known")
                    return "updated"
                elif response["data"]["updateDeviceBuildinAddonsHash"]["result"] == "hash_unknown":
                    self.logger.debug("Buildin hash unknown")
                    return "hash_unknown"
            except:
                self.logger.debug(
                    "Failed to send buildins hash, returned data is misformed")
                raise
        except:
            self.logger.debug("Failed to send buildins hash")
            raise

    def sync_buildins_to_web(self, buildin_addons, buildins_hash):

        addon_list = map(
            lambda x: {"aid": x['addonid'], "version": x['version']}, buildin_addons)

        query = """
            mutation ($input:CreateBuildinAddonsSetInput!){
                createBuildinAddonsSet(input: $input) 
                {
                    hash,
                    result
                }
            }
            """

        variables = {"input": {"buildinAddons": {
            "hash": buildins_hash, "buildinAddons": addon_list}}}

        try:
            self.add_to_payload(query, variables)
            self.send_payload()
        except:
            self.logger.error("Could not sync buildins to web")
            raise

    def sync_to_web(self, addons_list):

        self.logger.debug("Local -> Web: Start...")

        if addons_list:
            addons = map(lambda x: {"aid": x['addonid'], "version": x['version'], "isDep": str(
                x['isDep'])}, addons_list)
            query = """
            mutation syncInstalledAddons($input:  SyncInstalledAddonsInput!){
                syncInstalledAddons(input: $input){
                    added {
                        aid, 
                        version
                    }, 
                    result
                }
            }"""

            variables = {"input": {"installedAddons": {
                "token": self.state.device_token, "addons": addons}}}

            try:
                self.add_to_payload(query, variables)
                self.send_payload()
            except:
                self.logger.error("Could not sync to web")
                raise

    def get_pair_token(self):
        query = """
            query {deviceToken{token}}
        """
        try:
            self.add_to_payload(query)
            response = self.send_payload(url=self.config.get('UNAUTH_URL'))
        except:
            self.logger.debug("Could not get pairing token")
            raise

        try:
            return response["data"]["deviceToken"]["token"]
        except:
            self.logger.error(
                "Could not get pairing token, returned data is misformed")
            raise

    def get_device_token(self, pair_token):
        query = """
            query {deviceTokenExchange(token: "%s"){token}}
        """ % pair_token

        try:
            self.add_to_payload(query)
            response = self.send_payload()
        except:
            self.logger.debug("Could not pair device")
            raise

        try:
            return response["data"]["deviceTokenExchange"]["token"]
        except:
            self.logger.error(
                "Could not pair device, returned data is misformed")
            raise

    def get_web_addons(self):
        query = """query{s(d:"%s"){i{i,o,d,a{a,n},c,
                    s{u},v},d{a,o},u{i,o,d,a{a,n},c,s{u},v}}}""" % self.state.device_token

        try:
            self.add_to_payload(query)
            response = self.send_payload()
        except:
            self.logger.warning("Could not sync device updates (Web -> Local)")
            raise

        try:
            return response["data"]["s"]
        except Exception:
            return {}

    def get_alternative_locations(self, checksum):
        query = """
            query{sa(checksum: "%s") {u}}
        """ % checksum

        try:
            self.add_to_payload(query)
            response = self.send_payload()
        except:
            self.logger.error(
                "Issues while requesting alternative locations")
            raise

        try:
            return response["data"]["sa"]["u"]
        except Exception:
            return []

    def add_installed_addons_to_payload(self, installed):

        query = """
            mutation ($input:SyncInstalledAddonsInput!){
                syncInstalledAddons(input: $input) 
                {
                    result
                }
            }
            """

        variables = {"input": {"installedAddons": {
            "token": self.state.device_token, "addons": installed}}}

        try:
            self.add_to_payload(query, variables)
        except:
            self.logger.warning.error(
                "Could not add installed addons to payload")
            raise

    def add_uninstalled_addons_to_payload(self, uninstalled):

        query = """
            mutation ($input:SyncUninstalledAddonsInput!){
                syncUninstalledAddons(input: $input) 
                {
                    result
                }
            }
            """

        variables = {"input": {"uninstalledAddons": {
            "token": self.state.device_token, "addons": uninstalled}}}

        try:
            self.add_to_payload(query, variables)
        except:
            self.logger.warning.error(
                "Could not add uninstalled addons to payload")
            raise

    def graphql_query(self, data, url):

        if not url:
            url = self.config.get('BASE_URL')

        self.logger.debug("===== Send payload ========")
        self.logger.debug(url)
        self.logger.debug(data)

        request = urllib2.Request(url, data=data)
        request.add_header("Content-Type", 'application/json')
        request.add_header("User-Agent", self.user_agent)
        request.add_header("Accept", 'application/json')
        request.add_header("Accept-Charset", 'utf-8')
        request.get_method = lambda: "POST"

        if self.config.get('ALLOW_INSECURE_CONNECTION') == "true":
            context = ssl._create_unverified_context()
        else:
            context = ssl.create_default_context()

        context.options = (ssl.OP_NO_TLSv1 | ssl.OP_NO_TLSv1_1 |
                           ssl.OP_NO_COMPRESSION | ssl.PROTOCOL_TLSv1_2)
        handler = urllib2.HTTPSHandler(context=context)
        urllib2.build_opener(handler)

        try:
            with contextlib.closing(urllib2.urlopen(request, context=context)) as connection:
                raw_response = connection.read()

                response = json.loads(raw_response)

                errors = response.get("errors", False)

                if errors:
                    # Prevents stacking of modals by closing them every run based on interval
                    modal_timeout = max([
                        self.state.timestamp_last_run or 0 +
                        (self.state.addon_run_interval * 60),
                        self.state.timestamp_last_connect_attempt or 0 +
                        (self.state.addon_connect_interval * 60)
                    ]) - time.time()

                    for error in errors:
                        if (error.get("title", None) or error.get("message", None)):
                            if modal_timeout > 0:
                                self.notify.modal("AC Sync: " + error.get("title", "Error"),
                                                  line1=error.get(
                                                      "message", ""),
                                                  nolabel="Nope",
                                                  yeslabel="yes",
                                                  autoclose=int(modal_timeout * 1000))
                            raise Exception(error.get("title", "") +
                                            " | " + error.get("message", ""))

                        else:
                            self.notify.modal(
                                heading=self.addon.name,
                                line1="An unknown error occured. See log for details",
                                line2=self.addon.log_path,
                                nolabel="Nope",
                                yeslabel="yes",
                                autoclose=int(30 * 1000))
                            raise Exception("No message was given")

                self.interval.reset_connect_interval()
                return response
        except urllib2.HTTPError as e:
            raise exceptions.ServerError(
                "The server couldn\'t fulfill the request. Code:%s" % e.code)
        except urllib2.URLError as e:
            if hasattr(e, 'reason') and hasattr(e.reason, 'errno'):
                if e.reason.errno == errno.EHOSTDOWN:
                    raise exceptions.RemoteOfflineError(
                        "Could not connect to Addons.Center. "
                        "It is probably offline. "
                        "Check https://downforeveryoneorjustme.com/api.addons.center. "
                        "Reason: %s" % e.reason)
                else:
                    raise exceptions.UndefinedConnectionError(
                        "An error occured while connecting to Addons.Center. "
                        "Check your internet connection "
                        "and https://downforeveryoneorjustme.com/api.addons.center. "
                        "Reason: %s" % e.reason)
            else:
                raise exceptions.ServerError(
                    "The server couldn\'t fulfill the request. "
                    "See below:\n>> %s" % e)
        except httplib.BadStatusLine as e:
            raise exceptions.UndefinedConnectionError(
                "An error occured while connecting to Addons.Center. "
                "Check your internet connection "
                "and https://downforeveryoneorjustme.com/api.addons.center. "
                "Reason:\n>> %s" % e)
        except Exception as e:
            raise exceptions.UndefinedConnectionError(
                "Encountered undefined exception accessing HTTP(S) server. "
                "See below:\n>> %s" % e)
