package org.ovirt.engine.core.vdsbroker.libvirt;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;

import org.apache.commons.lang.StringUtils;
import org.ovirt.engine.core.common.businessentities.VmDevice;
import org.ovirt.engine.core.vdsbroker.builder.vminfo.LibvirtVmXmlBuilder;

/**
 * This class represents an XML in the form of:
 * <hotunplug>
 *   <devices>
 *     <disk>
 *       <alias name='mydisk'/>
 *     </disk>
 *   </devices>
 * </hotunplug>
 */
@XmlRootElement
public class Hotunplug {
    private Devices devices;

    public Devices getDevices() {
        return devices;
    }

    @XmlElement
    public Hotunplug setDevices(Devices devices) {
        this.devices = devices;
        return this;
    }

    public static class Devices  {
        private Device nic;
        private Device disk;

        public Device getInterface() {
            return nic;
        }

        @XmlElement
        public Devices setInterface(Device nic) {
            this.nic = nic;
            return this;
        }

        public Device getDisk() {
            return disk;
        }

        @XmlElement
        public Devices setDisk(Device disk) {
            this.disk = disk;
            return this;
        }
    }

    public static class Device {
        private Alias alias;

        public Device(VmDevice device) {
            /**
             * For VMs that were started in lower compatibility versions and that don't support user-aliases, we should
             * get the reported alias from the database, otherwise we have no way to tell what is the device alias since
             * it is generated by libvirt.
             * For newly created VMs, it may happen that the alias is not set in the database yet and in that case a
             * user alias needs to be set here and passed to libvirt with the hot unplug request. It will later be updated
             * in the database by the vm device monitoring.
             */
            String alias = StringUtils.isNotEmpty(device.getAlias()) ? device.getAlias() :
                    LibvirtVmXmlBuilder.generateUserAliasForDevice(device);
            this.alias = new Alias(alias);
        }

        public Alias getAlias() {
            return alias;
        }

        @XmlElement
        public void setAlias(Alias alias) {
            this.alias = alias;
        }
    }

    private static class Alias {
        private String name;

        public Alias(String name) {
            this.name = name;
        }

        public String getName() {
            return name;
        }

        @XmlAttribute
        public void setName(String name) {
            this.name = name;
        }
    }
}
