/*
 * Copyright (c) 2021, 2025 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */

package org.eclipse.trace4cps.ui.view;

import java.util.Collection;
import java.util.function.Supplier;

import org.eclipse.trace4cps.core.IClaim;
import org.eclipse.trace4cps.core.IDependency;
import org.eclipse.trace4cps.core.IEvent;
import org.eclipse.trace4cps.core.IPsopFragment;
import org.eclipse.trace4cps.core.ITSSeries;
import org.eclipse.trace4cps.core.TracePart;
import org.eclipse.trace4cps.core.impl.TraceHelper;
import org.eclipse.trace4cps.vis.jfree.TraceViewConfiguration;
import org.jfree.chart.labels.XYToolTipGenerator;
import org.jfree.data.xy.XYDataset;

/**
 * Works together with the {@link EclipseDataItemFactory}.
 */
public class EclipseToolTipGenerator implements XYToolTipGenerator {
    private final Supplier<TraceViewConfiguration> viewConfigSupplier;

    public EclipseToolTipGenerator(Supplier<TraceViewConfiguration> viewConfigSupplier) {
        this.viewConfigSupplier = viewConfigSupplier;
    }

    private Collection<String> getDescribingAttributes(TracePart part) {
        return viewConfigSupplier.get().getDescribingAttributes(part);
    }

    @Override
    public String generateToolTip(XYDataset ds, int series, int item) {
        EventDataItem eDataItem = EventDataItem.getFrom(ds, series, item);
        if (eDataItem != null) {
            IEvent e = (IEvent)eDataItem.getBackReference().getSelectedTraceItem();
            return TraceHelper.getValues(e, getDescribingAttributes(TracePart.EVENT), false);
        }
        DependencyDataItem dDataItem = DependencyDataItem.getFrom(ds, series, item);
        if (dDataItem != null) {
            IDependency d = (IDependency)dDataItem.getBackReference().getSelectedTraceItem();
            return TraceHelper.getValues(d, getDescribingAttributes(TracePart.DEPENDENCY), false);
        }
        ClaimDataItem cDataItem = ClaimDataItem.getFrom(ds, series, item);
        if (cDataItem != null) {
            IClaim c = (IClaim)cDataItem.getBackReference().getSelectedTraceItem();
            return TraceHelper.getValues(c, getDescribingAttributes(TracePart.CLAIM), false);
        }
        SignalDataItem sDataItem = SignalDataItem.getFrom(ds, series, item);
        if (sDataItem != null) {
            IPsopFragment f = (IPsopFragment)sDataItem.getBackReference().getSelectedTraceItem();
            return "Signal fragment: " + f.dom() + " : " + f.getC() + ", " + f.getB() + ", " + f.getA() + " ("
                    + f.getShape() + ")";
        }
        TimeSeriesDataItem tsDataItem = TimeSeriesDataItem.getFrom(ds, series, item);
        if (tsDataItem != null) {
            ITSSeries ts = (ITSSeries)tsDataItem.getBackReference().getSelectedTraceItem();
            String name = ts.getAttributes().getOrDefault("name", "Value");
            return "<html><body>Time = " + tsDataItem.getX().toString() + "<br>" + name + " = "
                    + tsDataItem.getY().toString() + "</body></html>";
        }
        return ds + " " + series + " " + item;
    }
}
