/**
 * Copyright (c) 2016 TypeFox GmbH (http://www.typefox.io) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.elk.graph.text.conversion;

import com.google.inject.Inject;
import com.google.inject.Provider;
import java.util.List;
import org.eclipse.elk.core.data.LayoutMetaDataService;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.text.services.ElkGraphGrammarAccess;
import org.eclipse.xtext.TerminalRule;
import org.eclipse.xtext.conversion.ValueConverterException;
import org.eclipse.xtext.conversion.impl.AbstractValueConverter;
import org.eclipse.xtext.conversion.impl.IDValueConverter;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

@SuppressWarnings("all")
public class PropertyKeyValueConverter extends AbstractValueConverter<IProperty<?>> {
  private IDValueConverter idValueConverter;
  
  @Inject
  public void initialize(final Provider<IDValueConverter> idValueConverterProvider, final ElkGraphGrammarAccess grammarAccess) {
    IDValueConverter _get = idValueConverterProvider.get();
    final Procedure1<IDValueConverter> _function = new Procedure1<IDValueConverter>() {
      @Override
      public void apply(final IDValueConverter it) {
        TerminalRule _iDRule = grammarAccess.getIDRule();
        it.setRule(_iDRule);
      }
    };
    IDValueConverter _doubleArrow = ObjectExtensions.<IDValueConverter>operator_doubleArrow(_get, _function);
    this.idValueConverter = _doubleArrow;
  }
  
  @Override
  public String toString(final IProperty<?> value) throws ValueConverterException {
    if ((value == null)) {
      throw new ValueConverterException("IProperty value may not be null.", null, null);
    }
    String _id = value.getId();
    List<String> _split = Strings.split(_id, ".");
    final Function1<String, String> _function = new Function1<String, String>() {
      @Override
      public String apply(final String it) {
        return PropertyKeyValueConverter.this.idValueConverter.toString(it);
      }
    };
    List<String> _map = ListExtensions.<String, String>map(_split, _function);
    return IterableExtensions.join(_map, ".");
  }
  
  @Override
  public IProperty<?> toValue(final String string, final INode node) throws ValueConverterException {
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(string);
    if (_isNullOrEmpty) {
      throw new ValueConverterException("Cannot convert empty string to a property idenfifier.", node, null);
    }
    List<String> _split = Strings.split(string, ".");
    final Function1<String, String> _function = new Function1<String, String>() {
      @Override
      public String apply(final String it) {
        return PropertyKeyValueConverter.this.idValueConverter.toValue(it, node);
      }
    };
    List<String> _map = ListExtensions.<String, String>map(_split, _function);
    final String idSuffix = IterableExtensions.join(_map, ".");
    LayoutMetaDataService _instance = LayoutMetaDataService.getInstance();
    final LayoutOptionData optionData = _instance.getOptionDataBySuffix(idSuffix);
    if ((optionData == null)) {
      throw new ValueConverterException((("No layout option with identifier \'" + idSuffix) + "\' can be found."), node, null);
    }
    return optionData;
  }
}
