package Bencher::Scenario::ShellGuess::Overhead;

our $DATE = '2021-07-31'; # DATE
our $VERSION = '0.003'; # VERSION

use 5.010001;
use strict;
use warnings;

our $scenario = {
    summary => 'Benchmark the startup overhead of guessing shell',
    participants => [
        {
            name => 'load Shell::Guess',
            perl_cmdline => ['-MShell::Guess', '-e1'],
        },
        {
            name => 'load Shell::Guess + running_shell',
            perl_cmdline => ['-MShell::Guess', '-e', '$sh = Shell::Guess->running_shell'],
        },
        {
            name => 'perl (baseline)',
            perl_cmdline => ['-e1'],
        },
    ],
};

1;
# ABSTRACT: Benchmark the startup overhead of guessing shell

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::ShellGuess::Overhead - Benchmark the startup overhead of guessing shell

=head1 VERSION

This document describes version 0.003 of Bencher::Scenario::ShellGuess::Overhead (from Perl distribution Bencher-Scenarios-ShellGuess), released on 2021-07-31.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m ShellGuess::Overhead

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARK PARTICIPANTS

=over

=item * load Shell::Guess (command)



=item * load Shell::Guess + running_shell (command)



=item * perl (baseline) (command)



=back

=head1 BENCHMARK SAMPLE RESULTS

=head2 Sample benchmark #1

Run on: perl: I<< v5.24.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark command (under bash):

 % 

Result formatted as table:

 #table1#
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                       | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | load Shell::Guess + running_shell |        98 |      10   |    0                  |      3.74052667964341 | 3.8e-05 |      21 |
 | load Shell::Guess                 |       100 |       9.7 |    0.0517047248148278 |      3.50746922381476 |   3e-05 |      20 |
 | perl (baseline)                   |       460 |       2.2 |    3.74052667964341   |      0                | 1.1e-05 |      20 |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                                      Rate  load Shell::Guess + running_shell  load Shell::Guess  perl (baseline) 
  load Shell::Guess + running_shell   98/s                                 --                -3%             -78% 
  load Shell::Guess                  100/s                                 3%                 --             -77% 
  perl (baseline)                    460/s                               354%               340%               -- 
 
 Legends:
   load Shell::Guess: participant=load Shell::Guess
   load Shell::Guess + running_shell: participant=load Shell::Guess + running_shell
   perl (baseline): participant=perl (baseline)

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


=head2 Sample benchmark #2

Benchmark command (under fish):

 % 

Result formatted as table:

 #table2#
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                       | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | load Shell::Guess + running_shell |        99 |      10   |    0                  |      4.50965048854215 | 4.6e-05 |      21 |
 | load Shell::Guess                 |       100 |       9.7 |    0.0452853002877057 |      4.27095376451354 | 3.4e-05 |      20 |
 | perl (baseline)                   |       550 |       1.8 |    4.50965048854215   |      0                | 6.6e-06 |      21 |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                                      Rate  load Shell::Guess + running_shell  load Shell::Guess  perl (baseline) 
  load Shell::Guess + running_shell   99/s                                 --                -3%             -82% 
  load Shell::Guess                  100/s                                 3%                 --             -81% 
  perl (baseline)                    550/s                               455%               438%               -- 
 
 Legends:
   load Shell::Guess: participant=load Shell::Guess
   load Shell::Guess + running_shell: participant=load Shell::Guess + running_shell
   perl (baseline): participant=perl (baseline)

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


=head2 Sample benchmark #3

Benchmark command (under tcsh ('c')):

 % 

Result formatted as table:

 #table3#
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                       | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | load Shell::Guess + running_shell |       100 |      10   |    0                  |      3.66106761739065 | 4.6e-05 |      20 |
 | load Shell::Guess                 |       100 |       9.7 |    0.0311256049185042 |      3.52036841598851 | 4.7e-05 |      20 |
 | perl (baseline)                   |       500 |       2   |    3.66106761739065   |      0                | 4.7e-05 |      20 |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                                      Rate  load Shell::Guess + running_shell  load Shell::Guess  perl (baseline) 
  load Shell::Guess + running_shell  100/s                                 --                -3%             -80% 
  load Shell::Guess                  100/s                                 3%                 --             -79% 
  perl (baseline)                    500/s                               400%               384%               -- 
 
 Legends:
   load Shell::Guess: participant=load Shell::Guess
   load Shell::Guess + running_shell: participant=load Shell::Guess + running_shell
   perl (baseline): participant=perl (baseline)

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


=head2 Sample benchmark #4

Benchmark command (under zsh ('z')):

 % 

Result formatted as table:

 #table4#
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                       | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | load Shell::Guess                 |       100 |        10 |    0                  |      4.17013158922127 |   0.00011 |      20 |
 | load Shell::Guess + running_shell |        99 |        10 |    0.0309140475235712 |      4.01509471293053 |   5e-05   |      20 |
 | perl (baseline)                   |       500 |         2 |    4.17013158922127   |      0                | 2.6e-05   |      20 |
 +-----------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                                      Rate  load Shell::Guess  load Shell::Guess + running_shell  perl (baseline) 
  load Shell::Guess                  100/s                 --                                 0%             -80% 
  load Shell::Guess + running_shell   99/s                 0%                                 --             -80% 
  perl (baseline)                    500/s               400%                               400%               -- 
 
 Legends:
   load Shell::Guess: participant=load Shell::Guess
   load Shell::Guess + running_shell: participant=load Shell::Guess + running_shell
   perl (baseline): participant=perl (baseline)

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenarios-ShellGuess>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenarios-ShellGuess>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenarios-ShellGuess>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2017, 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
