#!/usr/bin/perl
use strict;
use warnings;

use Test::More tests => 89; #qw(no_plan);

BEGIN {
    use_ok('AlignDB::IntSpan');
}

{
    my @at_test = (
        [ "-",          1,   undef ],
        [ "-",          -1,  undef ],
        [ "1-10,20-30", 25,  undef ],
        [ "1-10,20-30", -25, undef ],

        [ "0-9", 1,  0 ],
        [ "0-9", 6,  5 ],
        [ "0-9", 10, 9 ],
        [ "0-9", 11, undef ],

        [ "0-9", -1,  9 ],
        [ "0-9", -5,  5 ],
        [ "0-9", -10, 0 ],
        [ "0-9", -11, undef ],

        [ "1-10,21-30,41-50", 6,   6 ],
        [ "1-10,21-30,41-50", 16,  26 ],
        [ "1-10,21-30,41-50", 26,  46 ],
        [ "1-10,21-30,41-50", 31,  undef ],
        [ "1-10,21-30,41-50", -1,  50 ],
        [ "1-10,21-30,41-50", -11, 30 ],
        [ "1-10,21-30,41-50", -21, 10 ],
        [ "1-10,21-30,41-50", -30, 1 ],
        [ "1-10,21-30,41-50", -31, undef ],
    );

    for ( my $i = 1; $i <= @at_test; $i++ ) {
        my ( $runlist, $index, $expact ) = @{ $at_test[ $i - 1 ] };
        my $set = AlignDB::IntSpan->new($runlist);

        is( $set->at($index), $expact, "at_$i" );
    }
}

{
    my @index_test = (
        [ "-",          1,   undef ],
        [ "1-10,21-30", 25,  15 ],

        [ "0-9", 1,  2 ],
        [ "0-9", 6,  7 ],
        [ "0-9", 9, 10 ],
        [ "0-9", 11, undef ],

        [ "1-10,21-30,41-50", 6,   6 ],
        [ "1-10,21-30,41-50", 16,  undef ],
        [ "1-10,21-30,41-50", 21,  11 ],
        [ "1-10,21-30,41-50", 26,  16 ],
        [ "1-10,21-30,41-50", 30,  20 ],
    );

    for ( my $i = 1; $i <= @index_test; $i++ ) {
        my ( $runlist, $index, $expact ) = @{ $index_test[ $i - 1 ] };
        my $set = AlignDB::IntSpan->new($runlist);

        is( $set->index($index), $expact, "index_$i" );
    }
}

{
    my @splice_test = (

        # empty sets
        [ "-", 1, undef, "-" ],
        [ "-", 1, 0,     "-" ],
        [ "-", 1, 1,     "-" ],

        # empty slices
        [ "1-10", 11, 1, "-" ],
        [ "1-10", 5,  0, "-" ],

        # positive offset, no length
        [ "1-10", 1,  undef, "1-10" ],
        [ "1-10", 2,  undef, "2-10" ],
        [ "1-10", 3,  undef, "3-10" ],
        [ "1-10", 10, undef, "10" ],
        [ "1-10", 11, undef, "-" ],

        # positive offset, positive length
        [ "1-10", 1,  3,  "1-3" ],
        [ "1-10", 6,  3,  "6-8" ],
        [ "1-10", 10, 1,  "10" ],
        [ "1-10", 11, 1,  "-" ],
        [ "1-10", 1,  20, "1-10" ],

        # positive offset, no length
        [ "1-10,21-30,41-50", 1,  undef, "1-10,21-30,41-50" ],
        [ "1-10,21-30,41-50", 10, undef, "10,21-30,41-50" ],
        [ "1-10,21-30,41-50", 11, undef, "21-30,41-50" ],
        [ "1-10,21-30,41-50", 30, undef, "50" ],
        [ "1-10,21-30,41-50", 31, undef, "-" ],

        # positive offset, positive length
        [ "1-10,21-30,41-50", 1,  1, "1" ],
        [ "1-10,21-30,41-50", 10, 1, "10" ],
        [ "1-10,21-30,41-50", 11, 1, "21" ],
        [ "1-10,21-30,41-50", 20, 1, "30" ],
        [ "1-10,21-30,41-50", 30, 1, "50" ],

        [ "1-10,21-30,41-50", 1, 10, "1-10" ],
        [ "1-10,21-30,41-50", 1, 11, "1-10,21" ],
        [ "1-10,21-30,41-50", 1, 20, "1-10,21-30" ],
        [ "1-10,21-30,41-50", 1, 21, "1-10,21-30,41" ],
        [ "1-10,21-30,41-50", 1, 30, "1-10,21-30,41-50" ],

        [ "1-10,21-30,41-50", 6, 10, "6-10,21-25" ],
        [ "1-10,21-30,41-50", 6, 11, "6-10,21-26" ],
        [ "1-10,21-30,41-50", 6, 20, "6-10,21-30,41-45" ],
        [ "1-10,21-30,41-50", 6, 21, "6-10,21-30,41-46" ],
        [ "1-10,21-30,41-50", 6, 30, "6-10,21-30,41-50" ],

        [ "1-10,21-30,41-50", 16, 10, "26-30,41-45" ],
        [ "1-10,21-30,41-50", 16, 11, "26-30,41-46" ],
        [ "1-10,21-30,41-50", 16, 20, "26-30,41-50" ],
        [ "1-10,21-30,41-50", 16, 21, "26-30,41-50" ],
        [ "1-10,21-30,41-50", 16, 30, "26-30,41-50" ],

        [ "1-10,21-30,41-50", 26, 1, "46" ],
        [ "1-10,21-30,41-50", 26, 2, "46-47" ],
        [ "1-10,21-30,41-50", 26, 3, "46-48" ],
        [ "1-10,21-30,41-50", 26, 5, "46-50" ],
        [ "1-10,21-30,41-50", 26, 6, "46-50" ],
    );

    for ( my $i = 1; $i <= @splice_test; $i++ ) {
        my ( $runlist, $offset, $length, $expected )
            = @{ $splice_test[ $i - 1 ] };

        my $set = AlignDB::IntSpan->new($runlist);
        my $actual = $set->_splice( $offset, $length )->as_string();

        is( $actual, $expected, "splice_$i" );
    }
}

{

    my @slice_test = (

        # empty sets
        [ "-", 0, 0, "-" ],
        [ "-", 0, 1, "-" ],

        # empty slices
        [ "1-10", 11, 11, "-" ],
        [ "1-10", 5,  4,  "-" ],

        # positive indices
        [ "1-10", 1,  3,  "1-3" ],
        [ "1-10", 6,  8,  "6-8" ],
        [ "1-10", 8,  10, "8-10" ],
        [ "1-10", 10, 10, "10" ],
        [ "1-10", 11, 11, "-" ],
        [ "1-10", 1,  10, "1-10" ],
        [ "1-10", 1,  20, "1-10" ],
    );

    for ( my $i = 1; $i <= @slice_test; $i++ ) {
        my ( $runlist, $from, $to, $expected )
            = @{ $slice_test[ $i - 1 ] };

        my $set = AlignDB::IntSpan->new($runlist);
        my $actual = $set->slice( $from, $to )->as_string();

        is( $actual, $expected, "slice_$i" );
    }

}
