// StarPlot - A program for interactively viewing 3D maps of stellar positions.
// Copyright (C) 2000  Kevin B. McCarty
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// stringops.cc

#include "stringops.h"

// If necessary, (re)define toupper, tolower.

#ifdef NO_TOLOWER
char std::tolower(char c)
{
  if (c >= 'A' && c <= 'Z') return c + 'a' - 'A';
  else                      return c;
}
#endif

#ifdef NO_TOUPPER
char std::toupper(char c)
{
  if (c >= 'a' && c <= 'z') return c + 'A' - 'a';
  else                      return c;
}
#endif

// If necessary, define non-ANSI string functions.  This is ugly, so fixes
//  are welcome.

#ifdef NO_SNPRINTF
int snprintf(char * string, size_t size, const char * format, ...)
{
  char * buffer;
  int length;
  FILE * devnull = fopen("/dev/null", "w");
  va_list args;
  va_start(args, format);

  // there's no ANSI function to output the length of a printf-formatted
  //  string without actually creating such output, so this is a kludge.
  length = vfprintf(devnull, format, args);

  buffer = new char[length + 1];
  vsprintf(buffer, format, args);
  strncpy(string, buffer, size);

  // clean up
  va_end(args);
  fclose(devnull);
  delete [] buffer;

  // according to the Linux Programmer's Manual, the ISO C99 standard has
  //  snprintf returning the same number for a given format string as would
  //  any other member of the printf family, even if length > size.
  return length; 
}
#endif // #ifdef NO_SNPRINTF


#ifdef NO_STRCASECMP
int strcasecmp(const char * s1, const char * s2)
{
  char * s1a = new char[strlen(s1) + 1];
  char * s2a = new char[strlen(s2) + 1];
  int result;

  for (unsigned int i = 0; i <= strlen(s1); s1a[i] = toupper(s1[i]), i++) ;
  for (unsigned int i = 0; i <= strlen(s2); s2a[i] = toupper(s2[i]), i++) ;
  result = strcmp(s1a, s2a);

  delete [] s1a;
  delete [] s2a;
  return result;
}

int strncasecmp(const char * s1, const char * s2, size_t n)
{
  char * s1a = new char[strlen(s1) + 1];
  char * s2a = new char[strlen(s2) + 1];
  int result;

  for (unsigned int i = 0; i <= strlen(s1); s1a[i] = toupper(s1[i]), i++) ;
  for (unsigned int i = 0; i <= strlen(s2); s2a[i] = toupper(s2[i]), i++) ;
  result = strncmp(s1a, s2a, n);

  delete [] s1a;
  delete [] s2a;
  return result;
}
#endif // #ifdef NO_STRCASECMP


// A couple other useful string functions

// isempty(): returns true if a string is "" or consists only of whitespace.
bool isempty(const char * string)
{
  for (unsigned int i = 0; i < strlen(string); i++)
    if (! isspace(string[i]))
      return false;
  return true;
}


// stripspace(): removes leading and trailing whitespace from a string.
char * stripspace(char * string)
{
  unsigned int i = 0, j = 0;

  while (isspace(string[j])) j++;
  for (i = 0; i + j <= strlen(string); i++)
    string[i] = string[i + j];
  if (strlen(string))
    for (i = strlen(string) - 1; i > 0; i--) {
      if (isspace(string[i])) string[i] = 0;
      else break;
    }
  return string;
}


// We assume the strings we output to in these functions are big enough to
//  hold the results, which are guaranteed to be no longer than 15 characters,
//  including the terminating '\0', for the single string functions
//  and no longer than 5 characters each for the multiple string functions.

void RadiansToRAStrings(double phi, char * hours, char * minutes, 
                        char * seconds, bool celestial_coords)
{
  double ra_hms = phi / M_PI_12;
  int h, m;
  float s;

  if (! (hours && minutes && seconds)) return;

  ra_hms += 1e-8;          // fix a small rounding issue
  if (! celestial_coords)
    ra_hms *= 15.0;

  h = (int)floor(ra_hms);
  m = (int)floor(60.0 * (ra_hms - h));
  s = 3600.0 * (ra_hms - h) - 60.0 * m;
  snprintf(hours, 5, (celestial_coords ? "%02d" : "%03d"), h);
  snprintf(minutes, 5, "%02d", m);
  snprintf(seconds, 5, "%04.1f", s);
  hours[4] = minutes[4] = seconds[4] = 0;

  return;
}


void RadiansToDecStrings(double theta, char * deg, char * min, char * sec)
{
  double dec_dms = theta / M_PI_180;
  int d, m;
  float s;
  char sign = '+';

  if (! (deg && min && sec)) return;

  if (dec_dms < 0.0) {
    sign = '-';
    dec_dms *= -1;
  }
  if (dec_dms == 0.0) sign = ' ';

  dec_dms += 1e-8; // rounding issue again
  d = (int)floor(dec_dms);
  m = (int)floor(60.0 * (dec_dms - d));
  s = 3600.0 * (dec_dms - d) - 60.0 * m;
  snprintf(deg, 5, "%c%02d", sign, d);
  snprintf(min, 5, "%02d", m);
  snprintf(sec, 5, "%04.1f", s);
  deg[4] = min[4] = sec[4] = 0;

  return;
}


void RadiansToRAString(double phi, char * output, bool celestial_coords,
		       bool punctuation, char sep)
{
  char hrs[5], min[5], sec[5];

  if (!output) return;
  RadiansToRAStrings(phi, hrs, min, sec, celestial_coords);

  if (punctuation) {
    if (celestial_coords)
      snprintf(output, 15, "%sh%c%sm%c%ss", hrs, sep, min, sep, sec);
    else
      snprintf(output, 15, "%s" DEGREE_SYMBOL "%c%s'%c%s\"",
	       hrs, sep, min, sep, sec);
  }
  else
    snprintf(output, 15, "%s%c%s%c%s", hrs, sep, min, sep, sec);
  output[14] = 0;
  return;
}


void RadiansToDecString(double theta, char * output, 
			bool punctuation, char sep)
{
  char deg[5], min[5], sec[5];

  if (!output) return;
  RadiansToDecStrings(theta, deg, min, sec);

  if (punctuation)
    snprintf(output, 15, "%s" DEGREE_SYMBOL "%c%s'%c%s\"", 
	     deg, sep, min, sep, sec);
  else
    snprintf(output, 15, "%s%c%s%c%s", deg, sep, min, sep, sec);
  output[14] = 0;

  return;
}


// Finally, the StringList class definitions! ------------------

// This constructor "tokenizes" a string into a StringList.
//  Beware: "ssdf;fdas;;dasf" has three tokens, not four, since
//  strtok() ignores the double semicolon.
StringList::StringList(const char *cstring, char tokenizer)
{
  char delim[2] = { tokenizer, 0 };
  char *word, *datastring = new char[std::strlen(cstring) + 1];
  strcpy(datastring, cstring);

  if ( (word = strtok(datastring, delim)) ) {
    do { stringlist.push_back(string(word)); }
    while ((word = strtok((char *)0, delim)) != (char *)0) ;
  }
  delete [] datastring;
}

// flatten(): return a string consisting of the elements of the StringList,
// in order, separated by the spacer character
string StringList::flatten(char spacer)
{
  string returnstring = string("");
  for (unsigned int i = 0; i + 1 < size(); i++) {
    returnstring += stringlist[i];
    returnstring += spacer;
  }
  returnstring += stringlist[size() - 1];
  return returnstring;
}

// insert(): insert the given string into the given position, pushing
//  all strings at that position or greater one position higher in the list.
bool StringList::insert(const char *cstring, unsigned int posn)
{
  if (posn > stringlist.size()) return false;
  else if (posn == stringlist.size()) append(cstring);
  else {
    string temp = string(cstring);
    stringlist.insert(stringlist.begin() + posn, temp);
  }
  return true;
}


// set(): change the string at the given position to the given string.
bool StringList::set(const char *cstring, unsigned int posn)
{
  if (posn >= stringlist.size()) return false;
  else stringlist[posn] = string(cstring);
  return true;
}


// remove(): remove the string at the given position from the array, moving
//  back all strings at greater positions.
bool StringList::remove(unsigned int posn)
{
  if (posn >= stringlist.size()) return false;
  else stringlist.erase(stringlist.begin() + posn);
  return true;
}


// stripspace(): This function removes white space from the beginning and
//  end of all the strings in the StringList.
StringList StringList::stripspace()
{
  for (unsigned int i = 0; i < stringlist.size(); i++) {
    char * temp = new char[std::strlen(stringlist[i].c_str()) + 1];
    strcpy(temp, stringlist[i].c_str());
    stringlist[i] = string(::stripspace(temp));
    delete [] temp;
  }
  return *this;
}


// touppercase(), tolowercase(): Replace all strings in the StringList with 
//  their completely uppercase / lowercase equivalents.
StringList StringList::touppercase()
{
  for (unsigned int i = 0; i < stringlist.size(); i++) {
    unsigned int len = std::strlen(stringlist[i].c_str());
    char * temp = new char[len + 1];
    strcpy(temp, stringlist[i].c_str());
    for (unsigned int j = 0; j < len; j++)
      temp[j] = std::toupper(temp[j]);
    stringlist[i] = string(temp);
    delete [] temp;
  }
  return *this;
}


StringList StringList::tolowercase()
{
  for (unsigned int i = 0; i < stringlist.size(); i++) {
    unsigned int len = std::strlen(stringlist[i].c_str());
    char * temp = new char[len + 1];
    strcpy(temp, stringlist[i].c_str());
    for (unsigned int j = 0; j < len; j++)
      temp[j] = std::tolower(temp[j]);
    stringlist[i] = string(temp);
    delete [] temp;
  }
  return *this;
}
