#!/usr/bin/env python

# Qt tutorial 14.

import sys
import math
import whrandom
from qt import *


class LCDRange(QWidget):
    def __init__(self,s=None,parent=None,name=None):
        QWidget.__init__(self,parent,name)

        lcd = QLCDNumber(2,self,'lcd')
        self.slider = QSlider(Qt.Horizontal,self,'slider')
        self.slider.setRange(0,99)
        self.slider.setValue(0)

        self.label = QLabel(' ',self,'label')
        self.label.setAlignment(Qt.AlignCenter)

        self.connect(self.slider,SIGNAL('valueChanged(int)'),lcd,SLOT('display(int)'))
        self.connect(self.slider,SIGNAL('valueChanged(int)'),self,PYSIGNAL('valueChanged(int)'))

        self.setFocusProxy(self.slider)

        l = QVBoxLayout(self)
        l.addWidget(lcd,1)
        l.addWidget(self.slider)
        l.addWidget(self.label)

        if s is not None:
            self.setText(s)

    def value(self):
        return self.slider.value()

    def setValue(self,value):
        self.slider.setValue(value)

    def setRange(self,minVal,maxVal):
        if minVal < 0 or maxVal > 99 or minVal > maxVal:
            raise ValueError, 'LCDRange.setRange(): invalid range'
        self.slider.setRange(minVal,maxVal)

    def text(self):
        return self.label.text()

    def setText(self,s):
        self.label.setText(s)


class CannonField(QWidget):
    def __init__(self,parent=None,name=None):
        QWidget.__init__(self,parent,name)

        self.ang = 45
        self.f = 0
        self.timerCount = 0

        self.autoShootTimer = QTimer(self,'movement handler')
        self.connect(self.autoShootTimer,SIGNAL('timeout()'),self.moveShot)

        self.shoot_ang = 0
        self.shoot_f = 0
        self.target = QPoint(0,0)
        self.gameEnded = 0
        self.barrelPressed = 0

        self.setPalette(QPalette(QColor(250,250,200)))

        self.barrelRect = QRect(33,-4,15,8)

        self.newTarget()

    def angle(self):
        return self.ang

    def setAngle(self,degrees):
        if degrees < 5:
            degrees = 5
        if degrees > 70:
            degrees = 70
        if self.ang == degrees:
            return
        self.ang = degrees
        self.repaint(self.cannonRect(),0)
        self.emit(PYSIGNAL('angleChanged(int)'),(self.ang,))

    def force(self):
        return self.f

    def setForce(self,newton):
        if newton < 0:
            newton = 0
        if self.f == newton:
            return
        self.f = newton
        self.emit(PYSIGNAL('forceChanged(int)'),(self.f,))

    def shoot(self):
        if self.isShooting():
            return

        self.timerCount = 0
        self.shoot_ang = self.ang
        self.shoot_f = self.f
        self.autoShootTimer.start(50)
        self.emit(PYSIGNAL('canShoot(bool)'),(0,))

    def newTarget(self):
        r = QRegion(self.targetRect())
        self.target = QPoint(whrandom.randint(200,390),whrandom.randint(10,265))
        self.repaint(r.unite(QRegion(self.targetRect())))

    def gameOver(self):
        return self.gameEnded

    def setGameOver(self):
        if self.gameEnded:
            return
        if self.isShooting():
            self.autoShootTime.stop()
        self.gameEnded = 1
        self.repaint()

    def restartGame(self):
        if self.isShooting():
            self.autoShootTime.stop()
        self.gameEnded = 0
        self.repaint()
        self.emit(PYSIGNAL('canShoot(bool)'),(1,))

    def moveShot(self):
        r = QRegion(self.shotRect())
        self.timerCount = self.timerCount + 1

        shotR = self.shotRect()

        if shotR.intersects(self.targetRect()):
            self.autoShootTimer.stop()
            self.emit(PYSIGNAL('hit()'),())
            self.emit(PYSIGNAL('canShoot(bool)'),(1,))
        elif shotR.x() > self.width() or shotR.y() > self.height() or shotR.intersects(self.barrierRect()):
            self.autoShootTimer.stop()
            self.emit(PYSIGNAL('missed()'),())
            self.emit(PYSIGNAL('canShoot(bool)'),(1,))
        else:
            r = r.unite(QRegion(shotR))

        self.repaint(r)

    def mousePressEvent(self,ev):
        if ev.button() != Qt.LeftButton:
            return
        if self.barrelHit(ev.pos()):
            self.barrelPressed = 1

    def mouseMoveEvent(self,ev):
        if not self.barrelPressed:
            return
        pnt = ev.pos()
        if pnt.x() <= 0:
            pnt.setX(1)
        if pnt.y() >= self.height():
            pnt.setY(self.height() - 1)
        rad = math.atan(float(self.rect().bottom() - pnt.y()) / pnt.x())
        self.setAngle(int(round(rad * 180 / math.pi)))

    def mouseReleaseEvent(self,ev):
        if ev.button() == Qt.LeftButton:
            self.barrelPressed = 0

    def paintEvent(self,ev):
        updateR = ev.rect()
        p = QPainter(self)

        if self.gameEnded:
            p.setPen(Qt.black)
            p.setFont(QFont('Courier',48,QFont.Bold))
            p.drawText(self.rect(),Qt.AlignCenter,'Game Over')

        if updateR.intersects(self.cannonRect()):
            self.paintCannon(p)

        if updateR.intersects(self.barrierRect()):
            self.paintBarrier(p)

        if self.isShooting() and updateR.intersects(self.shotRect()):
            self.paintShot(p)

        if not self.gameEnded and updateR.intersects(self.targetRect()):
            self.paintTarget(p)

    def paintShot(self,p):
        p.setBrush(Qt.black)
        p.setPen(Qt.NoPen)
        p.drawRect(self.shotRect())

    def paintTarget(self,p):
        p.setBrush(Qt.red)
        p.setPen(Qt.black)
        p.drawRect(self.targetRect())

    def paintBarrier(self,p):
        p.setBrush(Qt.yellow)
        p.setPen(Qt.black)
        p.drawRect(self.barrierRect())

    def paintCannon(self,p):
        cr = self.cannonRect()
        pix = QPixmap(cr.size())
        pix.fill(self,cr.topLeft())

        tmp = QPainter(pix)
        tmp.setBrush(Qt.blue)
        tmp.setPen(Qt.NoPen)

        tmp.translate(0,pix.height() - 1)
        tmp.drawPie(QRect(-35,-35,70,70),0,90 * 16)
        tmp.rotate(-self.ang)
        tmp.drawRect(self.barrelRect)
        tmp.end()

        p.drawPixmap(cr.topLeft(),pix)

    def cannonRect(self):
        r = QRect(0,0,50,50)
        r.moveBottomLeft(self.rect().bottomLeft())
        return r

    def shotRect(self):
        gravity = 4.0

        time = self.timerCount / 4.0
        velocity = self.shoot_f
        radians = self.shoot_ang * math.pi / 180

        velx = velocity * math.cos(radians)
        vely = velocity * math.sin(radians)
        x0 = (self.barrelRect.right() + 5) * math.cos(radians)
        y0 = (self.barrelRect.right() + 5) * math.sin(radians)
        x = x0 + velx * time
        y = y0 + vely * time - 0.5 * gravity * time * time

        r = QRect(0,0,6,6)
        r.moveCenter(QPoint(x,self.height() - 1 - y))
        return r

    def targetRect(self):
        r = QRect(0,0,20,10)
        r.moveCenter(QPoint(self.target.x(),self.height() - 1 - self.target.y()))
        return r

    def barrierRect(self):
        return QRect(145,self.height() - 100,15,100)

    def barrelHit(self,p):
        mtx = QWMatrix()
        mtx.translate(0,self.height() - 1)
        mtx.rotate(-self.ang)
        (mtx, invertable) = mtx.invert()
        return self.barrelRect.contains(mtx.map(p))

    def isShooting(self):
        return self.autoShootTimer.isActive()

    def sizePolicy(self):
        return QSizePolicy(QSizePolicy.Expanding,QSizePolicy.Expanding)


class GameBoard(QWidget):
    def __init__(self,parent=None,name=None):
        QWidget.__init__(self,parent,name)

        quit = QPushButton('&Quit',self,'quit')
        quit.setFont(QFont('Times',18,QFont.Bold))
        self.connect(quit,SIGNAL('clicked()'),qApp,SLOT('quit()'))

        self.angle = LCDRange('ANGLE',self,'angle')
        self.angle.setRange(5,70)

        self.force = LCDRange('FORCE',self,'force')
        self.force.setRange(10,50)

        box = QVBox(self,'cannonFrame')
        box.setFrameStyle(QFrame.WinPanel | QFrame.Sunken)

        self.cannonField = CannonField(box,'cannonField')

        self.connect(self.angle,PYSIGNAL('valueChanged(int)'),self.cannonField.setAngle)
        self.connect(self.cannonField,PYSIGNAL('angleChanged(int)'),self.angle.setValue)

        self.connect(self.force,PYSIGNAL('valueChanged(int)'),self.cannonField.setForce)
        self.connect(self.cannonField,PYSIGNAL('forceChanged(int)'),self.force.setValue)

        self.connect(self.cannonField,PYSIGNAL('hit()'),self.hit)
        self.connect(self.cannonField,PYSIGNAL('missed()'),self.missed)

        self.shoot = QPushButton('&Shoot',self,'shoot')
        self.shoot.setFont(QFont('Times',18,QFont.Bold))
        self.connect(self.shoot,SIGNAL('clicked()'),self.fire)
        self.connect(self.cannonField,PYSIGNAL('canShoot(bool)'),self.shoot,SLOT('setEnabled(bool)'))

        restart = QPushButton('&New Game',self,'newgame')
        restart.setFont(QFont('Times',18,QFont.Bold))
        self.connect(restart,SIGNAL('clicked()'),self.newGame)

        self.hits = QLCDNumber(2,self,'hits')
        self.shotsLeft = QLCDNumber(2,self,'shotsleft')
        hitsL = QLabel('HITS',self,'hitsLabel')
        shotsLeftL = QLabel('SHOTS LEFT',self,'shotsleftLabel')

        accel = QAccel(self)
        accel.connectItem(accel.insertItem(Qt.Key_Enter),self.fire)
        accel.connectItem(accel.insertItem(Qt.Key_Return),self.fire)
        accel.connectItem(accel.insertItem(Qt.CTRL + Qt.Key_Q),qApp,SLOT('quit()'))

        grid = QGridLayout(self,2,2,10)
        grid.addWidget(quit,0,0)
        grid.addWidget(box,1,1)
        grid.setColStretch(1,10)

        leftBox = QVBoxLayout()
        grid.addLayout(leftBox,1,0)
        leftBox.addWidget(self.angle)
        leftBox.addWidget(self.force)

        topBox = QHBoxLayout()
        grid.addLayout(topBox,0,1)
        topBox.addWidget(self.shoot)
        topBox.addWidget(self.hits)
        topBox.addWidget(hitsL)
        topBox.addWidget(self.shotsLeft)
        topBox.addWidget(shotsLeftL)
        topBox.addStretch(1)
        topBox.addWidget(restart)

        self.angle.setValue(60)
        self.force.setValue(25)
        self.angle.setFocus()

        self.newGame()

    def fire(self):
        if self.cannonField.gameOver() or self.cannonField.isShooting():
            return
        self.shotsLeft.display(self.shotsLeft.intValue() - 1)
        self.cannonField.shoot()

    def hit(self):
        self.hits.display(self.hits.intValue() + 1)
        if self.shotsLeft.intValue() == 0:
            self.cannonField.setGameOver()
        else:
            self.cannonField.newTarget()

    def missed(self):
        if self.shotsLeft.intValue() == 0:
            self.cannonField.setGameOver()

    def newGame(self):
        self.shotsLeft.display(15)
        self.hits.display(0)
        self.cannonField.restartGame()
        self.cannonField.newTarget()


QApplication.setColorSpec(QApplication.CustomColor)
a = QApplication(sys.argv)

gb = GameBoard()
gb.setGeometry(100,100,500,355)
a.setMainWidget(gb)
gb.show()
a.exec_loop()
