#include <config.h>
#include "Python.h"
#include <all-headers.h>

#ifndef _PATH_DEV
#define _PATH_DEV "/dev/"
#endif

#define _PATH_DEV_PTYXX _PATH_DEV "ptyXX"


/*
   This opens a process as a pipe. 'in', 'out' and 'err' are
   pointers to file handles which are filled in by popen. 'in'
   refers to stdin of the process, to which you can write. 'out'
   and 'err' refer to stdout and stderr of the process from
   which you can read. 'in', 'out' and 'err' can be passed as
   0 if you want to ignore output or input of those pipes. If
   'mix' is 1, then both stderr and stdout of the process
   can be read from 'out'. If mix is 1, then 'err' must
   be passed as null. Popen forks and then calls execvp (see
   execvp(3)) --- which must also take argv[0] and args must
   terminate with a 0. Returns -1 if the fork failed, and -2
   if pipe() failed. Otherwise returns the pid of the child.
   If mix is 2 then stderr is not redirected in any way and
   may for example continue outputting to the terminal.
 */
static pid_t triple_pipe_open (int *in, int *out, int *err, int mix, const char *file, char *const argv[])
{
    pid_t p;
    int e;
    int f0[2], f1[2], f2[2];
    if (mix == 2)
	e = (pipe (f0) | pipe (f1));
    else
	e = (pipe (f0) | pipe (f1) | pipe (f2));
    if (e) {
	close (f0[0]);
	close (f0[1]);
	close (f1[0]);
	close (f1[1]);
	if (mix != 2) {
	    close (f2[0]);
	    close (f2[1]);
	}
	return -2;
    }
    p = fork ();
    if (p == -1) {
	close (f0[0]);
	close (f0[1]);
	close (f1[0]);
	close (f1[1]);
	if (mix != 2) {
	    close (f2[0]);
	    close (f2[1]);
	}
	return -1;
    }
    if (p) {
	if (in) {
	    *in = f0[1];
	} else {
	    close (f0[1]);
	}
	if (out) {
	    *out = f1[0];
	} else {
	    close (f1[0]);
	}
	if (mix != 2) {
	    if (err) {
		*err = f2[0];
	    } else {
		close (f2[0]);
	    }
	}
	close (f0[0]);
	close (f1[1]);
	if (mix != 2) {
	    close (f2[1]);
	}
	return p;
    } else {
	int nulldevice_wr, nulldevice_rd;
	nulldevice_wr = open ("/dev/null", O_WRONLY);
	nulldevice_rd = open ("/dev/null", O_RDONLY);
	close (0);
	if (in)
	    dup (f0[0]);
	else
	    dup (nulldevice_rd);
	close (1);
	if (out)
	    dup (f1[1]);
	else
	    dup (nulldevice_wr);
	if (mix != 2) {
	    close (2);
	    if (err)
		dup (f2[1]);
	    else {
		if (mix)
		    dup (f1[1]);
		else
		    dup (nulldevice_wr);
	    }
	}
	close (f0[0]);
	close (f0[1]);
	close (f1[0]);
	close (f1[1]);
	if (mix != 2) {
	    close (f2[0]);
	    close (f2[1]);
	}
	close (nulldevice_rd);
	close (nulldevice_wr);
	execvp (file, argv);
	exit (1);
    }
    return 0;			/* prevents warning */
}

#ifndef HAVE_FORKPTY

/* This is stolen from libc-5.4.46 with the intention of having
   this all still work on systems that do not have the forkpty()
   function (like non-BSD systems like solaris). No idea if it will
   actually work on these systems. */

static int openpty (int *amaster, int *aslave, char *name)
{
    char line[] = _PATH_DEV_PTYXX;
    const char *p, *q;
    int master, slave, ttygid;
    struct group *gr;
    if ((gr = getgrnam ("tty")) != NULL)
	ttygid = gr->gr_gid;
    else
	ttygid = -1;
    for (p = "pqrstuvwxyzabcde"; *p; p++) {
	line[sizeof (_PATH_DEV_PTYXX) - 3] = *p;
	for (q = "0123456789abcdef"; *q; q++) {
	    line[sizeof (_PATH_DEV_PTYXX) - 2] = *q;
	    if ((master = open (line, O_RDWR, 0)) == -1) {
		if (errno == ENOENT)
		    return -1;
	    } else {
		line[sizeof (_PATH_DEV) - 1] = 't';
		chown (line, getuid (), ttygid);
		chmod (line, S_IRUSR | S_IWUSR | S_IWGRP);
		if ((slave = open (line, O_RDWR, 0)) != -1) {
		    *amaster = master;
		    *aslave = slave;
		    strcpy (name, line);
		    return 0;
		}
		close (master);
		line[sizeof (_PATH_DEV) - 1] = 'p';
	    }
	}
    }
    errno = ENOENT;
    return -1;
}

static int forkpty (int *amaster, char *name,...)
{
    int master, slave;
    pid_t pid;
    if (openpty (&master, &slave, name) == -1)
	return -1;
    switch (pid = fork ()) {
    case -1:
	return -1;
    case 0:
	close (master);
#ifdef HAVE_SETSID
	pid = setsid ();
#define HAVE_PID_THIS_TTY
#elif defined (HAVE_SETPGRP)
	pid = setpgrp (0, 0);
#define HAVE_PID_THIS_TTY
#endif
#ifdef TIOCSCTTY
	ioctl (slave, TIOCSCTTY, 0);
#ifdef HAVE_PID_THIS_TTY
#elif defined (HAVE_TCSETPGRP)
	tcsetpgrp (slave, pid);
#elif defined (TIOCSPGRP)
	ioctl (slave, TIOCSPGRP, &pid);
#endif
#endif
	close (0);
	dup (slave);
	close (1);
	dup (slave);
	close (2);
	dup (slave);
	if (slave > 2)
	    close (slave);
	return 0;
    }
    *amaster = master;
    close (slave);
    return pid;
}

#endif				/* ! HAVE_FORKPTY */

static void set_termios (int fd)
{
#ifdef HAVE_TCGETATTR
    struct termios tios;
    memset (&tios, 0, sizeof (tios));
    if (tcgetattr (fd, &tios) != 0)
	return;
#ifdef B19200
#ifdef OCRNL
    tios.c_oflag &= ~(ONLCR | OCRNL);
#else
    tios.c_oflag &= ~ONLCR;
#endif
    tios.c_lflag &= ~(ECHO | ICANON | ISIG);
    tios.c_iflag &= ~(ICRNL);
#ifdef VTIME
    tios.c_cc[VTIME] = 1;
#endif
#ifdef VMIN
    tios.c_cc[VMIN] = 1;
#endif
    tios.c_iflag &= ~(ISTRIP);
#if defined(TABDLY) && defined(TAB3)
    if ((tios.c_oflag & TABDLY) == TAB3)
	tios.c_oflag &= ~TAB3;
#endif
/* disable interpretation of ^S: */
    tios.c_iflag &= ~IXON;
#ifdef VDISCARD
    tios.c_cc[VDISCARD] = 255;
#endif
#ifdef VEOL2
    tios.c_cc[VEOL2] = 255;
#endif
#ifdef VEOL
    tios.c_cc[VEOL] = 255;
#endif
#ifdef VLNEXT
    tios.c_cc[VLNEXT] = 255;
#endif
#ifdef VREPRINT
    tios.c_cc[VREPRINT] = 255;
#endif
#ifdef VSUSP
    tios.c_cc[VSUSP] = 255;
#endif
#ifdef VWERASE
    tios.c_cc[VWERASE] = 255;
#endif
    tcsetattr (fd, TCSADRAIN, &tios);
#endif
#endif				/* HAVE_TCGETATTR */
}

static pid_t open_under_pty (int *in, int *out, char **line, const char *file, char *const argv[])
{
    int master = 0;
    char l[80];
    pid_t p;
#ifdef HAVE_FORKPTY
#ifdef NEED_WINSIZE
    struct winsize {
	unsigned short ws_row;
	unsigned short ws_col;
	unsigned short ws_xpixel;
	unsigned short ws_ypixel;
    } win;
#else
    struct winsize win;
#endif
    memset (&win, 0, sizeof (win));
    p = (pid_t) forkpty (&master, l, NULL, &win);
#else
    p = (pid_t) forkpty (&master, l);
#endif
    if (p == -1)
	return -1;
    if (p) {
#if 0
	int not = 0;
	ioctl (master, FIONBIO, &not);
#endif
	*in = master;
	*out = master;
	*line = (char *) strdup (l);
	return p;
    }
    set_termios (0);
    execvp (file, argv);
    exit (1);
    return 0;
}

struct read_list {
/* types - a bit silly, should just have two linked lists. it had its reasons once though */
#define READ_STDERR 1
#define READ_STDOUT 2
    char type;
    unsigned char error;
/* must be less than short */
#define CHUNK_SIZE 16384
    short size;
    unsigned char *data;
    unsigned char *alloced;
    struct read_list *next;
    struct read_list *prev;
};

struct expect_object {
    int in;			/* stdin fd */
    int out;			/* stdout fd */
    int err;			/* error fd or -1 */
    int block;			/* set to non-zero if blocking */
    struct timeval *timeout;	/* timeout */
    int timeout_error;		/* non-zero if a timeout was reached */
    long count;			/* total bytes read */
    pid_t pid;			/* process id of the child */
    char *command;		/* shell command */
    char *line;			/* pseudo tty line */
    struct read_list *end;	/* queued data already read from in or err - last in queue */
    struct read_list *start;	/* queued data already read from in or err - first in queue */
};

typedef struct {
    PyObject_HEAD
    struct expect_object *obj;
} PyExpect_Object;

staticforward PyTypeObject PyExpect_Type;

static PyObject *PyExpect_New (struct expect_object *c)
{
    PyExpect_Object *self;
    self = (PyExpect_Object *) PyObject_NEW (PyExpect_Object, &PyExpect_Type);
    if (self == 0)
	return 0;
    self->obj = c;
    return (PyObject *) self;
}

static int child_close (struct expect_object *c, int *status)
{
    if (c->in >= 0)
	close (c->in);
    if (c->out >= 0)
	close (c->out);
    if (c->err >= 0)
	close (c->err);
    c->in = c->out = -1;
    c->err = -1;
    if (c->line) {
	char *p;
	p = c->line + sizeof (_PATH_DEV) - 1;
	chmod (c->line, S_IWUSR | S_IRUSR);
	chown (c->line, 0, 0);
	*p = 'p';
	chmod (c->line, 0666);
	chown (c->line, 0, 0);
	free (c->line);
	c->line = 0;
    }
    if (c->pid > 0) {
	int r;
	r = waitpid (c->pid, status, 0);
	c->pid = -1;
	return r;
    }
    return 0;
}

static void PyExpect_dealloc (PyExpect_Object * self)
{
    struct expect_object *c;
    struct read_list *p, *q;
    c = self->obj;
    if (c->pid > 0)
	kill (c->pid, SIGTERM);
    child_close (c, 0);
    if (c->command)
	free (c->command);
    for (p = c->start; p; p = q) {
	q = p->next;
	if (p->alloced) {
	    memset (p->data, 0, p->size);
	    free (p->alloced);
	}
	free (p);
    }
    PyMem_DEL (self);
}

static int PyExpect_compare (PyExpect_Object * self, PyExpect_Object * v)
{
    if (self->obj == v->obj)
	return 0;
    if (self->obj > v->obj)
	return -1;
    return 1;
}

static PyObject *PyExpect_repr (PyExpect_Object * self)
{
    struct expect_object *c;
    char buf[256], pid[32];
    c = self->obj;
    if (c->pid > 0)
	sprintf (pid, "as pid %ld", (long) c->pid);
    else
	sprintf (pid, "(killed)");
    sprintf (buf, "<ExpectObject running command \"%.30s%s\" %s at %lx>",
	     c->command,
	     strlen (c->command) > 30 ? "..." : "",
	     pid,
	     (long) c);
    return PyString_FromString (buf);
}

static char PyExpect_Type__doc__[] = "This is the type of Expect Objects";

#define PyExpect_Check(v) ((v)->ob_type == &PyExpect_Type)

static PyTypeObject PyExpect_Type =
{
    PyObject_HEAD_INIT (&PyType_Type)
    0,				/*ob_size */
    "ExpectObject",		/*tp_name */
    sizeof (PyExpect_Object),	/*tp_basicsize */
    0,				/*tp_itemsize */
    (destructor) PyExpect_dealloc,	/*tp_dealloc */
    (printfunc) 0,		/*tp_print */
    (getattrfunc) 0,		/*tp_getattr */
    (setattrfunc) 0,		/*tp_setattr */
    (cmpfunc) PyExpect_compare,	/*tp_compare */
    (reprfunc) PyExpect_repr,	/*tp_repr */
    0,				/*tp_as_number */
    0,				/*tp_as_sequence */
    0,				/*tp_as_mapping */
    (hashfunc) 0,		/*tp_hash */
    (ternaryfunc) 0,		/*tp_call */
    (reprfunc) 0,		/*tp_str */
    0L, 0L, 0L, 0L,
    PyExpect_Type__doc__
};

/* FIXME: optimise for one byte reads */
static void read_list_new_item (struct expect_object *c, int fd, int type)
{
    unsigned char data[CHUNK_SIZE];
    int size;
    struct read_list *p;
    p = (struct read_list *) malloc (sizeof (struct read_list));
    memset (p, 0, sizeof (struct read_list));
    p->prev = c->end;
    p->type = type;
    size = read (fd, data, CHUNK_SIZE);
    if (size <= 0) {
/* record the error */
	p->data = p->alloced = 0;
	p->error = errno;
    } else {
	p->data = p->alloced = (unsigned char *) malloc (size);
	memcpy (p->data, data, size);
    }
    p->size = size;
    if (!c->end) {
	c->end = c->start = p;
    } else {
	c->end->next = p;
	c->end = p;
    }
    return;
}

#define MAX(x,y) ((x) > (y) ? (x) : (y))
#define MIN(x,y) ((x) < (y) ? (x) : (y))

static int child_write (struct expect_object *c, unsigned char *data, int size)
{
    struct timeval tv, timeout, *_timeout = 0;
    int r, n = 0, total = 0;
    fd_set reading, writing;
    errno = 0;
    for (;;) {
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	FD_ZERO (&reading);
	FD_ZERO (&writing);
	if (c->out >= 0) {
	    FD_SET (c->out, &reading);
	    n = MAX (n, c->out);
	}
	if (c->err >= 0) {
	    FD_SET (c->err, &reading);
	    n = MAX (n, c->err);
	}
/* if size is zero we STILL do one last check on the read fd's */
	if (size && c->in) {
	    FD_SET (c->in, &writing);
	    n = MAX (n, c->in);
	}
/* don't block if we have filled the buffer already */
	r = 0;
	if (n) {
	    if (c->block && size) {
		if (c->timeout) {
/* block timeout */
		    timeout = *c->timeout;
		    _timeout = &timeout;
		} else {
/* block forever */
		    _timeout = 0;
		}
	    } else {
/* no block */
		_timeout = &tv;
	    }
	    r = select (n + 1, &reading, &writing, 0, _timeout);
	}
	if (r < 0)
	    return r;
	if (!r) {
	    if (_timeout == &timeout)
		c->timeout_error = 1;
	    break;
	}
	if (c->err >= 0)
	    if (FD_ISSET (c->err, &reading))
		read_list_new_item (c, c->err, READ_STDERR);
	if (c->out >= 0)
	    if (FD_ISSET (c->out, &reading))
		read_list_new_item (c, c->out, READ_STDOUT);
	if (!size)
	    break;
	if (c->in >= 0)
	    if (FD_ISSET (c->in, &writing)) {
		r = write (c->in, data, size);
		if (r <= 0)
		    return -1;
		size -= r;
		data += r;
		total += r;
	    }
	if (!c->block)
	    break;
    }
    return total;		/* prevents warning */
}

static int search (unsigned char *back, int back_size, struct read_list *p, unsigned char *find, int len)
{
    unsigned char *s, *l, x;
    x = find[len - 1];
/* search backwards to see if we have just read, say, half of the string */
    s = p->data;
    if (back_size < len - 1)
	s += len - back_size - 1;
    l = p->data + len - 1;
    if (len >= p->size)
	l = p->data + p->size;
    for (; (unsigned long) s < (unsigned long) l; s++)
	if (*s == x) {
	    int k;
/* k is the length of the latter ``unread'' half of the string */
	    k = (unsigned long) s - (unsigned long) p->data + 1;
	    if (!memcmp (back + back_size + k - len, find, len - k) && \
		!memcmp (p->data, find + len - k, k))
		return k;
	}
    x = find[0];
/* haven't even read enough to make a comparison */
    if (p->size < len)
	return -1;
/* search forward to find the first instance of the string */
    l = p->data + p->size - len;
    for (s = p->data; (unsigned long) s <= (unsigned long) l; s++)
	if (*s == x)
	    if (!memcmp (s, find, len))
		return (unsigned long) s - (unsigned long) p->data + len;
    return -1;
}

#define UNLINK(c) {			\
	if (p->next)			\
	    p->next->prev = p->prev;	\
	if (p->prev)			\
	    p->prev->next = p->next;	\
	if (p == c->start)		\
	    c->start = p->next;		\
	if (p == c->end)		\
	    c->end = p->prev;		\
	}

static long child_read (struct expect_object *c, PyObject ** v, long size, PyObject * find, int *found, int type)
{
    PyObject *u;
    struct read_list *p, *q;
    struct timeval tv, timeout, *_timeout = 0;
    int r, n = 0;
    long total = 0;
    fd_set reading;
    errno = 0;
    for (;;) {
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	FD_ZERO (&reading);
	if (c->out >= 0) {
	    FD_SET (c->out, &reading);
	    n = MAX (n, c->out);
	}
	if (c->err >= 0) {
	    FD_SET (c->err, &reading);
	    n = MAX (n, c->err);
	}
/* don't block if we have filled the buffer already 
   or if there is already other data in the read queue */
	r = 0;
	if (n) {
	    if (c->block && size && !c->start) {
		if (c->timeout) {
/* block timeout */
		    timeout = *c->timeout;
		    _timeout = &timeout;
		} else {
/* block forever */
		    _timeout = 0;
		}
	    } else {
/* no block */
		_timeout = &tv;
	    }
	    r = select (n + 1, &reading, 0, 0, _timeout);
	}
	if (r < 0)
	    return r;
	if (!r && !c->start) {
	    if (_timeout == &timeout)
		c->timeout_error = 1;
	    break;
	}
	if (c->err >= 0)
	    if (FD_ISSET (c->err, &reading))
		read_list_new_item (c, c->err, READ_STDERR);
	if (c->out >= 0)
	    if (FD_ISSET (c->out, &reading))
		read_list_new_item (c, c->out, READ_STDOUT);
	if (!size)
	    break;
	for (p = c->start; p; p = q) {
	    q = p->next;
/* we have separate err and out streams so only read the requested type */
	    if (!(p->type & type) && c->err >= 0)
		continue;
	    if (!p->data) {
/* at least return what we have so far */
		if (total)
		    return total;
/* read nothing yet, so return the error */
		errno = p->error;
		UNLINK (c);
		size = p->size;
		free (p);
		return size;
	    }
	    if (find) {
		int l, j;
		for (j = 0; j < PyTuple_Size (find); j++) {
		    l = search ((unsigned char *) PyString_AsString (*v), PyString_Size (*v), p, \
				(unsigned char *) PyString_AsString (PyTuple_GetItem (find, j)), PyString_Size (PyTuple_GetItem (find, j)));
		    if (l >= 0) {
			if (found)
			    *found = 1;
			if (l < size)
			    size = l;
		    }
		}
	    }
	    if (p->size > size) {
/* FIXME: quadratic algorithm? */
		PyString_Concat (v, u = PyString_FromStringAndSize ((char *) p->data, size));
		c->count += size;
		Py_DECREF (u);
/* protect passwords: */
		memset (p->data, 0, size);
		p->data += size;
		p->size -= size;
		total += size;
		size = 0;
	    } else {
		PyString_Concat (v, u = PyString_FromStringAndSize ((char *) p->data, p->size));
		c->count += p->size;
		Py_DECREF (u);
/* protect passwords: */
		memset (p->data, 0, p->size);
		free (p->alloced);
		UNLINK (c);
		size -= p->size;
		total += p->size;
		free (p);
	    }
	}
	if (!c->block)
	    break;
	if (!r) {
	    if (_timeout == &timeout) {
		c->timeout_error = 1;
		break;
	    }
	}
    }
    return total;
}

#define IS_WHITE(c) ((c) == '\t' || (c) == ' ')

static char **split_command_line (char *command)
{
    char **a, *p;
    int n = 0;
    for (p = command; *p; p++)
	if (*p == ' ')
	    n++;
    a = (char **) malloc ((n + 2) * sizeof (char *));
    memset (a, 0, (n + 2) * sizeof (char *));
    p = command;
    while (IS_WHITE (*p))
	p++;
    a[0] = p;
    for (n = 1; *p;) {
	switch (*p) {
	case ' ':
	    *p++ = '\0';
	    while (IS_WHITE (*p))
		p++;
	    if (!*p)
		return a;
	    a[n++] = p;
	    break;
	case '\'':
	    strcpy (p, p + 1);	/* delete ' character */
	    while (*p && *p != '\'')
		p++;
	    strcpy (p, p + 1);	/* delete ' character */
	    break;
	case '\"':
	    strcpy (p, p + 1);	/* delete ' character */
	    while (*p && *p != '\"') {
		switch (*p) {
		case '\\':
		    if (strchr ("$`\"\\", p[1])) {
			strcpy (p, p + 1);	/* delete \ character */
			p++;
		    } else if (p[1] == '\n') {
			strcpy (p, p + 2);	/* delete \<newline> character */
		    } else {
			p++;
		    }
		    break;
		case '$':
		case '`':
/* obviously wanting some functionality we can't give */
		    free (a);
		    return 0;
		default:
		    p++;
		}
	    }
	    strcpy (p, p + 1);	/* delete ' character */
	    break;
	case '\\':
	    if (p[1] == '\n')
		strcpy (p, p + 2);	/* delete \<newline> character */
	    else {
		strcpy (p, p + 1);	/* delete \ character */
		p++;
	    }
	    break;
	case '>':
	case '`':
	case '!':
	case '<':
	case '&':
	case '$':
	case '\n':
	case ';':
	case '(':
	case ')':
	case '{':
	case '}':
/* obviously wanting some functionality we can't give */
	    free (a);
	    return 0;
	case '=':
	    p++;
	    if (n == 1) {
/* trying a ENVVAR=something */
		free (a);
		return 0;
	    }
	    break;
	default:
	    p++;
	}
    }
    return a;
}

#define PIPE3_OPTION_MIX			1
#define PIPE3_OPTION_BLOCK			2
#define PIPE3_OPTION_STDERR_UNTOUCHED		4
#define PIPE3_OPTION_FORK_PTY			8

static struct expect_object *child_open (char *command, long options)
{
    struct expect_object *c;
    char **a, *s;
    a = split_command_line (s = (char *) strdup (command));
    if (!a) {
	a = (char **) malloc (sizeof (char *) * 4);
	a[0] = "/bin/sh";
	a[1] = "-c";
	a[2] = command;
	a[3] = 0;
    }
    c = (struct expect_object *) malloc (sizeof (struct expect_object));
    memset (c, 0, sizeof (struct expect_object));
    c->in = c->out = -1;
    c->err = -1;
    c->command = (char *) strdup (command);
    if (options & PIPE3_OPTION_FORK_PTY) {
	c->err = -1;
	c->pid = open_under_pty (&c->in, &c->out, &c->line, a[0], a);
    } else if (options & PIPE3_OPTION_STDERR_UNTOUCHED) {
	c->err = -1;
	c->pid = triple_pipe_open (&c->in, &c->out, 0, 2, a[0], a);
    } else if (options & PIPE3_OPTION_MIX) {
	c->err = -1;
	c->pid = triple_pipe_open (&c->in, &c->out, 0, 1, a[0], a);
    } else {
	c->pid = triple_pipe_open (&c->in, &c->out, &c->err, 0, a[0], a);
    }
    if (c->pid <= 0)
	goto error;
    if (options & PIPE3_OPTION_BLOCK)
	c->block = 1;
    free (a);
    free (s);
    return c;
  error:
    free (a);
    free (s);
    free (c);
    return 0;
}

static PyObject *wrap_popen (PyObject * self, PyObject * args)
{
    char *options = 0, *command = 0;
    struct expect_object *c;
    long o = PIPE3_OPTION_STDERR_UNTOUCHED | PIPE3_OPTION_BLOCK;
    if (!PyArg_ParseTuple (args, "s|s:popen", &command, &options))
	return 0;
    if (options) {
	if (strchr (options, 'b'))
	    o &= ~PIPE3_OPTION_BLOCK;
	if (strchr (options, 'p'))
	    o |= PIPE3_OPTION_FORK_PTY;
    }
    c = child_open (command, o);
    if (!c) {
	PyErr_SetFromErrno (PyExc_IOError);
	return 0;
    }
    return PyExpect_New (c);
}

static PyObject *wrap_popen2 (PyObject * self, PyObject * args)
{
    char *options = 0, *command = 0;
    struct expect_object *c;
    long o = PIPE3_OPTION_MIX | PIPE3_OPTION_BLOCK;
    if (!PyArg_ParseTuple (args, "s|s:popen2", &command, &options))
	return 0;
    if (options) {
	if (strchr (options, 'b'))
	    o &= ~PIPE3_OPTION_BLOCK;
	if (strchr (options, 'p'))
	    o |= PIPE3_OPTION_FORK_PTY;
    }
    c = child_open (command, o);
    if (!c) {
	PyErr_SetFromErrno (PyExc_IOError);
	return 0;
    }
    return PyExpect_New (c);
}

static PyObject *wrap_popen3 (PyObject * self, PyObject * args)
{
    char *options = 0, *command = 0;
    struct expect_object *c;
    long o = PIPE3_OPTION_BLOCK;
    if (!PyArg_ParseTuple (args, "s|s:popen3", &command, &options))
	return 0;
    if (options) {
	if (strchr (options, 'b'))
	    o &= ~PIPE3_OPTION_BLOCK;
	if (strchr (options, 'p'))
	    o |= PIPE3_OPTION_FORK_PTY;
    }
    c = child_open (command, o);
    if (!c) {
	PyErr_SetFromErrno (PyExc_IOError);
	return 0;
    }
    return PyExpect_New (c);
}

static PyObject *wrap_set_blocking (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    int block;
    if (!PyArg_ParseTuple (args, "O!i:set_blocking", &PyExpect_Type, &e, &block))
	return 0;
    c = (struct expect_object *) e->obj;
    c->block = block;
    Py_INCREF (Py_None);
    return Py_None;
}

static PyObject *wrap_set_timeout (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    double timeout = 0.0;
    if (!PyArg_ParseTuple (args, "O!|d:set_timeout", &PyExpect_Type, &e, &timeout))
	return 0;
    c = (struct expect_object *) e->obj;
    if (c->timeout) {
	free (c->timeout);
	c->timeout = 0;
    }
    if (timeout) {
	c->timeout = (struct timeval *) malloc (sizeof (struct timeval));
	c->timeout->tv_sec = (time_t) timeout;
	c->timeout->tv_usec = (time_t) ((double) (timeout - (double) c->timeout->tv_sec) * 1000000.0);
    }
    Py_INCREF (Py_None);
    return Py_None;
}

static PyObject *wrap_get_count (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:get_count", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    return PyInt_FromLong (c->count);
}

static PyObject *wrap_get_pid (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:get_pid", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    return PyInt_FromLong ((long) c->pid);
}

static PyObject *wrap_get_fd_in (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:get_pid", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    return PyInt_FromLong ((long) c->in);
}

static PyObject *wrap_get_fd_out (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:get_pid", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    return PyInt_FromLong ((long) c->out);
}

static PyObject *wrap_get_fd_err (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:get_pid", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    return PyInt_FromLong ((long) c->err);
}

static PyObject *wrap_get_command (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:get_command", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    return PyString_FromString (c->command);
}

static PyObject *wrap_read (PyObject * self, PyObject * args, int type)
{
    PyObject *v = 0, *s = 0;
    PyExpect_Object *e;
    struct expect_object *c;
    long l;
    int r;
    if (!PyExpect_Check (PyTuple_GetItem (args, 0))) {
	PyErr_SetString (PyExc_TypeError, "read, argument 1: expected ExpectObject");
	return 0;
    }
    e = (PyExpect_Object *) PyTuple_GetItem (args, 0);
    c = (struct expect_object *) e->obj;
    if (c->err < 0 && type & READ_STDERR) {
	PyErr_SetString (PyExc_ValueError, "No stderr stream. Use popen3() perhaps?");
	return 0;
    }
    v = PyString_FromString ("");
    if (PyTuple_Size (args) > 1)
	s = PyTuple_GetItem (args, 1);
    if (!s) {
	l = (unsigned long) -1 >> 1;
	r = child_read (c, &v, l, 0, 0, type);
    } else if (PyString_Check (s)) {
	PyObject *find;
	int found;
	found = !c->block;
	if (!PyString_Size (s)) {
	    Py_DECREF (v);
	    PyErr_SetString (PyExc_TypeError, "read, argument 2: string must be non-empty");
	    return 0;
	}
	l = (unsigned long) -1 >> 1;
	find = PyTuple_New (1);
	Py_INCREF (s);
	PyTuple_SetItem (find, 0, s);
	r = child_read (c, &v, l, find, &found, type);
	Py_DECREF (find);
	if (!found) {
	    Py_DECREF (v);
	    PyErr_SetString (PyExc_IOError, "read, string not found in output");
	    return 0;
	}
    } else if (PyInt_Check (s)) {
	l = PyInt_AsLong (s);
	r = child_read (c, &v, l, 0, 0, type);
    } else if (PyTuple_Check (s)) {
	int j;
	int found;
	found = !c->block;
	for (j = 0; j < PyTuple_Size (s); j++) {
	    if (!PyString_Check (PyTuple_GetItem (s, j))) {
		Py_DECREF (v);
		PyErr_SetString (PyExc_TypeError, "read, argument 2: tuple element must be a string");
		return 0;
	    }
	    if (!PyString_Size (PyTuple_GetItem (s, j))) {
		Py_DECREF (v);
		PyErr_SetString (PyExc_TypeError, "read, argument 2: tuple element must be a non-empty string");
		return 0;
	    }
	}
	l = (unsigned long) -1 >> 1;
	r = child_read (c, &v, l, s, &found, type);
	if (!found) {
	    Py_DECREF (v);
	    PyErr_SetString (PyExc_IOError, "read, strings not found in output");
	    return 0;
	}
    } else {
	Py_DECREF (v);
	PyErr_SetString (PyExc_TypeError, "read, argument 2: expected string, int or tuple");
	return 0;
    }
    if (r < 0) {
	Py_DECREF (v);
	PyErr_SetFromErrno (PyExc_IOError);
	return 0;
    }
    if (c->timeout_error) {
	c->timeout_error = 0;
	Py_DECREF (v);
	PyErr_SetString (PyExc_IOError, "read, timeout reached");
	return 0;
    }
    return v;
}

static PyObject *wrap_read_stderr (PyObject * self, PyObject * args)
{
    return wrap_read (self, args, READ_STDERR);
}

static PyObject *wrap_read_stdout (PyObject * self, PyObject * args)
{
    return wrap_read (self, args, READ_STDOUT);
}

static PyObject *wrap_write (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    unsigned char *s;
    int len, r;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!s#:write", &PyExpect_Type, &e, &s, &len))
	return 0;
    c = (struct expect_object *) e->obj;
    r = child_write (c, s, len);
    if (r < 0) {
	PyErr_SetFromErrno (PyExc_IOError);
	return 0;
    }
    if (c->timeout_error) {
	c->timeout_error = 0;
	PyErr_SetString (PyExc_IOError, "write, timeout reached");
	return 0;
    }
    return PyInt_FromLong ((long) r);
}

static char *get_sig_name (int s)
{
    struct sig_name {
	char *name;
	int signal;
    } sig_name[] = { {
	    "SIGHUP", SIGHUP
    }, {
	"SIGINT", SIGINT
    }, {
	"SIGQUIT", SIGQUIT
    }, {
	"SIGILL", SIGILL
    }, {
	"SIGABRT", SIGABRT
    }, {
	"SIGFPE", SIGFPE
    }, {
	"SIGKILL", SIGKILL
    }, {
	"SIGSEGV", SIGSEGV
    }, {
	"SIGPIPE", SIGPIPE
    }, {
	"SIGALRM", SIGALRM
    }, {
	"SIGTERM", SIGTERM
    }, {
	0, 0
    }
    }, *p;
    for (p = &sig_name[0]; p->name; p++)
	if (p->signal == s)
	    return p->name;
    return "";
}

static PyObject *wrap_close (PyObject * self, PyObject * args)
{
    PyExpect_Object *e;
    pid_t p;
    int status = 0;
    struct expect_object *c;
    if (!PyArg_ParseTuple (args, "O!:close", &PyExpect_Type, &e))
	return 0;
    c = (struct expect_object *) e->obj;
    if (c->pid <= 0) {
	PyErr_SetString (PyExc_IOError, "child no longer exists");
	return 0;
    }
    p = child_close (c, &status);
    if (p < 0) {
	PyErr_SetFromErrno (PyExc_IOError);
	return 0;
    }
    if (WIFSIGNALED (status)) {
	char buf[80];
	sprintf (buf, "child killed by signal %d - %s", WTERMSIG (status), get_sig_name (WTERMSIG (status)));
	PyErr_SetString (PyExc_IOError, buf);
	return 0;
    }
    if (WIFEXITED (status))
	return PyInt_FromLong (WEXITSTATUS (status));
    PyErr_SetString (PyExc_IOError, "child did not exit normally");
    return 0;
}

static PyMethodDef expect_methods[] =
{
    {"write", wrap_write, 1},
    {"read", wrap_read_stdout, 1},
    {"read_stderr", wrap_read_stderr, 1},
    {"popen", wrap_popen, 1},
    {"popen2", wrap_popen2, 1},
    {"popen3", wrap_popen3, 1},
    {"close", wrap_close, 1},
    {"set_blocking", wrap_set_blocking, 1},
    {"set_timeout", wrap_set_timeout, 1},
    {"get_count", wrap_get_count, 1},
    {"get_pid", wrap_get_pid, 1},
    {"get_command", wrap_get_command, 1},
    {"get_fd_in",  wrap_get_fd_in, 1},
    {"get_fd_out", wrap_get_fd_out, 1},
    {"get_fd_err", wrap_get_fd_err, 1},
    {0, 0}
};

void init_expect (void)
{
    PyObject *m, *d;
    m = Py_InitModule ("_expect", expect_methods);
    d = PyModule_GetDict (m);
}




