# btsutils -- Python module to interact with debbugs servers.
# Copyright (C) 2007 Gustavo R. Montesino
#
# Some parts of this code are based on reportbug-ng from Bastian Venthur
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import urllib
import re
from BeautifulSoup import BeautifulSoup

from bug import Bug
import BugExceptions
import BugTags
import BugStatus
import BugSeverity


class html:

    DEFAULT_URL = "http://bugs.debian.org/"
    
    def __init__(self, url=DEFAULT_URL, users=[]):
        self.url = url
        self.setUsers(users)

    def connect(self, url=DEFAULT_URL):
        """Change the debbugs server to use"""
        self.url = url

    def setUsers(self, users):
        self.users = users
    
    def get(self, bugnumber):
        """Returns a single bug through HTML parsing"""

        handle = urllib.urlopen(self.url + str(bugnumber))
        soup = BeautifulSoup(handle.read())

        if len(soup.h1.findAll("a")) == 0:
            raise BugExceptions.InvalidBugIdError("Invalid bug id: %s" % bugnumber)

        bug = Bug(forwarded='')

        severity_re = re.compile("^Severity:\s(.*);$", re.MULTILINE)
        tags_re = re.compile("^Tags:\s(.*);$", re.MULTILINE)

        # id and summary
        id = soup.h1.a
        bug.setBug(soup.h1.a.string[1:])
        bug.setSummary(self.decode_entities(soup.h1.br.nextSibling))

        # package
        package = soup.find(text = lambda(str): str.endswith("Package: "))
        bug.setPackage(self.decode_entities(package.nextSibling.string))

        # submitter
        submitter = soup.find(text = lambda(str): str.endswith("Reported by: "))
        bug.setSubmitter(self.decode_entities(submitter.nextSibling.string))

        # severity
        severity = soup.h3.find("em", attrs = { "class" : "severity"})
        if severity:
            bug.setSeverity(severity.string)
        else:
            severity = soup.h3.find(text = lambda(str): str.find("Severity") != -1)

            if severity:
                bug.setSeverity(severity_re.findall(severity)[0])
            else:
                bug.setSeverity("normal")

        # tags
        tags = soup.h3.find(text = lambda(str): str.find("Tags") != -1)
        if tags:
            tags = tags_re.findall(tags)[0]
            bug.setTags(tags.split(", "))
        else:
            bug.setTags([])

        # Status
        # FIXME: Is this thing right???
        forwarded = soup.h3.find(text = "Forwarded")
        if forwarded:
            bug.setStatus(BugStatus.FORWARDED)
            bug.setForwarded(forwarded.findNext("a").string)
        elif soup.h3.find(text = lambda(str): str.find("Done:") != -1):
            bug.setStatus(BugStatus.DONE)
        elif "fixed" in bug.getTags():
            bug.setStatus(BugStatus.FIXED)
        elif "pending" in bug.getTags():
            bug.setStatus(BugStatus.PENDING_FIXED)
        else:
            bug.setStatus(BugStatus.PENDING)

        # URL
        bug.setURL("%s/%s" % (self.url.rstrip('/'), bug.getBug()))

        return bug

    def queryToURL(self, query):
        """Transforms a query string into list of tuples
        
        Receives a btsutils query string and returns a list of tuples proper 
        to be used with urllib's urlencode to create a debbugs query"""

        args = []

        # Source package search
        if query.startswith("src:"):
            args = [ ("src", query[4:].strip()) ]

        # Binary package search
        elif query.startswith("pkg:"):
            args = [ ("pkg", query[4:].strip()) ]

        # Maintainer e-mail address search
        elif query.startswith("maint:"):
            args = [ ("maint", query[6:].strip()) ]

        # Submitter e-mail address search
        elif query.startswith("from:"):
            args = [ ("submitter", query[5:].strip()) ]

        # Usertags
        elif query.startswith("usertag:"):
            tags = query[8:].strip().replace(" ", ",")
            args = [ ("tag", tags) ]

        else:
            raise BugExceptions.InvalidQueryError("Invalid query: %s" % query)

        # Users to get usertags for
        if len(self.users) > 0:
            args.append(("users", ",".join(self.users)))

        return args;
    
    def query(self, query):
        """Returns a list with the bugs which match query."""

        queries = re.findall("([&|+-])?\s*\((.*?)\)", query)

        if queries == []:
            queries = [('', query)]

        result = []
        for query in queries:

            arguments = query[1].split("&")

            args = []
            for argument in arguments:
                current = []

                if argument.startswith("bug:"):
                   # Specifing a bug number with other stuff makes no sense,
                   # just return the specified bug and ignore everything else

                    current = [ self.get(argument[4:].strip()) ];
                    break

                else:
                    for tuple in self.queryToURL(argument):
                        args.append(tuple)

            if current == []:
                args = urllib.urlencode(args)
                url = "%s/%s?%s" % (self.url.rstrip("/"), "cgi-bin/pkgreport.cgi", args)
                handle = urllib.urlopen(url);
                soup = BeautifulSoup(handle)

                id_re = re.compile("^#(\d+):\s(.*)$")

                # Get bug sections (ie, bug groups by pending and severity)
                section_re = re.compile("^_\d_\d_\d$")
                sections = soup.findAll("a", attrs = { "name" : section_re })

                for s in sections:

                    # status and severity values for bugs in this session
                    status, severity = s["name"][1:-2].split("_")
                    status = BugStatus.VALID[int(status)]
                    severity = BugSeverity.VALID[int(severity)]

                    # List of bugs on this session
                    buglist = s.parent.findNext("ul", attr = { "class" : "bugs" })
                    buglist = buglist.findAll("li")

                    for report in buglist:
                        bug = Bug(severity=severity, status=status)

                        # id and summary
                        (id, summary) = id_re.findall(report.contents[0].string)[0]
                        bug.setBug(id)
                        bug.setSummary(self.decode_entities(summary))

                        # package
                        package = report.find(text = "Package: ")
                        if package == None:
                            package = report.find(text = "Packages: ")
                        bug.setPackage(self.decode_entities(package.nextSibling.string))

                        # submitter
                        submitter = report.find(text =  lambda(str): str.endswith("Reported by: "))
                        bug.setSubmitter(self.decode_entities(submitter.nextSibling.string))

                        # tags
                        tagstxt = report.find(text = lambda(str): str.endswith("Tags: "))
                        if tagstxt:
                            for tag in tagstxt.nextSibling.string.split(", "):
                                try:
                                    bug.addTag(tag)
                                except BugExceptions.InvalidBugTagError:
                                    bug.addUserTag(tag)
                        else:
                            bug.setTags([])

                        # forwarded
                        forwarded = report.find(text="Forwarded")
                        if forwarded:
                            bug.setForwarded(forwarded.findNext("a").string)
                        else:
                            bug.setForwarded('')

                        # URL
                        bug.setURL("%s/%s" % (self.url.rstrip('/'), bug.getBug()))

                        current.append(bug)

            if query[0] == '':
                result = current

            elif query[0] == '&':
                new = []
                for bug in result:
                    for cur_bug in current:
                        if bug.getBug() == cur_bug.getBug():
                            new.append(bug)
                            break
                result = new

            elif query[0] == '|' or query[0] == '+':
                ids = [ bug.getBug() for bug in result ]
                for bug in current:
                    if not bug.getBug() in ids:
                        result.append(bug)

            elif query[0] == '-':
                toremove = [ bug.getBug() for bug in current ]
                for bug in result[:]:
                    if bug.getBug() in toremove:
                        result.remove(bug)
            
        return result

    def decode_entities(self, htmlstring):
        """Decodes html entities from string"""

        # Debbugs uses HTML::Entites perl module for encoding "dangerous
        # characters"; so we have to decode them. According to perl docs,
        # The encoded entities which interests us are <, &, >, ' and "

        htmlstring = htmlstring.replace("&lt;", "<")
        htmlstring = htmlstring.replace("&amp;", "&")
        htmlstring = htmlstring.replace("&gt;", ">")
        htmlstring = htmlstring.replace("&quot;", "\"")
        htmlstring = htmlstring.replace("&#39;", "'")

        return htmlstring

# vim: tabstop=4 expandtab shiftwidth=4
