########################################################################
#
# File Name:            XmlWriter.py
#
# Documentation:        http://docs.4suite.org/4XSLT/XmlWriter.py.html
#
"""
Implements the XML output writer for XSLT processor output
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc., USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import os, re, string
import xml.dom.ext
from xml.dom.ext.Printer import TranslateCdata, TranslateCdataAttr, utf8_to_code
from xml.dom.html import TranslateHtmlCdata
from xml.xslt import XSL_NAMESPACE, NullWriter, XsltException, Error
from xml.dom.html import HTML_4_TRANSITIONAL_INLINE, HTML_4_STRICT_INLINE
from xml.dom import XML_NAMESPACE
from xml.dom.html import HTML_FORBIDDEN_END

GENERATED_PREFIX = "genprefix%s"

class ElementData:
    def __init__(self, name, cdataElement, attrs, extraNss=None):
        self.name = name
        self.cdataElement = cdataElement
        self.attrs = attrs
        self.extraNss = extraNss or {}
        return


class XmlWriter(NullWriter.NullWriter):
    def __init__(self, outputParams, stream=None):
        NullWriter.NullWriter.__init__(self, outputParams, stream)
        self._outputParams.setDefault('encoding', 'UTF-8')
        self._outputParams.setDefault('indent', 'no')
        self._outputParams.setDefault('mediaType', 'text/xml')
        self._currElement = None
        self._namespaces = [{'': '', 'xml': XML_NAMESPACE}]
        self._indent = ''
        self._nextNewLine = 0
        self._cdataSectionElement = 0
        self._first_element = 1
        self._cached = []
        self._lastChars = ''
        return

    def _doctype(self, docElem):
        external_id = ''
        if self._outputParams.doctypePublic and self._outputParams.doctypeSystem:
            external_id = ' PUBLIC "' + self._outputParams.doctypePublic + '" "' + self._outputParams.doctypeSystem + '"'
        elif self._outputParams.doctypeSystem:
            external_id = ' SYSTEM "' + self._outputParams.doctypeSystem + '"'
        if external_id:
            self._stream.write('<!DOCTYPE %s%s>\n' % (docElem, external_id))
        self._first_element = 0

    def startDocument(self):
        if self._outputParams.omitXmlDeclaration=='no':
            self._stream.write("<?xml version='%s' encoding='%s'" % (
                self._outputParams.version,
                self._outputParams.encoding))
            if self._outputParams.standalone is not None:
                self._stream.write(" standalone='%s'" % self._outputParams.standalone)
            self._stream.write("?>\n")
        return
        
    def endDocument(self):
        self._completeLastElement(0)
        return

    def text(self, text, escapeOutput=1, asis=0):
        #print "text", repr(text), escapeOutput
        self._completeLastElement(0)
        if escapeOutput:
            if text and text[0] == '>':
                last_chars = self._lastChars
            else:
                last_chars = ''
            text = TranslateCdata(
                text,
                self._outputParams.encoding,
                last_chars,
                markupSafe=self._cdataSectionElement
                )
        else:
            text = TranslateCdata(
                text,
                self._outputParams.encoding,
                '',
                markupSafe=1
                )
        #FIXME: would break under perverse condition where user builds
        #erroneous output one character at a time.
        #True fix requires stateful buffering of the output stream
        self._lastChars = text[-2:]
        self._stream.write(text)
        self._nextNewLine = 0
        return

    def attribute(self, name, value, namespace=''):
        #print "attribute", name, value, namespace
        if not self._currElement:
            raise XsltException(Error.ATTRIBUTE_ADDED_AFTER_ELEMENT)
        value = TranslateCdata(value, self._outputParams.encoding)
        (prefix, local) = xml.dom.ext.SplitQName(name)
        if namespace:
            if not self._namespaces[-1].has_key(prefix):
                # A new prefix/namespace pair
                self._namespaces[-1][prefix] = namespace
            elif self._namespaces[-1][prefix] != namespace:
                # If the prefix/namespace pair doesn't match what we already
                # have, generate a new prefix
                ctr = 1
                while self._namespaces[-1].has_key(prefix):
                    prefix = GENERATED_PREFIX % ctr
                    if self._namespaces[-1].get(prefix) == namespace:
                        break
                    ctr = ctr + 1
                else:
                    self._namespaces[-1][prefix] = namespace
                name = string.joinfields([prefix, local], ':')
        self._currElement.attrs[name] = TranslateCdataAttr(value)
        return

    def processingInstruction(self, target, data):
        self._completeLastElement(0)
        target = string.strip(TranslateCdata(target, self._outputParams.encoding, ''))
        data = string.strip(TranslateCdata(data, self._outputParams.encoding, ''))
        pi = '<?%s %s?>' % (target, data)
        if self._outputParams.indent == 'yes':
            self._stream.write("\n%s%s" % (self._indent, pi))
        else:
            self._stream.write(pi)
        self._nextNewLine = 1
        return

    def comment(self, body):
        self._completeLastElement(0)
        body = utf8_to_code(body, self._outputParams.encoding)
        comment = "<!--%s-->" % body
        if self._outputParams.indent == 'yes':
            self._stream.write("\n%s%s" % (self._indent, comment))
        else:
            self._stream.write(comment)
        self._nextNewLine = 1
        return

    def startElement(self, name, namespace='', extraNss=None):
        #print "startElement", name, namespace, extraNss
        extraNss = extraNss or {}

        self._completeLastElement(0)

        if self._first_element:
            self._doctype(name)

        (prefix, local) = xml.dom.ext.SplitQName(name)
        cdatas_flag = (namespace, local) in self._outputParams.cdataSectionElements
        self._currElement = ElementData(name, cdatas_flag, {}, extraNss)
        self._namespaces.append(self._namespaces[-1].copy())
        self._namespaces[-1][prefix] = namespace
        return

    def endElement(self, name):
        #print "endElement", name
        if self._currElement:
            elementIsEmpty = 1
            self._completeLastElement(1)
        else:
            elementIsEmpty = 0
        if self._outputParams.indent=='yes':
            self._indent = self._indent[:-2]
        if self._cdataSectionElement:
            self._stream.write(']]>')
            self._cdataSectionElement = 0
        if self._outputParams.indent=='yes' and self._nextNewLine and not elementIsEmpty:
            self._stream.write('\n' + self._indent)
        self._stream.write((not elementIsEmpty) and ('</%s>' % name) or '')
        self._nextNewLine = 1
        del self._namespaces[-1]
        return

    def _completeLastElement(self, elementIsEmpty):
        #print "_completeLastElement", elementIsEmpty, self._currElement
        if self._currElement:
            elem = self._currElement
            if self._outputParams.indent=='yes' and self._nextNewLine:
                self._stream.write('\n' + self._indent)
            self._stream.write('<' + elem.name)
            for (name, (value, delimiter)) in elem.attrs.items():
                self._stream.write(' %s=%s' % (name,delimiter))
                self._stream.write(value)
                self._stream.write(delimiter)
            #Handle namespaces
            nss = elem.extraNss
            nss.update(self._namespaces[-1])
            for prefix in nss.keys():
                ns = nss[prefix]
                prev_ns = self._namespaces[-2].get(prefix)
                if ns != prev_ns:
                    if prefix:
                        self._stream.write(" xmlns:%s='%s'" % (prefix, ns))
                    else:
                        self._stream.write(" xmlns='%s'" % ns)
            self._namespaces[-1] = nss
            if elementIsEmpty:
                self._stream.write('/>')
            else:
                self._stream.write('>')
                if self._currElement.cdataElement:
                    self._stream.write('<![CDATA[')
                    self._cdataSectionElement = 1
            self._nextNewLine = 1

            if self._outputParams.indent=='yes':
                self._indent = self._indent + '  '
            self._currElement = None
        return
