# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python binding timing example
#
# Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: Loïc Molinari <loic@fluendo.com>

import sys, pgm, gobject
from pgm.timing import controller, modifier, keyframe, ticker

# Update the position
def on_update_pass(viewport, tck):
    tck.tick()

# Terminate the mainloop on a delete event
def on_delete(viewport, event, ctrl):
    ctrl.stop()
    pgm.main_quit()

# Key press handler
def on_key_press(viewport, event):
    if event.keyval == pgm.keysyms.f:
        viewport.fullscreen = not viewport.fullscreen

def main(args):
    # OpenGL viewport creation
    gl = pgm.viewport_factory_make('opengl')
    gl.title = 'Animated image'

    # Canvas and image drawable creation
    cvs = pgm.Canvas()
    img = pgm.Image()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(cvs)

    # Load an image
    img.set_from_file('examples/pictures/fluendo.png')
    img.bg_color = (255, 255, 255, 0)

    # Center the image
    img.position = (1.5, 1.0, 0.0)

    # A drawable needs to be shown
    img.show()

    # Add it to the middle layer of the canvas
    cvs.add(pgm.DRAWABLE_MIDDLE, img)

    # Set the ticker used by controllers
    tck = ticker.Ticker()
    controller.Controller.set_ticker(tck)

    # Y position timeline
    tml_y = [keyframe.KeyFrame(2.0, 0.0),
             keyframe.KeyFrame(0.25, 1.0)]
    mdf_y = modifier.Modifier([img], 'y', tml_y)

    # Y position timeline
    tml_ry = [keyframe.KeyFrame(3.1415, 0.0),
              keyframe.KeyFrame(3.1415 * 2.0, 1.0)]
    mdf_ry = modifier.Modifier([img], 'ry', tml_ry)

    # Create the animation controller
    ctrl = controller.Controller(duration=1000,
                                 repeat_behavior=controller.REVERSE,
                                 repeat_count=controller.INFINITE,
                                 transformation=controller.DECELERATE,
                                 modifiers=[mdf_y, mdf_ry])

    # Start the controller
    ctrl.start()

    # Let's start the mainloop
    gl.connect('delete-event', on_delete, ctrl)
    gl.connect('key-press-event', on_key_press)
    gl.connect('update-pass', on_update_pass, tck)
    gl.show()
    pgm.main()

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))
