// pvsynthesizer.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/
#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "envelope.h"
#include "localdefs.h"
#include "pvocrequester.h"
#include "request.h"
#include "phasevocoder.h"
#include "pipeaction.h"
#include "pvsynthesizer.h"
#include "pvocdata.h"

#undef debug

class PVSynthesisRequester : public PvocRequester {
	friend class PVSynthesizer;
protected:
	PVSynthesisRequester(const char* , PhaseVocoder::Info &,
	                     PvocData *, double dur);
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	PvocData* _pvdata;
	static float _lowLimit;
	static float _highLimit;
	static float _gain;
	double _synthDur;
};

float PVSynthesisRequester::_lowLimit = 0.0;
float PVSynthesisRequester::_highLimit = 0.0;
float PVSynthesisRequester::_gain = 1.0;

PVSynthesisRequester::PVSynthesisRequester(const char* title,
                                           PhaseVocoder::Info &info,
                                           PvocData* data,
                                           double dur)
    : PvocRequester(title, info), _pvdata(data), _synthDur(dur) {
        if (_highLimit == 0.0 || _highLimit > _pvdata->sRate()/2.0)
	    _highLimit = _pvdata->sRate()/2.0;	// limit to NQF of anal data
}
		
void
PVSynthesisRequester::configureRequest(Request* request) {
	request->appendLabel("All \"0\" values will be set to defaults.");
	request->appendValue("Output frame size (samples):",
		&_pvocInfo.outputFrameSize, NonNegativeIntegers);
	request->appendValue("Pitch scaling factor:",
			     &_pvocInfo.pchScaleFactor);
	request->appendValue("Spectral envelope warp factor:", &_pvocInfo.warp);
	request->appendValue("Gain:", &_gain);
	Range allFreqs(0.0F, _pvdata->sRate()/2.0);
	request->appendValue("Lower band limit (hz.):", &_lowLimit, allFreqs, true);
	request->appendValue("Upper band limit: (hz.)", &_highLimit, allFreqs, true);
#ifdef NOT_FINISHED
	request->appendChoice("Use envelope for factor", "|No|Yes|",
			      &useEnvelope, true);
#endif
	PvocRequester::configureRequest(request);
}

boolean
PVSynthesisRequester::confirmValues() {
	int status = true;
	char msg[120];
	char msg2[120];
	double sampleRate = _pvocInfo.samplingRate;
	if(sampleRate != double(_pvdata->sRate())) {
	    sprintf(msg,
	        "Warning: Phase Vocoder data samprate (%d) != selection samprate (%f).",
	            _pvdata->sRate(), sampleRate);
		sprintf(msg2,
			"Resynthesized length and spectrum will be skewed by a factor of %f.",
			sampleRate / double(_pvdata->sRate()));
	    status = Application::confirm(msg, msg2, "Continue anyway?", Cancel);
	}
	if(!status)
		return status;
	PvocRequester::confirmValues();
	_pvocInfo.firstBand = _pvdata->getBandNumber(_lowLimit);
	_pvocInfo.lastBand = _pvdata->getBandNumber(_highLimit);
	// Scale synth gain
	_pvocInfo.inputScalingFactor *= _gain;
	double analDur = _pvdata->duration();
	printf("anal dur = %f, selected dur = %f, ratio = %f\n",
	       analDur, _synthDur, _synthDur/analDur);
	printf("new timeScaleFactor = %f\n\n",
	       _pvocInfo.timeScaleFactor /= _synthDur/analDur);
	return status;
}

//********

PhaseVocoder::Info PVSynthesizer::_savedPvocInfo(0.0, 0.0f, 0);

PVSynthesizer::PVSynthesizer(Data* data, PvocData* anal)
	: QueuedOutputFunction(data, 1024),
	  _pvocInfo(_savedPvocInfo),
	  _pvoc(nil),
	  _pvdata(anal) {
    // always reset these to match current data
    _pvocInfo.samplingRate = data->sRate();                   // srate
    _pvocInfo.inputScalingFactor = (data->dataType() == FloatData) ? 1.0 : 32767.0; // scaling
    _pvocInfo.fftSize = anal->channels() - 2,                     // fftsize
    _pvocInfo.fundFreq = 0;                                       // fundfrq
    _pvocInfo.inputFrameOffset = anal->frameOffset();             // infrmoffset
    _pvocInfo.outputFrameSize = anal->channels() - 2;             // outfrmsize
    _pvocInfo.timeScaleFactor = 1.0;	// default
    _pvdata->ref();
}

PVSynthesizer::PVSynthesizer(
	Data* data,  PvocData* anal,
	int L, int I, double T, int i, int j, boolean useKaiser)
		: QueuedOutputFunction(data, 2*L),
		  _pvocInfo(
			data->sRate(),                           // srate	
			(data->dataType() == FloatData) ? 1.0 : 32767.0,
		  	anal->channels() - 2,                    // fftsize
		  	0,
		  	anal->frameOffset(),                     // infrmoffset
			L, I, T, i, j,
			0.0,                                     // warp
			useKaiser
		  ),
		  _pvoc(nil), _pvdata(anal) {
	_pvdata->ref();
	initialize();
}

Modifier *
PVSynthesizer::create(DataEditor *de) {
	return nil;
}

PVSynthesizer::~PVSynthesizer() {
	Resource::unref(_pvdata);
	delete _pvoc;
}

void
PVSynthesizer::restoreState() {
	Super::restoreState();
	_pvoc->reset();
	_currentFrame.set(0, 0);
}

Requester *
PVSynthesizer::createRequester() {
	return new PVSynthesisRequester(
		"Phase Vocoder Resynthesis into Selected Region:",
		_pvocInfo,
		_pvdata,
		double(target()->length()) / sampRate()
	);
}

void
PVSynthesizer::initialize() {
	_pvoc = new PhaseVocoder(_pvocInfo);
	okIf(_pvoc->isGood());
	if (ok()) {
		setOutQueueSize(2 * _pvoc->getOutputFrameOffset());
		Super::initialize();
	}
}

void
PVSynthesizer::saveConfig() {
    _savedPvocInfo = _pvocInfo;
}


typedef int (QueuedOutputFunction::*(InQueueFunction))(double *, int);

class PVInPipeAction : public InPipeAction {
public:
	PVInPipeAction(PVSynthesizer *pvs, InQueueFunction fun)
		: pv(pvs), pfun(fun) {}
	redefined int add(double *val, int count) {
	    return (pv->*pfun)(val, count);
	}
private:
	PVSynthesizer* pv;
	InQueueFunction pfun;
};

int
PVSynthesizer::doProcessing() {
	BUG("PVSynthesizer::doProcessing");
	int status = (_pvdata->nFrames() > _currentFrame);
	if(status) {
#ifdef debug
		printf("PVSynthesizer::doProcessing: cloning frame %d\n", _currentFrame.intMin());
#endif
		Data* frame = _pvdata->clone(_currentFrame);
		InPipeAction* action = new PVInPipeAction(
			this, &PVSynthesizer::addToOutQueue
		);
		status = _pvoc->runSynthesis(frame, action);
		Resource::unref(frame);
		_currentFrame += _pvocInfo.timeScaleFactor;
	}
	return status;
}
