// iallpole.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include "application.h"
#include "iallpole.h"
#include "lpcdata.h"
#include "sound.h"
#include "array.h"
#include "request.h"
#include "requester.h"
#include <math.h>

class InverseAllPoleRequester : public TitledRequester {
public:
	InverseAllPoleRequester(InverseAllPole *);
protected:
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
private:
	InverseAllPole* client;
	int foffset;
	double frate;
};

InverseAllPoleRequester::InverseAllPoleRequester(InverseAllPole* i)
	: TitledRequester("Linear Predictive (LPC) Analysis of Selected Region:"),
	  client(i),
	  foffset(i->offset()),
	  frate(i->framerate()) {}

void
InverseAllPoleRequester::configureRequest(Request* request) {
	request->appendLabel("(Frame size will be 2 x frame offset)");
	request->appendValue("Number of filter poles:", &client->npoles,
	                     Range(12, 64), true);
	request->appendValue("Frame offset (samples):", &foffset,
	                     Range(13, client->target()->length() / 2), true);
	request->appendValue("Frame rate (frames/sec):", &frate,
	                     Range(1.0, client->sampRate()/13.0), true);
}

boolean
InverseAllPoleRequester::confirmValues() {
	boolean ret = true;
	if(foffset * 2 <= client->npoles
			|| client->sampRate()/frate <= client->npoles)
		ret = Application::confirm(
			"The calculated frame size may be too small",
		        "for this number of filter poles.",
		        "Attempt anyway?");
	return ret && client->setBaseValues(foffset * 2, frate, foffset);
}

//********

int InverseAllPole::_savedNPoles = LPCData::defaultNumberOfPoles();
int InverseAllPole::_savedOffset = 
	  int(Sound::defaultSampleRate()/LPCData::defaultFrameRate());
float InverseAllPole::_savedFrameRate = 0.0;

InverseAllPole::InverseAllPole(Data* data, int poles, int framelength,
		int frameoffset)
		: ArrayFunction(data, framelength, frameoffset),
		gmatrix(nil), b(nil), v(nil), c(nil), npoles(poles) {
	initialize();
}

InverseAllPole::InverseAllPole(Data* data)
	: ArrayFunction(data, _savedOffset * 2, _savedOffset),
	gmatrix(nil), b(nil), v(nil), c(nil), npoles(_savedNPoles) {}

InverseAllPole::~InverseAllPole() {
	delete [] b;
	delete [] v;
	delete [] c;
	delete gmatrix;
}

Requester *
InverseAllPole::createRequester() {
	return new InverseAllPoleRequester(this);
}

void
InverseAllPole::initialize() {
	gmatrix = create2DArray(npoles, npoles);
	b = new double[npoles];
	v = new double[npoles];
	c = new double[npoles];
	ArrayFunction::initialize();
	setAnalysis(
		new LPCData(analysisLength(),
		npoles, sampRate(), framerate())
	);
}

void
InverseAllPole::saveConfig()
{
	_savedNPoles = npoles;
	_savedOffset = offset();
	_savedFrameRate = framerate();
}

int
InverseAllPole::operator () (double *signal, Data *frame) {
	double sum, sumx, sumy;
	int i, l, k;
	int npm1 = npoles - 1;
	int fsize = framesize();
	register double* sig = signal;
	register double* vp = v;
	register double* bp = b;
	for (i = 0; i < npoles; ++i)  {
		int ip1 = i + 1;
		sum = 0;
		for (k = npoles; k < fsize; ++k)
			sum += *(sig+k-ip1) * *(sig+k);
		*(vp+i) = -sum;
		if (i != npm1)  {
			int limit = npoles - ip1;
			for (l=0; l < limit; ++l)  {
				int lp1 = l + 1;
				sum +=	*(sig+npoles-ip1-lp1) * *(sig+npoles-lp1) -
					*(sig+fsize-ip1-lp1) * *(sig+fsize-lp1);
				gmatrix[ip1+l][l] = gmatrix[l][ip1+l] = sum;
			}
		}
	}
	sum = 0;
	for (k = npoles; k < fsize; ++k)
		sum += *(sig+k) * *(sig+k);
	sumy = sumx = sum;
	for (l = 0; l < npoles; ++l)  {
		int lp1 = l + 1;
		double tsig1 = *(sig+npoles-lp1);
		double tsig2 = *(sig+fsize-lp1);
		sum += tsig1*tsig1 - tsig2*tsig2;
		gmatrix[l][l] = sum;
	}
	if(!gauss(vp, bp))
		return false;
	for (i = 0; i < npoles; ++i)
		sumy = sumy - *(bp+i) * *(vp+i);

	// store "mean squared" values to avoid having to square them again
	double ms1 = sumx/double(fsize - npoles);
	double ms2 = sumy/double(fsize - npoles);

	frame->set(sqrt(ms2), 0, 0);							// Residual RMS
	frame->set(sqrt(ms1), 0, 1);							// Original RMS
	frame->set(ms2/ms1, 0, 2);								// Error
	// reverse the coefs & change sign, load into frame
	int loc;
	for (bp = b, loc = npoles + nDataValues - 1;
			bp - b < npoles; ++bp, --loc) {
		frame->set(-(*bp), 0, loc);		// coeffs loaded inverted, rev. order
	}
	return true;
}

int
InverseAllPole::gauss(double *bold, double *out) {
	double amax, dum, pivot;
	int i, istar=0;
	int npm1 = npoles - 1;
	register double* cp = c;
	
	for (i=0; i < npoles ;++i)
		*(cp+i) = *bold++;

	/* eliminate i-th unknown */
	for (i = 0; i < npm1; ++i)  {        /* find largest pivot */
		amax = 0.0;
		for (int ii = i; ii < npoles; ++ii)  {
			double tmpmax = 0;
			if ((tmpmax = fabs(gmatrix[ii][i])) >= amax)  {
				istar = ii;
				amax = tmpmax;
			}
		}
		if (amax < 1e-20) {
			Application::alert("Error:", "cannot analyze frames with zero amplitude.");
			return false;
		}
		double* gmpi = gmatrix[i];				// optimizing
		int j;
		for (j=0; j < npoles ;++j)  {		    // switch rows
			dum = gmatrix[istar][j];
			gmatrix[istar][j] = *(gmpi+j);
			*(gmpi+j) = dum;
		}
		dum = *(cp+istar);
		*(cp+istar) = *(cp+i);
		*(cp+i) = dum;
		for (j = i + 1; j < npoles; ++j)  {		// pivot
			double* gmpj = gmatrix[j];			// optimizing again
			pivot = *(gmpj+i) / *(gmpi+i);
			*(cp+j) = *(cp+j) - pivot * *(cp+i);
			for (int k = 0; k < npoles; ++k) {
				double& j_k = *(gmpj+k);		// to save one set of [][]
				j_k = j_k - pivot * *(gmpi+k);
			}
		}
	}
	/* return if last pivot is too small */
	if (fabs(gmatrix[npm1][npm1]) < 1e-20) {
			Application::alert("Error:", "cannot analyze frames with zero amplitude.");
		return false;
	}

	*(out+npm1) = *(cp+npm1) / gmatrix[npm1][npm1];
	/* back substitute */
	for (int k = 0; k < npm1; ++k)  {
		int l = npm1 - (k+1);
		*(out+l) = *(cp+l);
		double* gmpl = gmatrix[l];
		for (int m = l + 1; m < npoles; ++m)
			*(out+l) += -*(gmpl+m) * *(out+m);
		*(out+l) /= *(gmpl+l);
	}
	return(true);
}
