#!/usr/bin/perl

# Copyright (c) Kim Holviala <kim@holviala.com> 1997-2002

require 5.002;


# Configuration variables ---------------------------------------------------

# Software version
$CONFIG{'SoftwareVersion'} = "v2.3.36 (2005-03-14)";

# Path to scripts and .cfg's
$_ = $0; tr#\\#/#;

if (m#/#) {
	s#(.*/).*$#$1#;
	$CONFIG{'ScriptPath'} = $_;
}

# Are we running under Windows NT?
$WinNT = 1 if ($ENV{'OS'} =~ /Windows/);

# Fixed domainname of this server
$_ = $ENV{'HTTP_HOST'} ? $ENV{'HTTP_HOST'} : $ENV{'SERVER_NAME'};
tr/A-Z/a-z/;
s/\.*$//;
$CONFIG{'ServerName'} = $_;

# Full name & path of this script
$ENV{'SCRIPT_NAME'} = $ENV{'REDIRECT_URL'} if ($ENV{'REDIRECT_URL'});
$CONFIG{'ScriptName'} = ($ENV{'HTTPS'} eq 'on' ? 'https://' : 'http://') . $CONFIG{'ServerName'} . $ENV{'SCRIPT_NAME'};

# Set network timeout stuff
if (!$WinNT) {
	$SIG{ALRM} = sub { &error('ErrorTimeout', 1000); };
	$timeoutOn = 'alarm $CONFIG{\'Timeout\'}';
	$timeoutOff = 'alarm 0';
}

# Seed for random numbers
srand(time() ^ ($$ + ($$ << 15)));
$CONFIG{'RandomNumber'} = int(rand 100000000);

# OS-specific file stuff
if ($WinNT) {
	binmode STDOUT;
	binmode STDIN;
}

# CRLF as required by RFC's
$crlf = "\x0d\x0a";

# Enumerate weekdays
%weekdays = ('Sun', 0,
			'Mon', 1,
			'Tue', 2,
			'Wed', 3,
			'Thu', 4,
			'Fri', 5,
			'Sat', 6);

# Enumerate months
%months = ('Jan', 0,
			'Feb', 1,
			'Mar', 2,
			'Apr', 3,
			'May', 4,
			'Jun', 5,
			'Jul', 6,
			'Aug', 7,
			'Sep', 8,
			'Oct', 9,
			'Nov', 10,
			'Dec', 11);


# Main ======================================================================

# Load mandatory configuration file & handle query

&loadConfig('server.cfg', 1);
&getQuery;
&getCookies;


# Load correct configuration files

if ($FORM{'u'}) {
	if (!&loadConfig($CONFIG{'TempPath'}.$FORM{'u'}, 1)) {
		&error('ErrorLoggedout', 1010) unless $FORM{'do'} eq 'logout';
	}

	&form2config;

	$CONFIG{'LinkQueryString'} = "p=$FORM{'p'}&u=$FORM{'u'}&r=" . int(rand 100000000);
}
else {
	&loadConfig('main.cfg', 1);
	&loadConfig('server.cfg', 1);
	&form2config;

	&loadConfig("$CONFIG{'Language'}.cfg");
}


# Tweak some configuration variables

$CONFIG{'LanguageOptions'} =~ s/ SELECTED//g;
$CONFIG{'LanguageOptions'} =~ s/\"$CONFIG{'Language'}\"/$& SELECTED/;


# Handle different actions, part #1

for ($FORM{'do'}) {
	/^$/		and do { &loginForm; exit; };

	/logout/	and do { &loadModule('network.cgi'); &logout; last; };
	/loginhelp/	and do { &loadModule('misc.cgi'); &loginHelp; last; };
	/savepwd/	and do { &loadModule('misc.cgi'); &savePassword; last; };
}


# Handle login

if ($FORM{'do'} eq 'login') {
	&loadModule('network.cgi');
	&login;
}


# Already logged in - check security

else {
	$FORM{'msg'} =~ tr/0-9//cd;
	$FORM{'p'} =~ tr/0-9//cd;
	$FORM{'u'} =~ tr/0-9//cd;

	-e $CONFIG{'TempPath'}.$FORM{'p'} or &error('ErrorLoggedout', 1050);
	-e $CONFIG{'TempPath'}.$FORM{'u'} or &error('ErrorLoggedout', 1060);

	if ($COOKIE{'SessionID'} and !$CONFIG{'HasCookies'}) {
		$CONFIG{'HasCookies'} = 1;
		$CONFIG{'SessionID'} = $COOKIE{'SessionID'};
	}

	if ($CONFIG{'HasCookies'}) {
		if ($CONFIG{'SessionID'} ne $COOKIE{'SessionID'}) { &error('ErrorIPChanged', 666); }
	}

# Make sure IP is within the same C-class 

	$ip = $ENV{'REMOTE_ADDR'};
	$ip =~ s/\d+$//;

#	$CONFIG{'RemoteAddr'} eq $ip or &error('ErrorIPChanged', 1070, $CONFIG{'RemoteAddr'} . "xxx -> ${ip}xxx");


# Show options if name or email missing

	if (!$CONFIG{'RealName'} or !$CONFIG{'RealEmail'}) {

		$FORM{'do'} = 'options' unless $FORM{'do'} =~ /logout/;
	}
}


# Handle different actions, part #2

for ($FORM{'do'}) {
	/login/		and do { last; };
	/confirm/	and do { &loadModule('compose.cgi'); &confirmSend; last; };

	/inbox/		and do { &inbox; last; };
	/delete/	and do { &delete; last; };
	/checkmail/	and do { &loadModule('network.cgi'); &checkmail; last; };
	/read/		and do { &read; last; };
	/printable/	and do { &loadModule('misc.cgi'); &printable; last; };
	/source/	and do { &loadModule('network.cgi'); &viewSource; last; };

	/compose/	and do { &loadModule('compose.cgi'); &compose; last; };
	/(reply|repall|forward|redirect)/ and do { &loadModule('compose.cgi'); &reply; last; };
	/send/		and do { &loadModule('compose.cgi'); &sendMail; last; };

	/savedraft/	and do { &loadModule('compose.cgi'); &saveDraft; last; };
	/showdraft/	and do { &loadModule('compose.cgi'); &showDraft; last; };

	/download/	and do { &loadModule('misc.cgi'); &download; last; };
	/attach/	and do { &loadModule('misc.cgi'); &attach; last; };
	/saveaddress/ and do { &loadModule('misc.cgi'); &saveAddress; last; };

	/abook/		and do { &loadModule('misc.cgi'); &addressbook; last; };
	/options/	and do { &loadModule('misc.cgi'); &options; last; };
	/saveopt/	and do { &loadModule('misc.cgi'); &saveOptions; last; };

	/spellcheck/ and do { &loadModule('misc.cgi'); &checkSpelling; last; };
	/addword/	and do { &loadModule('misc.cgi'); &addWord; last; };

	/click/		and do { &loadModule('misc.cgi'); &adClick; last; };
	/showkey/	and do { &loadModule('misc.cgi'); &showKey; last; };
	/keylist/	and do { &loadModule('misc.cgi'); &keyList; last; };

	/help/		and do { &loadModule('misc.cgi'); &showHelp; last; };

# Show inbox if 'do' was invalid
	&inbox;
}


# Finish the session

#if ($CONFIG{'Debug'}) {
#	print "<XMP>\n";
#	&dumpHash(%CONFIG);
#	print "</XMP>\n";
#}



&saveConfig;

exit;


# Subroutines ===============================================================

# Load an external module ---------------------------------------------------

sub loadModule {

	my ($module) = shift;

	if ($module !~ /[\\\/]/) {
		$module = $CONFIG{'ScriptPath'}.$module;
	}

	return eval { require $module; };
}


# Fork a new process --------------------------------------------------------

sub forkProcess {

	my ($code, $reverse) = @_;

	undef $pid;

# Handle parent process
	if (eval { $pid = fork; }) {

		if ($reverse) {
			eval $code;
			exit;
		}
		else {
			return;
		}
	}

# Handle child
	elsif (defined $pid) {

		if ($reverse) {
			return;
		}
		else {
			$noErrorMessage = 1;
			eval $code;
			exit;
		}
	}

# Fork failed -> handle both parent & the child
	else {
		eval $code;
	}
}


# Get current date in RFC format --------------------------------------------

sub getDate {

	($sec,$min,$hour,$day,$mon,$year,$wday) = gmtime;

	return (Sun,Mon,Tue,Wed,Thu,Fri,Sat)[$wday] . ", $day " .
		(Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec)[$mon] . ' ' .
		($year + 1900) . ' ' . 
		($hour < 10 and '0') . "$hour:" .
		($min < 10 and '0') . "$min:" .
		($sec < 10 and '0') . "$sec GMT";
}


# Print normal HTTP headers -------------------------------------------------

sub httpHeaders {

	return if $httpHeadersPrinted;

	$now = &getDate;

	print "$ENV{'SERVER_PROTOCOL'} 200 OK${crlf}" if $CONFIG{'FullHTTPHeaders'};
	print "Date: $now${crlf}Expires: $now${crlf}",
		"Pragma: no-cache${crlf}Cache-Control: no-cache${crlf}";
	print "Set-Cookie: SessionID=" . int(rand 100000000) . $crlf unless $COOKIE{'SessionID'};

	$httpHeadersPrinted = 1;
}


# Redirect browser ----------------------------------------------------------

sub redirect {
	my ($url) = shift;

	print "$ENV{'SERVER_PROTOCOL'} 302 Moved Temporarily$crlf" if $CONFIG{'FullHTTPHeaders'};

	print "Location: ${url}${crlf}Connection: close${crlf}Date: ", &getDate,
		"${crlf}Content-Type: text/html${crlf}${crlf}",
		"<HTML><HEAD>\n<TITLE>302 Moved Temporarily</TITLE>\n</HEAD><BODY>\n<H1>Moved Temporarily</H1>\n",
		"The document has moved <A HREF=\"${url}\">here</A>.<P>\n</BODY></HTML>\n";

	&flush;
}


# Handle loginform ----------------------------------------------------------

sub loginForm {

# Handle stuff
	$CONFIG{'JavaScript'} = 1;		# Force JavaScript ON

	if ($FORM{'p'} > 0) {
		$CONFIG{'Login'} = '';
		$CONFIG{'Server'} = '';
	}

	$CONFIG{'Server'} ||= $CONFIG{'DefaultMailserver'};

# Print the page
	&httpHeaders;
	print "Content-Type: text/html$crlf$crlf", &getTemplate('login.html');
	&flush;
}


# Save user's configuration -------------------------------------------------

sub saveConfig {

	open(FILE, ">$CONFIG{'TempPath'}$FORM{'u'}");
	flock(FILE, 2);
	seek FILE, 0, 2;

# Back up user's configuration - it sometimes gets deleted for no apparent reason..... :-(
	rename("$CONFIG{'CachePath'}$CONFIG{'Userfile'}", "$CONFIG{'CachePath'}$CONFIG{'Userfile'}_backup");

	open(CACHE, ">$CONFIG{'CachePath'}$CONFIG{'Userfile'}");
	flock(CACHE, 2);
	seek CACHE, 0, 2;

	foreach (keys %CONFIG) {
		next unless $_;
		next unless $CONFIG{$_};

		/^(RandomNumber|LinkQueryString|Advertisement|AdvertisementHREF|.*Path|HelpButton)$/ and next;

		$CONFIG{$_} =~ s/\n/\n\+/g;
		print FILE "$_=$CONFIG{$_}\n";

		if ($CONFIG{'DefaultSave'} =~ /\b$_\b/ or
			$CONFIG{'AdditionalSave'} =~ /\b$_\b/) {

			print CACHE "$_=$CONFIG{$_}\n";
		}
	}

	flock(CACHE, 8);
	close CACHE;

	flock(FILE, 8);
	close FILE;
}


# Flush output to browser ---------------------------------------------------

sub flush {

	$| = 1;
	print "\n";

	$noErrorMessage = 1;
}


# Dump a hash to STDOUT -----------------------------------------------------

sub dumpHash {

	my (%hash) = @_;

	foreach (sort keys %hash) {
		print "$_=$hash{$_}\n";
	}
}


# Get and parse one entry from %CONFIG --------------------------------------

sub config {

	my ($cfg) = shift;

	$cfg = $CONFIG{$cfg};
	$cfg =~ s/\{(\w+?)\}/$CONFIG{$1}/g;

	return $cfg;
}


# Convert $FORM -variables to $CONFIG ones ----------------------------------

sub form2config {

	foreach (keys %FORM) {
		if (/^config(.+)/) {

			next if ($CONFIG{'ModifyNotAllowed'} =~ /\b$1\b/i);
			$CONFIG{$1} = $FORM{$_};
		}
	}
}


# Load one configuration file -----------------------------------------------

sub loadConfig {

	my ($file, $allow) = @_;

	if ($file !~ /[\\\/]/) {
		$file = $CONFIG{'ScriptPath'}.$file;
	}

	open(FILE, "<$file") or return 0;

	while (<FILE>) {
		/^#/ and next;
		tr/\x0A\x0D//d;

		if (/^\+(.*)/) {
			$CONFIG{$name} .= "\n$1";
		}
		else {
			($name, $value) = split(/\s*=\s*/, $_, 2);

			next if (!$allow and $CONFIG{'ModifyNotAllowed'} =~ /\b$name\b/i);
			$CONFIG{$name} = $value;
		}
	}

	close(FILE);
	return 1;
}


# Create a random filename --------------------------------------------------

sub getRandomFilename {

	my ($smtp) = shift;
	my ($filename, $path);

	$filename = int(rand 10000000000);

	$path = $smtp ? $CONFIG{'SMTPPath'} : $CONFIG{'TempPath'};
	while (-e $path.$filename) { $filename += $$; }

	return $filename;
}


# Load one file -------------------------------------------------------------

sub loadFile {

	my ($block, $file);

	open(LOADFILE, '<'.shift);

	$file = $block = '';
	while( read(LOADFILE, $block, 10000) ) { $file .= $block; }

	close LOADFILE;
	return $file;
}


# Delete all users files from the cache -------------------------------------

sub deleteCache {

	unlink "$CONFIG{'TempPath'}$FORM{'p'}" if $FORM{'p'};
	unlink "$CONFIG{'TempPath'}$FORM{'u'}" if $FORM{'u'};

	$msg = $CONFIG{'FirstMessage'} - 1;
	do { $msg++; } while ( unlink "$CONFIG{'TempPath'}$FORM{'u'}\.$msg" );

	$msg = 0;
	do { $msg++; } while ( unlink "$CONFIG{'TempPath'}$FORM{'u'}\.atch\.$msg" );

	$msg = 0;
	do { $msg++; } while ( unlink "$CONFIG{'TempPath'}$FORM{'u'}\.out\.$msg" );
}


# Get HTML-template ---------------------------------------------------------

sub getTemplate {

	$file = shift;
	$file =~ tr/A-Za-z0-9_.-//cd;

	$html = &loadFile($CONFIG{'HTMLPath'}.$file);

	$html =~ s/<SCRIPT(.|\n)*?<\/SCRIPT>//g if ($CONFIG{'JavaScript'} == 0);
	$html =~ s/\{(\w+?)\}/&config($1)/ge;

	return $html;
}


# Webify string -------------------------------------------------------------

sub webify {

	my ($string) = @_;

	$string =~ s/\"/\&quot;/g;
	$string =~ s/</\&lt;/g;
	$string =~ s/>/\&gt;/g;

	return $string;
}


# Mark one or more messages as deleted --------------------------------------

sub delete {

# Go through all available messages
	for ($msg = 1; $msg <= $CONFIG{'Messages'}; $msg++) {

		if ($FORM{"msg$msg"} and $CONFIG{'Deleted'} !~ /\b$msg\b/) {
			$CONFIG{'Deleted'} .= "$msg ";

# Get the headers of the message deleted
			open(FILE, "<$CONFIG{'TempPath'}$FORM{'u'}\.$msg");

			$headers = '';
			while (<FILE>) {
				$_ =~ /^<\/X-MAILREADER-HEADER>/ and last;
				$headers .= $_;
			}
			close(FILE);

# Was the message unread?
			$headers =~ /^Message-ID:\s*(.*)/mi or next;
			$msgid = $id = $1;
			$msgid =~ s/\W/\\$&/g;

			$unread = $CONFIG{'ReadMessages'} !~ /$msgid/ ? 1 : 0;

# Yes -> Disposition notification requested?
			if ($headers =~ /^Disposition-Notification-To:/mi and $unread) {
				&loadModule('misc.cgi');
				&sendDeliveryNote($headers, 'delete');

# Mark the message as read to prevent further disposition notes
				$CONFIG{'ReadMessages'} .= "$id\n";
			}
		}
	}

# Calc da number of messages left
	$CONFIG{'MessagesLeft'} = $CONFIG{'Messages'} - $CONFIG{'FirstMessage'} + 1;
	while ($CONFIG{'Deleted'} =~ /\b\d+\b/g) { $CONFIG{'MessagesLeft'}--; }

# Print next page
	if ($FORM{'msg'}) { &read; }
	else { &inbox; };
}


# Get next, prev & whatever message number ----------------------------------

sub getMsgNumbers {

	my ($msg) = shift;

	$CONFIG{'MessageNumber'} = $msg - $CONFIG{'FirstMessage'} + 1;
	$nextmsg = 0;
	$prevmsg = 0;

	for ($count = $CONFIG{'FirstMessage'}; $count < $msg; $count++) {
		$CONFIG{'MessageNumber'}-- if $CONFIG{'Deleted'} =~ /\b$count\b/;
	}

	for ($count = $msg + 1; $count <= $CONFIG{'Messages'}; $count++) {

		if ($CONFIG{'Deleted'} !~ /\b$count\b/) {
			$nextmsg = $count;
			last;
		}
	}

	for ($count = $msg - 1; $count >= $CONFIG{'FirstMessage'}; $count--) {

		if ($CONFIG{'Deleted'} !~ /\b$count\b/) {
			$prevmsg = $count;
			last;
		}
	}

	if ($CONFIG{'NewMessagesFirst'}) {

		$CONFIG{'MessageNumber'} = $CONFIG{'MessagesLeft'} - $CONFIG{'MessageNumber'} + 1;
		($nextmsg, $prevmsg) = ($prevmsg, $nextmsg);
	}
}


# Get the cookies from the browser ------------------------------------------

sub getCookies {

	@pairs = split(/; /, $ENV{'HTTP_COOKIE'});

	foreach $pair (@pairs) {
		($name, $value) = split(/=/, $pair);
		$COOKIE{$name} = $value;
	}
}


# Get the input from the browser --------------------------------------------

sub getQuery {

	if ($ENV{'CONTENT_TYPE'} =~ /multipart\/form-data/i) {

		&loadModule('misc.cgi');

		&getBinaryUpload;
		return;
	}

	if ($ENV{'REQUEST_METHOD'} eq 'POST') {

		read(STDIN, $buffer, $ENV{'CONTENT_LENGTH'});
		@pairs = split(/&/, $buffer);
	}
	else {
		@pairs = split(/&/, $ENV{'QUERY_STRING'});
	}

	foreach $pair (@pairs) {
		($name, $value) = split(/=/, $pair);
 
		$name = &urlDecode($name);
		$value = &urlDecode($value);

		if ($name eq 'alsoto' and $value ne '-') {
			if ($FORM{'to'}) { $FORM{'to'} .= ", $value"; }
			else { $FORM{'to'} = $value; }
		}
		else { $FORM{$name} = $value; };
	}
}


# URLencode -----------------------------------------------------------------

sub urlEncode {

	my ($string) = @_;

	$string =~ s/[^\w\.\-]/"%" . unpack("H2", $&)/ge;

	return $string;
}


# URLdecode -----------------------------------------------------------------

sub urlDecode {

	my ($string) = @_;

	$string =~ tr/+/ /;
	$string =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/eg;

	return $string;
}


# Functions for generating a row of buttons ---------------------------------

sub btnStart {
	if (!$CONFIG{'Buttons'}) { $buttonrow = ""; }
	else { $buttonrow = "<TABLE><TR>\n"; }
}

sub btnEnd { $buttonrow .= "</TR></TABLE>\n" if $CONFIG{'Buttons'}; }
sub btnBreak { $buttonrow .= "<TD>&nbsp; &nbsp; &nbsp;</TD>\n" if $CONFIG{'Buttons'}; }
sub btnFixBottom { $buttonrow =~ s/Btn_\w+/$&_Bottom/g; }

sub btnAdd {
	my ($href, $conf, $target) = @_;
	my ($link, $name, $img, $omo, $btns, $text);

	return unless $CONFIG{'Buttons'};

	if ($href =~ /^javascript:/) {
		return if ($CONFIG{'JavaScript'} == 0);
		$link = $href;
	}
	elsif ($href eq 'SUBMIT') {
		if ($CONFIG{'JavaScript'} > 0) {
			$link = "javascript:submitForm();";
		}
		else {
			$value = $CONFIG{"Text${conf}Button"};
			$value =~ s/<.*?>/ /g;

			$buttonrow .= "<TD VALIGN=top ALIGN=middle><INPUT TYPE=\"submit\" VALUE=\"$value\"></TD>\n";
			return;
		}
	}
	else {
		$link = "$CONFIG{'ScriptName'}?$href\&$CONFIG{'LinkQueryString'}";
	}

# Add image name & create onMouseOver code
	$img = &config("Image${conf}Button");
	$img =~ s/<BR>/ /gi;
	$omo = '';

	if ($CONFIG{'JavaScript'} > 0 and !$CONFIG{'NoOnMouseOver'} and
		$img =~ /^<IMG/i and $CONFIG{'Buttons'} ne 'text') {

		$name = "Btn_" . $conf;
		$img =~ s/>/ NAME=\"$name\">/;
		$omo = " onMouseOver=\"zap('$name', 1)\" onMouseOut=\"zap('$name', 0)\"";
	}

# Text-only buttons for lynx
	$btns = $ENV{'HTTP_USER_AGENT'} =~ /^Lynx/ ? 'text' : $CONFIG{'Buttons'};
	$text = $CONFIG{"Text${conf}Button"};

	if ($ENV{'HTTP_USER_AGENT'} =~ /^Lynx/) {
		$text =~ s/<BR>/ /g;
	}

	$buttonrow .= "<TD VALIGN=top ALIGN=middle>" .
		"<A HREF=\"$link\"" . ($target and " TARGET=\"$target\"") . "$omo>" .
		($btns =~ /(both|images)/ and "$img<BR>") .

		($btns eq 'both' and $CONFIG{SmallTextStart}) .
		($btns =~ /(both|text)/ and $text) .
		($btns eq 'both' and $CONFIG{SmallTextEnd}) .

		"</A></TD>\n";
}


# Inbox ---------------------------------------------------------------------

sub inbox {

	my ($noct) = shift;

# Automatic mailcheck?
	if ($CONFIG{'AutoMailCheck'} > 0 and (time - $CONFIG{'LastMailCheck'}) > ($CONFIG{'AutoMailCheck'} * 60)) {
		$CONFIG{'LastMailCheck'} = time;

		&loadModule('network.cgi');
		&checkmail;

		return;
	}

# Print headers
	&httpHeaders;

# Get template
	$CONFIG{'MessagesLeft'} ||= 0;

	&advertisement;
	&helpButton('Inbox');

	$_ = &getTemplate('inbox.html');
	($htmlhead, $htmlfoot) = split(/<!--begin-->/);

# Generate buttons
	&btnStart;

	&btnAdd('do=logout', 	'Logout');
	&btnAdd('SUBMIT', 		'DeleteSelected') if $CONFIG{'MessagesLeft'};
	&btnAdd("do=inbox&configDeleted=&configMessagesLeft=" . ($CONFIG{'Messages'} - $CONFIG{'FirstMessage'} + 1), 'Undelete') if $CONFIG{'Deleted'};
	&btnBreak;

#	&btnAdd('do=checkmail', 'Checkmail') if ($CONFIG{'AutoMailCheck'} == 0);
	&btnAdd('do=checkmail', 'Checkmail');

	&btnAdd('do=compose', 	'Compose');
	&btnAdd('do=showdraft',	'ShowDraft') if $CONFIG{'Draft'};
#	&btnAdd('do=search', 	'Search');
	&btnBreak;

	&btnAdd('do=options', 	'Options');
	&btnAdd('do=abook',		'Addresses');
	&btnBreak;

	if ($CONFIG{'JavaScript'} == 0) {
		&btnBreak;
		&btnAdd('do=help&size=full', 'Help');
	}

	&btnEnd;

# Print inbox
	print "Content-Type: text/html$crlf$crlf" unless $noct;

	print "$htmlhead\n",
	 	($CONFIG{'ButtonRowLocation'} =~ /(both|top)/ and "$buttonrow\n<P>\n");

	$fonton = "<FONT COLOR=\"$CONFIG{'HdrTextColor'}\">$CONFIG{'BigTextStart'}";
	$fontoff = "$CONFIG{'BigTextEnd'}</FONT>";
	$bc = " BGCOLOR=\"$CONFIG{'HdrBgrColor'}\"";

	if ($CONFIG{'MessagesLeft'} > 0) {
		print "<TABLE><TR><TD COLSPAN=3$bc>&nbsp;</TD>\n",
			"<TD$bc>$fonton$CONFIG{'TextFrom'}:$fontoff</TD>\n",
			"<TD$bc>$fonton$CONFIG{'TextSubject'}:$fontoff</TD>\n",

			"<TD$bc><NOBR>$fonton$CONFIG{'TextDate'}:$fontoff ",
			"<A HREF=\"$CONFIG{'ScriptName'}?do=inbox\&$CONFIG{'LinkQueryString'}\&configNewMessagesFirst=",
			1 - $CONFIG{'NewMessagesFirst'}, "\">", &config('IconReverseSort'), "</A></NOBR></TD>\n",

			"<TD COLSPAN=2$bc>$fonton$CONFIG{'TextSize'}:$fontoff</TD>\n</TR>\n";

		$readmessages = "";
		$repliedmessages = "";

		if ($CONFIG{'NewMessagesFirst'}) {
			for ($msg = $CONFIG{'Messages'}; $msg >= $CONFIG{'FirstMessage'}; $msg--) { &inboxRow; }
		}
		else {
			for ($msg = $CONFIG{'FirstMessage'}; $msg <= $CONFIG{'Messages'}; $msg++) { &inboxRow; }
		}

		$CONFIG{'ReadMessages'} = $readmessages;
		$CONFIG{'RepliedMessages'} = $repliedmessages;

		print "<TR><TD COLSPAN=8><IMG SRC=\"$CONFIG{'VirtualGfxPath'}blankdot.gif\" HSPACE=",
			$CONFIG{'TableWidth'} / 2 - 10, " WIDTH=1 HEIGHT=1 ALT=\"\"></TD></TR>\n</TABLE>\n";

		if ($CONFIG{'JavaScript'} > 0) {
			print "$CONFIG{'TButtonStart'}\n",
				"<A HREF=\"javascript:selectAll(1);\">",
				"$CONFIG{'TextSelectAll'}</A>\n",

				"<A HREF=\"javascript:selectAll(0);\">",
				"$CONFIG{'TextClearAll'}</A>\n",
				"$CONFIG{'TButtonEnd'}\n";
		}
	}
	else {
		print "$CONFIG{'BigTextStart'}$CONFIG{'TextMailboxEmpty'}$CONFIG{'BigTextEnd'}\n";
	}

	&btnFixBottom;

	print "<P>\n", &hiddenFields('delete'), 
		($CONFIG{'ButtonRowLocation'} =~ /(both|bottom)/ and "$buttonrow\n<P>\n"), 
		"$htmlfoot\n";
}


# Print one row of inbox ----------------------------------------------------

sub inboxRow {

# Clear the %HEADERS -hash
	undef %HEADER;

# Load the header
	open(FILE, "<$CONFIG{'TempPath'}$FORM{'u'}\.$msg");
	$thisheader = '';

	while (<FILE>) {
		/^<X-MAILREADER-HEADER>/ and next;
		/^<\/X-MAILREADER-HEADER>/ and last;

		$thisheader .= $_;
	}
	close FILE;

# Parse all headers of this message
	@pairs = split(/\n/, $thisheader);

	foreach $pair (@pairs) {
		($name, $value) = split(/:[ \t]/, $pair, 2);

		$name or next;

		$name =~ s/\s//g;
		$name =~ tr/A-Z/a-z/;
		$value =~ s/^\s*//;
		$value =~ s/\s*$//;
		$HEADER{$name} = $value;
	}

# New/read/replied message?
	$status = &config('IconNew');
	$msgid = $HEADER{'message-id'};
	$msgid =~ s/\W/\\$&/g;

	if ($CONFIG{'ReadMessages'} =~ /$msgid/) {
		$status = "";
		$readmessages .= "$HEADER{'message-id'}\n";
	}

	if ($CONFIG{'RepliedMessages'} =~ /$msgid/) {
		$status = &config('IconReplied');
		$repliedmessages .= "$HEADER{'message-id'}\n";
	}

# Discard deleted messages
	return if ($CONFIG{'Deleted'} =~ /\b$msg\b/);

# Get the priority
	$_ = $HEADER{'x-priority'};
	$priority = $_ ? getPriorityImage($_) : "";

# Fix header "From:"
	$from = (&parseEmailaddress($HEADER{'from'}))[0];

	if (length($from) > 20) {
		$from = substr($from, 0, 18) . '...';
	}

	if (($from =~ s/[A-Z]/$&/g) > 4 and length($from) > 14) {
		$from = substr($from, 0, 12) . '...';
	}

	$from = &webify($from);

# Fix header "Date:"
	$_ = $HEADER{'date'};
	s/,/ /g;
	@date = split(/\ +/);

	if ($_) {
		if (/(Mon|Tue|Wed|Thu|Fri|Sat|Sun)/) { shift @date; }

		$date[0] =~ s/^0//;

		$date = $CONFIG{'TextDateFormat'};
		$date =~ s/d+/$date[0]/;
		$date =~ s/m+/$months{$date[1]} + 1/e;
		$date =~ s/y+/$date[2]/;
	}

	if (!$_ or $date =~ /[A-Za-z]/) {
		$date = "($CONFIG{'TextUnknown'})";
	}

	$date = &webify($date);

# Get header "Subject:"
	$subject = $HEADER{'subject'} =~ /\w/ ? $HEADER{'subject'} : "($CONFIG{'TextUnknown'})";

	if (length($subject) > 26) {
		$subject = substr($subject, 0, 24) . '...';
	}

	if (($subject =~ s/[A-Z]/$&/g) > 6 and length($subject) > 20) {
		$subject = substr($subject, 0, 18) . '...';
	}

	$subject = &webify($subject);

# Get the size of the message
	$size = int($CONFIG{"LengthOf$msg"} / 1024);
	$size++;

# Attachments in message?
	$_ = $HEADER{'content-type'};

	$attachment = &config('IconAttachment');
	/text\//i and $attachment = "";
	/multipart\/alternative/i and $attachment = "";

# Print HTML
	print "<TR><TD>$status</TD>\n",
		"<TD><INPUT TYPE=\"checkbox\" NAME=\"msg$msg\"></TD>\n<TD ALIGN=middle>$priority</TD>\n",
		"<TD><A HREF=\"$CONFIG{'ScriptName'}?do=read\&msg=$msg\&$CONFIG{'LinkQueryString'}\"><STRONG>$from</STRONG></A></TD>\n",
		"<TD>$subject</TD>\n<TD ALIGN=right>$date </TD>\n",
		"<TD ALIGN=right>$size <SMALL>$CONFIG{'TextSizeUnit'}</SMALL></TD>\n",
		"<TD>$attachment</TD></TR>\n";
}


# Read a message ------------------------------------------------------------

sub read {

# Print headers
	&httpHeaders;

# Get da message
	$msg = $FORM{'msg'};
	&getMessage($msg);
	&getMsgNumbers($msg);

	&advertisement;
	&helpButton('Read');

	$_ = &getTemplate('read.html');
	($htmlhead, $htmlfoot) = split(/<!--begin-->/);

	$message =~ /^(.+?)<\/X-MAILREADER-HEADER>/s;
	$headers = $1;

# Mark the message as read
	if ($headers =~ /^Message-ID:\s*(.*)/mi) {
		$msgid = $id = $1;
		$msgid =~ s/\W/\\$&/g;

		if ($CONFIG{'ReadMessages'} !~ /$msgid/) {
			$CONFIG{'ReadMessages'} .= "$id\n";
			$unread = 1;
		}
		else { $unread = 0; }
	}

# Disposition notification requested?
	if ($headers =~ /^Disposition-Notification-To:/mi and $unread) {
		&loadModule('misc.cgi');
		&sendDeliveryNote($headers, 'read');
	}

# Generate buttons
	&btnStart;

	&btnAdd('do=logout', 			'Logout', '_top');
	&btnAdd("do=delete&msg$msg=on&msg=$nextmsg", 'Delete');
	&btnBreak;

	&btnAdd("do=compose&msg=$msg",	'Compose');
	&btnAdd("do=showdraft&msg=$msg", 'ShowDraft') if $CONFIG{'Draft'};
	&btnAdd("do=reply&msg=$msg",	'Reply');
	&btnAdd("do=repall&msg=$msg",	'ReplyAll');
	&btnAdd("do=forward&msg=$msg",	'Forward');
	&btnAdd("do=redirect&msg=$msg",	'Redirect');
	&btnBreak;

	&btnAdd('do=inbox', 			'Inbox');
	&btnAdd("do=read&msg=$prevmsg",	'Previous') if ($prevmsg);
	&btnAdd("do=read&msg=$nextmsg",	'Next') if ($nextmsg);

	&btnEnd;

# Fix the message
	$message =~ s/\{(\w+?)\}/&config($1)/ge;
	$message =~ s/<X-MAILREADER-HEADER>(.|\n)*?<\/X-MAILREADER-HEADER>/&prepareHeader($&, $msg)/ge;

# Print the page
	print "Content-Type: text/html$crlf$crlf$htmlhead\n",
	 	($CONFIG{'ButtonRowLocation'} =~ /(both|top)/ and "$buttonrow\n<P>\n"), "$message\n";

	&btnFixBottom;

	print '', ($CONFIG{'ButtonRowLocation'} =~ /(both|bottom)/ and "$buttonrow\n<P>\n"),
		"$htmlfoot\n";

# Handle scheduled events
	&fakeCron;
}


# Handle scheduled tasks ----------------------------------------------------

sub fakeCron {

# Parse logfile
	$noErrorMessage = 1;
	$file = "$CONFIG{'ScriptPath'}log\.parsed";

	if (!-e $file) { open(FILE, ">$file"); }

	if (-M $file > 2) {

		if (((localtime)[6] == 1) or (-M $file > 7)) {
			$_ = time;
			utime($_, $_, $file);

			&loadModule('logparser.cgi');
			&makeLog;

			return;
		}
	}

# Send unsent outgoing messages

	if (int(rand 5) == 2) {
		&loadModule('network.cgi');
		$count = 0;

		opendir(DIR, $CONFIG{'SMTPPath'});

		while ($_ = readdir DIR) {
			$_ =~ /^\./ and next;
			$_ = $CONFIG{'SMTPPath'}.$_;

			if (-M $_ > 5) {					# Kill unsent messages older than 5 days
				unlink $_;
				next;
			}

			if (-M $_ > 0.02) {					# Resend emails older than half an hour
				&mailFile($_);
				last if ($count++ > 5);
			}
		}

		close DIR;
		return;
	}

# Clean up the temporary directory
	$maxtime = $CONFIG{'MaxLoginTime'} / 24;
	$count = 0;

	opendir(DIR, $CONFIG{'TempPath'});

	while ($_ = readdir DIR) {
		$_ =~ /^\./ and next;
		$_ = $CONFIG{'TempPath'}.$_;

		if (-M $_ > $maxtime) {
			unlink $_;
			last if ($count++ > 100);			# Stop after ~100 deleted files
		}
	}

	close DIR;
}


# Parse an emailaddress with real name --------------------------------------

sub parseEmailaddress {

	my ($address) = @_;
	my ($email, $name);

	$email = "($CONFIG{'TextUnknown'})";
	$name = $address;

	$address =~ /[\w\.\-]+\@[\w\.\-]+\.\w+/ and $email = $&;

	if ($address =~ /</) {			# Someone <joe@company.com>
		$name =~ s/<.*?>//g;
	}
	elsif ($address =~ /\(/) {		# joe@company.com (Someone)
		$name =~ /\((.*?)\)/ and $name = $1;
	}

	if ($name !~ /\w/) {			# Did we truncate too much?
		$name = $address;
		$name =~ s/[<>]//g;
	}

	$name =~ s/^\s*\"*//;
	$name =~ s/\"*\s*$//;

	$name !~ /\w/ and $name = $email;
	$email !~ /[a-z]/ and $email =~ tr/A-Z/a-z/;

	return ($name, $email);
}


# Get a link for help -------------------------------------------------------

sub helpButton {

	my ($position) = shift;

	$CONFIG{'HelpButton'} = ($CONFIG{'JavaScript'} > 0) ?
		"<A HREF=\"javascript:openPopUp(\'$CONFIG{'ScriptName'}?do=help\&size=small\&" .
		"$CONFIG{'LinkQueryString'}\#$position\');\">$CONFIG{'IconHelpButton'}</A>\n" : '';
}


# Return correct image for the priority -------------------------------------

sub getPriorityImage {

	my ($priority) = @_;

	$priority =~ /1/ and return &config('IconHighestPriority');
	$priority =~ /2/ and return &config('IconHighPriority');

	$priority =~ /4/ and return &config('IconLowPriority');
	$priority =~ /5/ and return &config('IconLowestPriority');
}


# Decode BASE64 (MIME) -coding ----------------------------------------------

sub base64Decode {

	($_) = @_;
	my ($bin, $len);

	tr#A-Za-z0-9+/##cd;					# Remove non-base64 characters

	if (length > 60) {					# Handle long BASE64-strings in pieces
		s/.{0,60}/&base64Decode($&);/ge;
		return $_;
	}

	tr#A-Za-z0-9+/# -_#;				# Convert to uuencode

	$len = pack("c", 32 + 0.75 * length);
	$bin = unpack("u", $len.$_);		# Decode

	return $bin;
}


# Wrap a long line ----------------------------------------------------------

sub wordWrap {

	my ($string, $len) = @_;
	my ($new, $row);

	foreach $char ($string =~ /./g) {
		$row .= $char;

		if ($char =~ /[\x0d\x0a]/) {
			$new .= $row;
			$row = '';
		}
		elsif (length($row) >= $len and $char =~ /[ .,:;-]/) {
			$new .= "$row\x00";
			$row = '';
		}
	}

	$new .= $row;
	$new =~ s/\x00\s*/\n/g;

	return $new;
}


# Return a block of common hidden fields ------------------------------------

sub hiddenFields {

	my ($do) = shift;

	return "<INPUT NAME=\"do\" TYPE=\"hidden\" VALUE=\"$do\">\n" .
		"<INPUT NAME=\"u\" TYPE=\"hidden\" VALUE=\"$FORM{'u'}\">\n" .
		"<INPUT NAME=\"p\" TYPE=\"hidden\" VALUE=\"$FORM{'p'}\">\n" .
		($FORM{'msg'} > 0 and "<INPUT NAME=\"msg\" TYPE=\"hidden\" VALUE=\"$FORM{'msg'}\">\n");
}


# Translate a date string ---------------------------------------------------

sub translateDate {

	$date = shift;

	foreach (keys %weekdays) {
		$date =~ s#\b$_\b#(split(/\s*,\s*/, $CONFIG{'TextWeekdays'}))[$weekdays{$_}]#ie and last;
	}

	foreach (keys %months) {
		$date =~ s#\b$_\b#(split(/\s*,\s*/, $CONFIG{'TextMonths'}))[$months{$_}]#ie and last;
	}

	return $date;
}


# Prepare header for printing -----------------------------------------------

sub prepareHeader {

	my ($header, $msg, $printable) = @_;

	$savefrom = '';
	$saverepto = '';

	if (!$printable) {
		if ($header =~ /^From:\ (.*)/im) {
			($name, $email) = &parseEmailaddress($1);

			$savefrom = "&nbsp; &nbsp;$CONFIG{'TButtonStart'}" .
				"<A HREF=\"$CONFIG{'ScriptName'}?do=saveaddress\&" .
				"msg=$msg\&name=" . &urlEncode($name) . 
				"\&email=" . &urlEncode($email) . "\&$CONFIG{'LinkQueryString'}\">" .
				"$CONFIG{'TextSaveAddress'}</A>$CONFIG{'TButtonEnd'}\n";

			if ($header =~ /^Reply-To:\ (.*)/im) {
				$email = (&parseEmailaddress($1))[1];

				$saverepto = "&nbsp; &nbsp;$CONFIG{'TButtonStart'}" .
					"<A HREF=\"$CONFIG{'ScriptName'}?do=saveaddress\&" .
					"msg=$msg\&name=" . &urlEncode($name) . 
					"\&email=" . &urlEncode($email) . "\&$CONFIG{'LinkQueryString'}\">" .
					"$CONFIG{'TextSaveAddress'}</A>$CONFIG{'TButtonEnd'}\n";

				$savefrom = "";
			}
		}

		$email =~ s/\W/\\$&/g;
		if ($CONFIG{'AddressBook'} =~ /\b$email\b/) {
			$savefrom = "";
			$saverepto = "";
		}
	}

	$header =~ s#^(Date:\s*)(.*)#$1 . &translateDate($2)#ime;

	$header = &webify($header);
	$new = '';
	$fonton = $printable ? '<STRONG>' : "<FONT COLOR=\"$CONFIG{'AltTextColor'}\"><STRONG>";
	$fontoff = $printable ? '</STRONG>' : '</STRONG></FONT>';

	foreach ($header =~ /.*/g) {
		/^\s*$/m and next;
		/^\&lt;.*X-MAILREADER-HEADER&gt;/m and next;

		tr/\x0d\x0a//d;

		/^(Date|Subject|To|Newsgroups):(.+)/i and do { $new .= $fonton . $CONFIG{"Text\u\L$1"} . ": $2${fontoff}<BR>\n"; next; };
		/^CC:(.+)/i and do { $new .= "${fonton}$CONFIG{'TextCC'}: $1${fontoff}<BR>\n"; next; };

		/^From:(.+)/i and do { $new .= "${fonton}$CONFIG{'TextFrom'}: $1${fontoff} $savefrom<BR>\n"; next; };
		/^Reply-To:(.+)/i and do { $new .= "${fonton}$CONFIG{'TextReplyTo'}: $1${fontoff} $saverepto<BR>\n"; next; };
		/^X-Priority:(.+)/i and do { $new .= "${fonton}$CONFIG{'TextPriority'}: $1${fontoff} " . &getPriorityImage($&) . "<BR>\n"; next; };

		next unless $CONFIG{'ShowAllHeaders'};

		$new .= "$CONFIG{'SmallTextStart'}$_$CONFIG{'SmallTextEnd'}<BR>\n";
	}

	if (!$printable) {
		if ($CONFIG{'ShowAllHeaders'}) {
			$new .= "$CONFIG{'TButtonStart'}" .
				"<A HREF=\"$CONFIG{'ScriptName'}?do=read\&msg=$msg\&configShowAllHeaders=\&" .
				"$CONFIG{'LinkQueryString'}\">$CONFIG{'TextShowBasicHeaders'}</A>" .
				"$CONFIG{'TButtonEnd'}\n";
		}
		else {
			$new .= "$CONFIG{'TButtonStart'}" .
				"<A HREF=\"$CONFIG{'ScriptName'}?do=read\&msg=$msg\&configShowAllHeaders=1\&" .
				"$CONFIG{'LinkQueryString'}\">$CONFIG{'TextShowAllHeaders'}</A>" .
				"$CONFIG{'TButtonEnd'}\n";
		}

		$new .= "$CONFIG{'TButtonStart'}" .
			"<A HREF=\"$CONFIG{'ScriptName'}?do=printable\&msg=$msg\&$CONFIG{'LinkQueryString'}\">" .
			"$CONFIG{'TextPrintableVersion'}</A>$CONFIG{'TButtonEnd'}\n";

		if ($CONFIG{'ViewSource'}) {
			$new .= "$CONFIG{'TButtonStart'}" .
				"<A HREF=\"$CONFIG{'ScriptName'}?do=source\&msg=$msg\&$CONFIG{'LinkQueryString'}\" " .
				"TARGET=\"_blank\">$CONFIG{'TextViewSource'}</A>" .
				"$CONFIG{'TButtonEnd'}\n";
		}
	}

	return "<TABLE WIDTH=$CONFIG{'TableWidth'} CELLPADDING=0 CELLSPACING=0 BORDER=0>" .
		"<TD>\n$new</TD></TABLE><P>\n";
}


# Get one message -----------------------------------------------------------

sub getMessage {

	my ($msg) = shift;

	$message = &loadFile("$CONFIG{'TempPath'}$FORM{'u'}\.$msg")
		or &error('ErrorLoggedout', 1090);

	if ($CONFIG{'HeadersOnly'} =~ /\b$msg\b/) {

		&loadModule('network.cgi');

		&ConnectPOP3;
		&getMessageFromServer($msg);
		&DisconnectPOP3;

		$message = &loadFile("$CONFIG{'TempPath'}$FORM{'u'}\.$msg")
			or &error('ErrorLoggedout', 1100);

		$CONFIG{'HeadersOnly'} =~ s/$msg\b//;
	}
}


# Get an advertisement ------------------------------------------------------

sub advertisement {

# No ads?
	return if $CONFIG{'NoAdvertisements'};

# Get an array of available ads
	opendir(FILE, $CONFIG{'AdvertisementPath'});
	@DIR = grep !/^\./, readdir FILE;
	close FILE;

	$adcount = scalar @DIR;
	$adnum = int(rand $adcount);

	$adnum++ if ($DIR[$adnum] eq $CONFIG{'LastAd'});
	$adnum = 0 if ($adnum >= $adcount);

	$day = (localtime)[3];
	$month = (localtime)[4] + 1;
	$year = (localtime)[5] + 1900;

	$now = $year . ($month < 10 and '0') . $month . ($day < 10 and '0') . $day;

# Loop through all available ads
	for ($i = 0; $i < $adcount; $i++) {

		$thisad = $DIR[$adnum];
		&loadAdConfig("$CONFIG{'AdvertisementPath'}$thisad/ad.cfg") or next;

		$adnum = 0 if (++$adnum >= $adcount);

# OK to show?
		$AD{'Impressions'}++;
		next if ($AD{'ImpressionLimit'} and ($AD{'Impressions'} > $AD{'ImpressionLimit'}));

		next if ($AD{'ClickLimit'} and ($AD{'Clicks'} >= $AD{'ClickLimit'}));
		next if ($AD{'Probability'} and (rand 100 > $AD{'Probability'}));

		next if ($AD{'StartDate'} and ($now < $AD{'StartDate'}));
		next if ($AD{'EndDate'} and ($now > $AD{'EndDate'}));

# Ad OK, show & exit
		$CONFIG{'AdvertisementHREF'} = "$CONFIG{'ScriptName'}?do=click\&ad=$thisad\&$CONFIG{'LinkQueryString'}";

		if (-e "$CONFIG{'AdvertisementPath'}$thisad/ad.html") {
			$CONFIG{'Advertisement'} = &loadFile("$CONFIG{'AdvertisementPath'}$thisad/ad.html");
			$CONFIG{'Advertisement'} =~ s/\{(\w+?)\}/&config($1)/ge;
		}
		else {
			&loadModule("$CONFIG{'AdvertisementPath'}$thisad/ad.cgi") or next;
		}

		$CONFIG{'Advertisement'} .= "\n<BR CLEAR=all><BR>\n";

		&saveAdConfig("$CONFIG{'AdvertisementPath'}$thisad/ad.cfg");
#		&saveString("$CONFIG{'LogPath'}main.log", "ad,1,$thisad," . time . ",$AD{'Impressions'},$AD{'Clicks'}\n");

		$CONFIG{'AdImpressions'}++;
		$CONFIG{'LastAd'} = $thisad;

		last;
	}
}


# Load advertisement's configuration ----------------------------------------

sub loadAdConfig {

	open(FILE, '<'.shift) or return 0;
	undef %AD;

	while (<FILE>) {
		/^#/ and next;
		tr/\x0A\x0D//d;

		($name, $value) = split(/\s*=\s*/, $_, 2);
		$AD{$name} = $value;
	}
	close FILE;

	return 1;
}


# Save advertisement's configuration ----------------------------------------

sub saveAdConfig {

	my ($file) = shift;

	open(FILE, ">$file") or return;
	flock(FILE, 2);
	seek FILE, 0, 2;

	foreach (keys %AD) {
		print FILE "$_=$AD{$_}\n";
	}

	flock(FILE, 8);
	close FILE;
}


# Add a string to a file using file locking ---------------------------------

sub saveString {

	my ($file) = shift;

	open(FILE, ">>$file");
	flock(FILE, 2);
	seek FILE, 0, 2;

	print FILE shift;

	flock(FILE, 8);
	close(FILE);
}


# Handle errors -------------------------------------------------------------

sub error {

	my ($name, $code, $details) = @_;

# Error messages disabled?
	exit if ($noErrorMessage);

# Reset configuration variables
	$language = "$CONFIG{'Language'}.cfg";

	&loadConfig('server.cfg', 1);
	&loadConfig('main.cfg', 1);

	if ($language) { &loadConfig($language); }
	else { &loadConfig("$CONFIG{'Language'}.cfg"); }

# Clean up stuff
	eval $timeoutOff;
	close(S);
#	&deleteCache;

# Print the errormessage
	$CONFIG{'ErrorCode'} = $code;
	$CONFIG{'ErrorDetails'} = $details;
	$CONFIG{'ErrorInfo'} = &config($name);

	if (!$httpHeadersPrinted) {
		$CONFIG{'FullHTTPHeaders'} and $_ = "$ENV{'SERVER_PROTOCOL'} 200 OK\n";
		$_ .= "NoCacheHeaders=Expires: 1 Jan 1997\nPragma: no-cache\nCache-Control: no-cache\n";

		$_ =~ s/\n/$crlf/g;
		print;
	}

	print "Content-Type: text/html$crlf$crlf", &getTemplate('error.html');

	&saveString("$CONFIG{'LogPath'}main.log", "err,1,$name,$code\n");
	exit;
}

