/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.ibm.websphere.web;

import xdoclet.XDocletException;

import xdoclet.XmlSubTask;
import xdoclet.util.Translator;

/**
 * Generates WebSphere specific deployment descriptors for Web modules. The following files are generated:
 * ibm-web-bnd.xmi and ibm-web-ext.xmi. <p>
 *
 * NOTE: Since the WebSphere specific deployment descriptors depend on that id's are set in the web.xml file you must
 * genererate it with useIds set to true, e.g. <code>&lt;deploymentdescriptor useIds="true"/&gt;</code>. </p> <p>
 *
 * This Ant task defines the following attributes (the rest of the attributes in the list below comes from its
 * baseclass):
 * <ul>
 *   <li> virtualHostName</li>
 *   <li> reloadInterval</li>
 *   <li> reloadingEnabled</li>
 *   <li> defaultErrorPage</li>
 *   <li> additionalClassPath</li>
 *   <li> fileServingEnabled</li>
 *   <li> directoryBrowsingEnabled</li>
 *   <li> serveServletsByClassnameEnabled</li>
 *   <li> preCompileJSPs</li>
 *   <li> autoRequestEncoding</li>
 *   <li> autoResponseEncoding</li>
 *   <li> autoLoadFilters</li>
 * </ul>
 * </p> NOTE: All attributes except "virtualHostName" are IBM specific WebSphere extensions to the web.xml file and are
 * the same as the attributes that can be found in the IBM WSAD wizard (v5.1.2) for Web Deployment Descriptors (see tab
 * "Extensions" and section "General")
 *
 * @author        <a href="mailto:ed@whatsa.co.uk">Ed Ward</a>
 * @author        <a href="mailto:ml at callista.se">Magnus Larsson</a>
 * @created       22 August 2002
 * @version       $Revision: 1.8 $
 * @ant.element   display-name="WebSphere" name="webspherewebxml" parent="xdoclet.modules.web.WebDocletTask"
 */
public class WebSphereWebXmlSubTask extends XmlSubTask
{
    /**
     * A config paramater value: true
     */
    private final static String TRUE = "true";

    /**
     * A config parameter value: false
     */
    private final static String FALSE = "false";

    /**
     * The template file for the bindings. The template generates a stylesheet, which when styled with a web.xml file
     * (modified with id attributes) will produce the ibm-web-bnd.xmi bindings deployment descriptor
     */
    private final static String
        BINDINGS_TEMPLATE_FILE = "resources/ibm-web-bnd_xmi.xdt";

    /**
     * The name of the IBM bindings deployment descriptor - ibm-web-bnd.xmi
     */
    private final static String
        GENERATED_BINDINGS_FILE_NAME = "ibm-web-bnd.xmi";

    /**
     * The template file for the extensions. The template generates a stylesheet, which when styled with a web.xml file
     * (modified with id attributes) will produce the ibm-web-ext.xmi extensions deployment descriptor
     */
    private final static String
        EXTENSIONS_TEMPLATE_FILE = "resources/ibm-web-ext_xmi.xdt";

    /**
     * The name of the IBM bindings deployment descriptor - ibm-web-bnd.xmi
     */
    private final static String
        GENERATED_EXTENSIONS_FILE_NAME = "ibm-web-ext.xmi";

    /**
     * Flag indicating whether we've styled the web.xml file
     */
    private boolean hasTransformedWebXml = false;

    /**
     * @see   #setVirtualHostName(String)
     */
    private String  virtualHostName = "default_host";

    /**
     * @see   #setReloadInterval(String)
     */
    private String  reloadInterval = "3";

    /**
     * @see   #setReloadingEnabled(String)
     */
    private String  reloadingEnabled = "true";

    /**
     * @see   #setDefaultErrorPage(String)
     */
    private String  defaultErrorPage = "";

    /**
     * @see   #setAdditionalClassPath(String)
     */
    private String  additionalClassPath = "";

    /**
     * @see   #setFileServingEnabled(String)
     */
    private String  fileServingEnabled = "true";

    /**
     * @see   #setDirectoryBrowsingEnabled(String)
     */
    private String  directoryBrowsingEnabled = "true";

    /**
     * @see   #setServeServletsByClassnameEnabled(String)
     */
    private String  serveServletsByClassnameEnabled = "true";

    /**
     * @see   #setPreCompileJSPs(String)
     */
    private String  preCompileJSPs = "true";

    /**
     * @see   #setAutoRequestEncoding(String)
     */
    private String  autoRequestEncoding = "false";

    /**
     * @see   #setAutoResponseEncoding(String)
     */
    private String  autoResponseEncoding = "false";

    /**
     * @see   #setAutoLoadFilters(String)
     */
    private String  autoLoadFilters = "false";

    public WebSphereWebXmlSubTask()
    {
        setUseIds(true);
    }

    /**
     * @return   the virtual host name
     * @see      #setVirtualHostName(String)
     */
    public String getVirtualHostName()
    {
        return virtualHostName;
    }

    /**
     * @return   the reload interval
     * @see      #setReloadInterval(String)
     */
    public String getReloadInterval()
    {
        return reloadInterval;
    }

    /**
     * @return
     * @see      #setReloadingEnabled(String)
     */
    public String getReloadingEnabled()
    {
        return reloadingEnabled;
    }

    /**
     * @return
     * @see      #setDefaultErrorPage(String)
     */
    public String getDefaultErrorPage()
    {
        return defaultErrorPage;
    }

    /**
     * @return
     * @see      #setAdditionalClassPath(String)
     */
    public String getAdditionalClassPath()
    {
        return additionalClassPath;
    }

    /**
     * @return
     * @see      #setFileServingEnabled(String)
     */
    public String getFileServingEnabled()
    {
        return fileServingEnabled;
    }

    /**
     * @return
     * @see      #setDirectoryBrowsingEnabled(String)
     */
    public String getDirectoryBrowsingEnabled()
    {
        return directoryBrowsingEnabled;
    }

    /**
     * @return
     * @see      #setServeServletsByClassnameEnabled(String)
     */
    public String getServeServletsByClassnameEnabled()
    {
        return serveServletsByClassnameEnabled;
    }

    /**
     * @return
     * @see      #setPreCompileJSPs(String)
     */
    public String getPreCompileJSPs()
    {
        return preCompileJSPs;
    }

    /**
     * @return
     * @see      #setAutoRequestEncoding(String)
     */
    public String getAutoRequestEncoding()
    {
        return autoRequestEncoding;
    }

    /**
     * @return
     * @see      #setAutoResponseEncoding(String)
     */
    public String getAutoResponseEncoding()
    {
        return autoResponseEncoding;
    }

    /**
     * @return
     * @see      #setAutoLoadFilters(String)
     */
    public String getAutoLoadFilters()
    {
        return autoLoadFilters;
    }

    /**
     * The virtual host name. "A virtual host is a configuration enabling a single host machine to resemble multiple
     * host machines. This property allows you to bind the application to a virtual host in order to enable execution on
     * that virtual host."
     *
     * @param name         the virtual host name
     * @ant.not-required   No. Default is "default_host"
     */
    public void setVirtualHostName(String name)
    {
        virtualHostName = name;
    }

    /**
     * A Reload Interval. Every 'reload interval' seconds, the web application's files are checked and reloaded if they
     * have been modified. Requires that reloadingEnabled is set to true.
     *
     * @param reloadInterval
     * @see                   #setReloadingEnabled(String)
     * @ant.not-required      No. Default is "3"
     */
    public void setReloadInterval(String reloadInterval)
    {
        this.reloadInterval = validateIntegerValue(reloadInterval, "reloadInterval");
    }

    /**
     * Specifies whether reloading is enabled.
     *
     * @param reloadingEnabled
     * @ant.not-required        No. Default is "true"
     */
    public void setReloadingEnabled(String reloadingEnabled)
    {
        this.reloadingEnabled = reloadingEnabled;
    }

    /**
     * Specifies a file name for the default error page. If no other error page is specified in the application, this
     * error page is used.
     *
     * @param defaultErrorPage
     * @ant.not-required        No. Default is ""
     */
    public void setDefaultErrorPage(String defaultErrorPage)
    {
        this.defaultErrorPage = defaultErrorPage;
    }

    /**
     * Specifies an additional class path that will be used to reference resources outside of those specified in the
     * archive.
     *
     * @param additionalClassPath
     * @ant.not-required           No. Default is ""
     */
    public void setAdditionalClassPath(String additionalClassPath)
    {
        this.additionalClassPath = additionalClassPath;
    }

    /**
     * Specifies whether file serving is enabled. File serving allows the application to serve static file types, such
     * as HTML and GIF. File serving can be disabled if, for example, the application contains only dynamic components.
     *
     * @param fileServingEnabled
     * @ant.not-required          No. Default is "true"
     */
    public void setFileServingEnabled(String fileServingEnabled)
    {
        this.fileServingEnabled = validateBooleanValue(fileServingEnabled, "fileServingEnabled");
    }

    /**
     * Specifies whether directory browsing is enabled. Directory browsing allows the application to browse disk
     * directories. Directory browsing can be disabled if, for example, you want to protect data.
     *
     * @param directoryBrowsingEnabled
     * @ant.not-required                No. Default is "true"
     */
    public void setDirectoryBrowsingEnabled(String directoryBrowsingEnabled)
    {
        this.directoryBrowsingEnabled = validateBooleanValue(directoryBrowsingEnabled, "directoryBrowsingEnabled");
    }

    /**
     * Specifies whether a servlet can be served by requesting its class name. Usually, servlets are served only through
     * a URI reference. The class name is the actual name of the servlet on disk. For example, a file named
     * SnoopServlet.java compiles into SnoopServlet.class. (This is the class name.) SnoopServlet.class is normally
     * invoked by specifying snoop in the URI. However, if Serve Servlets by Classname is enabled, the servlet is
     * invoked by specifying SnoopServlet.
     *
     * @param serveServletsByClassnameEnabled
     * @ant.not-required                       No. Default is "true"
     */
    public void setServeServletsByClassnameEnabled(String serveServletsByClassnameEnabled)
    {
        this.serveServletsByClassnameEnabled = validateBooleanValue(serveServletsByClassnameEnabled, "serveServletsByClassnameEnabled");
    }

    /**
     * Specifies wheter JSP pages will be precompiled at deploy time or not.
     *
     * @param preCompileJSPs
     * @ant.not-required      No. Default is "true"
     */
    public void setPreCompileJSPs(String preCompileJSPs)
    {
        this.preCompileJSPs = validateBooleanValue(preCompileJSPs, "preCompileJSPs");
    }

    /**
     * See IBM WebSphere documentation regarding this attribute.
     *
     * @param autoRequestEncoding
     * @ant.not-required           No. Default is "false"
     */
    public void setAutoRequestEncoding(String autoRequestEncoding)
    {
        this.autoRequestEncoding = validateBooleanValue(autoRequestEncoding, "autoRequestEncoding");
    }

    /**
     * See IBM WebSphere documentation regarding this attribute.
     *
     * @param autoResponseEncoding
     * @ant.not-required            No. Default is "false"
     */
    public void setAutoResponseEncoding(String autoResponseEncoding)
    {
        this.autoResponseEncoding = validateBooleanValue(autoResponseEncoding, "autoResponseEncoding");
    }

    /**
     * See IBM WebSphere documentation regarding this attribute.
     *
     * @param autoLoadFilters
     * @ant.not-required       No. Default is "false"
     */
    public void setAutoLoadFilters(String autoLoadFilters)
    {
        this.autoLoadFilters = validateBooleanValue(autoLoadFilters, "autoLoadFilters");
    }

    /**
     * Called to validate configuration parameters.
     *
     * @exception XDocletException  Description of Exception
     */
    public void validateOptions() throws XDocletException
    {
        // Skip the default validation to avoid the following error message:
        //     "webspherewebxml: 'templateFile' parameter missing. Specify both 'destinationFile' and 'templateFile' configuration parameters please."
        // The WebSphereWebXmlSubTask does not require a template url or a destination file anyhow.
        // super.validateOptions();
    }

    /**
     * Called by xdoclet to execute the subtask.
     *
     * @exception XDocletException
     */
    public void execute() throws XDocletException
    {
        setTemplateURL(getClass().getResource(BINDINGS_TEMPLATE_FILE));
        setDestinationFile(GENERATED_BINDINGS_FILE_NAME);
        startProcess();

        setTemplateURL(getClass().getResource(EXTENSIONS_TEMPLATE_FILE));
        setDestinationFile(GENERATED_EXTENSIONS_FILE_NAME);
        startProcess();
    }

    /**
     * TODO: Describe what the method does
     *
     * @exception XDocletException
     */
    protected void engineStarted() throws XDocletException
    {
    }

    /**
     * Helper method that validates if a value can be interpreted as an integer
     *
     * @param integerValue
     * @param name
     * @return              the value if it is ok or else throws an runtime exception
     */
    private String validateIntegerValue(String integerValue, String name)
    {
        try {
            Integer.parseInt(integerValue);
        }
        catch (NumberFormatException e) {
            throwInvalidConfigValueException(integerValue, name);
        }

        return integerValue;
    }

    /**
     * Helper method that validates if a value can be interpreted as a boolean
     *
     * @param booleanValue
     * @param name
     * @return              the value if it is ok or else throws an runtime exception
     */
    private String validateBooleanValue(String booleanValue, String name)
    {
        if (booleanValue == null) {
            booleanValue = "";
        }

        if (!(TRUE.equals(booleanValue) ||
            FALSE.equals(booleanValue) ||
            "".equals(booleanValue))) {
            throwInvalidConfigValueException(booleanValue, name);
        }

        return booleanValue;
    }

    /**
     * Helper method for throwing an exception indicating that an configuration attribute was set incorect.
     *
     * @param value
     * @param name
     */
    private void throwInvalidConfigValueException(String value, String name)
    {
        throw new RuntimeException(
            Translator.getString(
            XDocletModulesIbmWebsphereWebMessages.class,
            XDocletModulesIbmWebsphereWebMessages.INVALID_CONFIG_VALUE,
            new String[]{value, name}));
    }
}
