/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef SA_NETWORK
#warning You should not include this file directly from your program.
#endif

/**
 * \defgroup network Network interface
 * @{
 */
#ifndef SA_COMMON_NETWORK_H_
#define SA_COMMON_NETWORK_H_

/** \struct sa_network_interface network.h
 * This structure gathers the details about the activity of one network interface.
 */
SA_EXPORT struct sa_network_interface {
#ifdef SA_NETWORK_INTERFACE_NAME
	char name[SA_NETWORK_INTERFACE_NAME]; //!< Interface's name. It's used as the unique identifier of the interface.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_BYTES
	uint64_t received_bytes; //!< Total number of received bytes.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_PACKETS
	uint64_t received_packets; //!< Total number of received packets.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_ERRORS
	uint64_t received_errors; //!< Amount of received errors.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_DROP
	uint64_t received_drop; //!< Total number of received packets that had been dropped.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_FIFO
	uint64_t received_fifo; //!< The number of fifo buffer errors received.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_COMPRESSED
	uint64_t received_compressed; //!< The number of compressed packets received by the device driver.
#endif
#ifdef SA_NETWORK_INTERFACE_RECEIVED_MULTICAST
	uint64_t received_multicast; //!< Number of packets received which were sent by link-layer multicast.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_BYTES
	uint64_t sent_bytes; //!< Total number of transmitted bytes.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_PACKETS
	uint64_t sent_packets; //!< Total number of sent packets.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_ERRORS
	uint64_t sent_errors; //!< Amount of sent errors.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_DROP
	uint64_t sent_drop; //!< Total number of sent packets that had been dropped.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_FIFO
	uint64_t sent_fifo; //!< The number of fifo buffer errors sent.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_COMPRESSED
	uint64_t sent_compressed; //!< The number of compressed packets transmitted by the device driver.
#endif
#ifdef SA_NETWORK_INTERFACE_SENT_MULTICAST
	uint64_t sent_multicast; //!< Number of packets sent by link-layer multicast.
#endif
};

#ifdef SA_OPEN_NETWORK
/**
 * Prepares the resources needed for retrieving network statistics. This function exists (and is needed) only when SA_OPEN_NETWORK is defined.
 * @return If successful 0 is returned, otherwise an error code is returned. If the operative system is not supported the return value will be ENOTSUP.
 * @see sa_close_network()
 */
SA_EXPORT int sa_open_network(void);
#endif

#ifdef SA_CLOSE_NETWORK
/**
 * This function closes the resources used for retrieving network statistics. You should call it even when there was a previous error in another function of this API. This function exists (and is needed) only when SA_CLOSE_NETWORK is defined.
 * @return If successful 0 is returned, otherwise an error code is returned.
 * @see sa_open_network()
 */
SA_EXPORT int sa_close_network(void);
#endif

/**
 * Gives the total number of network interfaces.
 * @param number The number will be stored here
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_count_network_interfaces(uint16_t* number);

/**
 * Retrieves statistics from a network interface identified by its name.
 * @param name The name of the network interface.
 * @param dst Where the statistics will be stored.
 * @return If successful 0 is returned, otherwise an error code is returned. ENODEV is returned when the requested device was not found.
 */
SA_EXPORT int sa_get_network_interface(char* name, struct sa_network_interface* dst);

/**
 * Retrieves statistics about all the network interfaces' activity.
 * @param dst A buffer where the statistics will be stored.
 * @param dst_size The number of interfaces that fits in the dst buffer. If it's not big enough dst will be filled but ENOMEM will be returned.
 * @param written The amount of interface statistics written.
 * @return If successful 0 is returned, otherwise an error code is returned.
 */
SA_EXPORT int sa_get_network_interfaces(struct sa_network_interface* dst, uint16_t dst_size, uint16_t* written);

/*@}*/
#endif /* SA_COMMON_NETWORK_H_ */
