/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <pthread.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/disk.h>
#include <string.h>

#include "libsysactivity.h"
#include "utils.h"

pthread_key_t buffer_key;
pthread_key_t buffer_size_key;

static void create_keys();
void assign(struct sa_data_storage* dst, struct diskstats* src);

int sa_open_data_storage() {
	pthread_once_t key_once = PTHREAD_ONCE_INIT;
	if (pthread_once(&key_once, create_keys))
		return ENOSYS;

	return 0;
}

int sa_count_data_storages(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	int mib[] = { CTL_HW, HW_DISKCOUNT };
	int n_disks;
	size_t size = sizeof number;
	if (sysctl(mib, 2, &n_disks, &size, NULL, 0) < 0)
		return ENOSYS;

	*number = n_disks;
	return 0;
}

int sa_get_data_storage(char* name, struct sa_data_storage* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	uint16_t number;
	int ret = sa_count_data_storages(&number);
	if (ret != 0)
		return ret;

	struct diskstats* buffer = pthread_getspecific(buffer_key);
	size_t size = number * sizeof(struct diskstats);
	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	if (buffer_size < size) {
		buffer = realloc(buffer, size);
		if (pthread_setspecific(buffer_key, buffer))
			return ENOSYS;
		if (buffer == NULL)
			return ENOMEM;
		buffer_size = size;
		if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
			return ENOSYS;
	}

	int mib[] = { CTL_HW, HW_DISKSTATS };
	if (sysctl(mib, 2, buffer, &buffer_size, NULL, 0) < 0)
		return ENOSYS;

	number = buffer_size / sizeof(struct diskstats);
	int i;
	for (i = 0; i < number; i++) {
		if (strncmp(buffer[i].ds_name, name, DS_DISKNAMELEN) != 0)
			continue;

		assign(dst, &buffer[i]);
		return 0;
	}

	return ENODEV;
}

int sa_get_data_storages(struct sa_data_storage* dst, uint16_t dst_size, uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	*written = 0;
	uint16_t number;
	int ret = sa_count_data_storages(&number);
	if (ret != 0)
		return ret;

	struct diskstats* buffer = pthread_getspecific(buffer_key);
	size_t size = number * sizeof(struct diskstats);
	size_t buffer_size = (size_t) pthread_getspecific(buffer_size_key);
	if (buffer_size < size) {
		buffer = realloc(buffer, size);
		if (pthread_setspecific(buffer_key, buffer))
			return ENOSYS;
		if (buffer == NULL)
			return ENOMEM;
		buffer_size = size;
		if (pthread_setspecific(buffer_size_key, (void *) buffer_size))
			return ENOSYS;
	}

	int mib[] = { CTL_HW, HW_DISKSTATS };
	if (sysctl(mib, 2, buffer, &buffer_size, NULL, 0) < 0)
		return ENOSYS;

	number = buffer_size / sizeof(struct diskstats);
	int i;
	for (i = 0; i < number; i++) {
		if (i == dst_size)
			return ENODEV;

		assign(&dst[i], &buffer[i]);
		(*written)++;
	}

	return 0;
}

static void create_keys() {
	static short keys_created = 0;
	if (keys_created) {
		return;
	}

	pthread_key_create(&buffer_key, free_keys);
	pthread_key_create(&buffer_size_key, NULL);

	keys_created = 1;
}

void assign(struct sa_data_storage* dst, struct diskstats* src) {
	strlcpy(dst->name, src->ds_name, DS_DISKNAMELEN);
	dst->reads = src->ds_rxfer;
	dst->writes = src->ds_wxfer;
	dst->bytes_read = src->ds_rbytes;
	dst->bytes_written = src->ds_wbytes;
}
