/*
 * libsysactivity
 * http://sourceforge.net/projects/libsysactivity/
 * Copyright (c) 2009, 2010 Carlos Olmedo Escobar <carlos.olmedo.e@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <errno.h>
#include <string.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>

#include "libsysactivity.h"
#include "utils.h"

static void parse_interface(struct sa_network_interface* dst, char* pos);

extern int kernel_version;
__thread FILE* file_net_dev;
__thread char line_buffer[256];
__thread long pos_useful_data;

int sa_open_network() {
	file_net_dev = NULL;

	if (kernel_version < 11)
		return ENOTSUP;

	file_net_dev = fopen("/proc/net/dev", "r");
	if (file_net_dev == NULL)
		return EIO;

	fgets(line_buffer, sizeof line_buffer, file_net_dev);
	fgets(line_buffer, sizeof line_buffer, file_net_dev);
	pos_useful_data = ftell(file_net_dev);
	if (pos_useful_data == -1)
		return ENOTSUP;

	return 0;
}

int sa_close_network() {
	if (file_net_dev != NULL)
		fclose(file_net_dev);
	return 0;
}

int sa_count_network_interfaces(uint16_t* number) {
	if (number == NULL)
		return EINVAL;

	if (fflush(file_net_dev) != 0)
		return EIO;
	fseek(file_net_dev, pos_useful_data, SEEK_SET);

	*number = 0;
	while (fgets(line_buffer, sizeof line_buffer, file_net_dev))
		(*number)++;

	return 0;
}

int sa_get_network_interface(char* name, struct sa_network_interface* dst) {
	if (name == NULL || dst == NULL)
		return EINVAL;

	if (fflush(file_net_dev) != 0)
		return EIO;
	fseek(file_net_dev, pos_useful_data, SEEK_SET);

	char* begin;
	char* end;
	while (fgets(line_buffer, sizeof line_buffer, file_net_dev)) {
		end = strrchr(line_buffer, ':');
		*end = ' ';

		begin = line_buffer;
		while (*begin == ' ')
			begin++;
		if (strncmp(begin, name, sizeof(dst->name) < end - begin ? sizeof(dst->name) : end - begin) != 0)
			continue;

		errno = 0;
		parse_interface(dst, begin);
		if (errno != 0)
			return ENOSYS;

		return 0;
	}
	return ENODEV;
}

int sa_get_network_interfaces(struct sa_network_interface* dst, uint16_t dst_size,
		uint16_t* written) {
	if (dst == NULL || dst_size == 0 || written == NULL)
		return EINVAL;

	if (fflush(file_net_dev) != 0)
		return EIO;
	fseek(file_net_dev, pos_useful_data, SEEK_SET);

	*written = 0;
	int i;
	char* begin;
	for (i = 0; i < dst_size; i++) {
		if (fgets(line_buffer, sizeof line_buffer, file_net_dev) == NULL)
			return ENODEV;
		*strrchr(line_buffer, ':') = ' ';

		begin = line_buffer;
		while (*begin == ' ')
			begin++;

		errno = 0;
		parse_interface(&dst[i], begin);
		if (errno != 0)
			return ENOSYS;

		(*written)++;
	}
	return 0;
}

static void parse_interface(struct sa_network_interface* dst, char* pos) {
	int length = strchr(pos, ' ') - pos;
	if (sizeof(dst->name) - 1 < length)
		length = sizeof(dst->name) - 1;
	int i;
	for (i = 0; i < length; i++)
		dst->name[i] = pos[i];
	dst->name[length] = '\0';
	pos = skip_value(pos);
	dst->received_bytes = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->received_packets = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->received_errors = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->received_drop = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->received_fifo = strtoull(pos, NULL, 10);
	pos = skip_values(pos, 2);
	dst->received_compressed = strtoull(pos, NULL, 10);
	pos = skip_values(pos, 2);
	dst->sent_bytes = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->sent_packets = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->sent_errors = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->sent_drop = strtoull(pos, NULL, 10);
	pos = skip_value(pos);
	dst->sent_fifo = strtoull(pos, NULL, 10);
	pos = skip_values(pos, 3);
	dst->sent_compressed = strtoull(pos, NULL, 10);
}
