/* MouseTool.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.mouse;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import javax.swing.JComponent;

/**
 * Wraps a list of mouse handlers for use with a {@link MouseOverlay}.
 */
public class MouseTool implements Iterable<MouseHandler> {
    
    /**
     * Default constructor.
     */
    public MouseTool () {
        handlers = new ArrayList<MouseHandler> ();
    }
    
    //
    private final List<MouseHandler> handlers;
    
    /**
     * Add a handler to this tool. The order in which handlers
     * are registered determines the order in which they are tried when
     * the mouse is pressed on the overlay which uses this tool.<p>
     * Do not call this method while the tool is registered with
     * some overlay.
     */
    public void addHandler (MouseHandler handler) {
        handlers.add (handler);
    }
    
    /**
     * Returns true if and only if all handlers registered with this tool
     * can handle the given component.
     */
    public boolean canHandle(JComponent parent) {
        for (MouseHandler h: handlers)
            if (! h.canHandle (parent))
                return false;
        return true;
    }
    
    /**
     * Return an iterator over the handlers of this tool.
     */
    public Iterator<MouseHandler> iterator () {
        return handlers.iterator ();
    }
    
}
