/* FileSaveAction.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.actions;

import java.awt.Window;
import java.io.File;
import javax.swing.filechooser.FileFilter;

/**
 * Abstract super class for actions that allow files to be saved.
 * Clients should at least override {@link #saveFile}.
 */
public abstract class FileSaveAction extends FileAction {

    /**
     * Create an action of this type
     * @param parent Parent window for the file chooser dialog
     * @param filters File filters for the file chooser dialog
     */
    public FileSaveAction(Window parent, FileFilter... filters) {
        super(parent, filters);
        this.fileChooser.setMultiSelectionEnabled(false);
    }

    /**
     * Called for the file selected by the user. Must be overridden 
     * by clients.
     */
    protected abstract void saveFile(File file);

    /**
     * Alert the user that a file with the chosen name already exists.
     * @return true if the user allows the file to be overwritten.
     */
    protected abstract boolean showFileExistsDialog(File file);

    /**
     * Shows the file chooser save dialog.
     */
    protected int showDialog() {
        return fileChooser.showSaveDialog(parent);
    }

    /**
     * Invoke {@link #saveFile} on the file chosen.
     * When the file already exists, the user is asked to
     * confirm the save using {@link #showFileExistsDialog}. If the user types 
     * in a file name without extension and the selected file filter is of type
     * {@link ExtensionFileFilter}, then an extension is appended 
     * to the file name automatically.
     */
    protected void dialogApproved() {
        File file = fileChooser.getSelectedFile();
        FileFilter selectedFilter = fileChooser.getFileFilter();
        if (selectedFilter instanceof ExtensionFileFilter) {
            file = ((ExtensionFileFilter) selectedFilter).ensureExtension(file);
        }
        if (!file.exists() || showFileExistsDialog(file)) {
            saveFile(file);
        }
    }
}
