/*


    ========== licence begin GPL
    Copyright (C) 2002-2003 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


*/
package com.sap.dbtech.jdbc.translators;


import java.sql.SQLException;

import com.sap.dbtech.jdbc.DBProcParameterInfo;
import com.sap.dbtech.jdbc.DBProcParameterInfo.StructureElement;
import com.sap.dbtech.jdbc.exceptions.SQLExceptionSapDB;
import com.sap.dbtech.util.MessageKey;
import com.sap.dbtech.util.MessageTranslator;
import com.sap.dbtech.util.StringUtil;
import com.sap.dbtech.util.StructuredBytes;
import com.sap.dbtech.util.StructuredMem;
import com.sap.dbtech.vsp00.StreamHandle;

public abstract class StructMemberTranslator
{
	protected DBProcParameterInfo.StructureElement structureElement;
	protected int index;
	protected int offset;
	
	public StructMemberTranslator(DBProcParameterInfo.StructureElement structureElement,
								  int index,
								  boolean unicode) {
		this.structureElement = structureElement;
		this.index = index;						
		this.offset = unicode ? structureElement.unicodeOffset : structureElement.asciiOffset;	  	    
	}
	
	public abstract Object getObject(StructuredMem memory, int recordOffset);
	public abstract void putObject(StructuredMem memory, Object object) throws SQLException;
	public abstract void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException;
	
	protected void throwConversionError(String sourceObject) throws SQLException {
		throw new SQLExceptionSapDB(MessageTranslator.translate(
			MessageKey.ERROR_STRUCT_ELEMENT_CONVERSION,
			structureElement.getSQLTypeName(),
			sourceObject
		));
	}
	
	// --- CHAR BYTE	
	static class ByteStructureElementTranslator extends StructMemberTranslator {


		public ByteStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			byte[] bytes = memory.getBytes(offset + recordOffset, structureElement.length);
			StructuredBytes bb=new StructuredBytes(bytes);
//			System.err.println("===== BYTES AT " + (offset + recordOffset));
			bb.traceOn(System.err);
			if(structureElement.length == 1) {
				return new Byte(bytes[0]);				
			} else {
				return bytes;
			}
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof byte[]) {
				byte[] ba = (byte[])o;
				memory.putBytes(ba, offset);
			} else if(o instanceof Byte) {
				byte[] ba = new byte[1];
				ba[0] = ((Byte)o).byteValue();
			} else {
				throwConversionError(o.getClass().getName());
			}
		}

		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Char_C,
							 structureElement.length,
							 0,
							 structureElement.length,
							 offset);
		}
	}		
	
	// --- CHAR ASCII
	static class CharAsciiStructureElementTranslator extends StructMemberTranslator {

		public CharAsciiStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			byte[] bytes = memory.getBytes(offset + recordOffset, structureElement.length);
			if(structureElement.length == 1) {
				return new Character((char)bytes[0]);		
			} else {
				return new String(bytes);
			}
		}
		
		public void putObject(StructuredMem memory, Object o)
			throws SQLException {
			String convStr = null;	
			if(o instanceof char[]) {
				convStr = new String(((char[])o));
			} else if(o instanceof String) {
				convStr = (String) o;
			} else if(o instanceof Character) {
				char ca[] = new char[] {
				((Character)o).charValue() 
				};
				convStr = new String(ca);
			} else {
				throwConversionError(o.getClass().getName());
			}
			byte[] bytes = StringUtil.bytes_ascii7bit(convStr);
			memory.putStringBytes(bytes, offset, this.structureElement.length);
		}
		
		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Char_C,
							 structureElement.length,
							 0,
							 structureElement.length,
							 offset);
		}
	}

	static class WydeStructureElementTranslator extends StructMemberTranslator {

		public WydeStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			char[] ca  = memory.getBigUnicode(offset + recordOffset, structureElement.length * 2);
			if(structureElement.length == 1) {
				return new Character(ca[0]);
			} else {
				return new String(ca);
			}
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			String convStr = null;
			if(o instanceof char[]) {
				convStr = new String((char[])o);
			} else if(o instanceof String) {
				convStr = (String) o;
			} else if(o instanceof Character) {
				char ca[] = new char[] {
					((Character)o).charValue() 
				};
				convStr = new String(ca);
			} else {
				throwConversionError(o.getClass().getName());
			}
			memory.putBigUnicode(convStr.toCharArray(), offset, structureElement.length*2);
		}
		
		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_WYDE_C,
							 structureElement.length * 2,
							 0,
							 structureElement.length,
							 offset);
		}
		
	}
	
	static class ShortStructureElementTranslator extends StructMemberTranslator {

		public ShortStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			int shortval = memory.getInt2(offset + recordOffset);
			return new Short((short)shortval); 
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof Short) {
				short sval = ((Short)o).shortValue();
				memory.putInt2(sval, offset);
			} else if(o instanceof Number) {
				double dval = ((Number)o).doubleValue();
				if(dval > Short.MAX_VALUE || dval < Short.MIN_VALUE) {
					throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STRUCT_ELEMENT_OVERFLOW, structureElement.getSQLTypeName(), o.toString()));
				}
				short sval = ((Number)o).shortValue();
				memory.putInt2(sval, offset);
			} else {
				throwConversionError(o.getClass().getName());
			} 
		}
		
		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Int2_C,
							 2,
							 0,
							 1,
							 offset);
		}

	}
	
	static class IntStructureElementTranslator extends StructMemberTranslator {

		public IntStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			int val = memory.getInt4(offset+recordOffset);
			return new Integer(val); 
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof Integer) {
				int ival = ((Integer)o).intValue();
				memory.putInt4(ival, offset);
			} else if(o instanceof Number) {
				double dval = ((Number)o).doubleValue();
				if(dval > Integer.MAX_VALUE || dval < Integer.MIN_VALUE) {
					throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STRUCT_ELEMENT_OVERFLOW, structureElement.getSQLTypeName(), o.toString()));
				}
				int ival = ((Number)o).intValue();
				memory.putInt4(ival, offset);
			} else {
				throwConversionError(o.getClass().getName());
			} 
		}

		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Int4_C,
							 4,
							 0,
							 1,
							 offset);
		}

	}

	static class LongStructureElementTranslator extends StructMemberTranslator {

		public LongStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			long val = memory.getInt8(offset+recordOffset);
			return new Long(val); 
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof Long) {
				long lval = ((Long)o).longValue();
				memory.putInt8(lval, offset);
			} else if(o instanceof Number) {
				double dval = ((Number)o).doubleValue();
				if(dval > Long.MAX_VALUE || dval < Long.MIN_VALUE) {
					throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STRUCT_ELEMENT_OVERFLOW, structureElement.getSQLTypeName(), o.toString()));
				}
				long lval = ((Number)o).longValue();
				memory.putInt8(lval, offset);
			} else {
				throwConversionError(o.getClass().getName());
			} 
		}

		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Int8_C,
							 8,
							 0,
							 1,
							 offset);
		}

	}

	static class DoubleStructureElementTranslator extends StructMemberTranslator {

		public DoubleStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			long val = memory.getInt8(offset + recordOffset);
			return new Double(Double.longBitsToDouble(val)); 
		}

		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof Number) {
				double dval = ((Number)o).doubleValue();
				long lval = Double.doubleToLongBits(dval);
				memory.putInt8(lval, offset);
			} else {
				throwConversionError(o.getClass().getName());
			} 
		}
		
		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Double_C,
							 8,
							 0,
							 1,
							 offset);
		}

	}

	static class FloatStructureElementTranslator extends StructMemberTranslator {

		public FloatStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			int val = memory.getInt4(offset);
			return new Float(Float.intBitsToFloat(val)); 
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof Float) {
				float fval = ((Float)o).floatValue();
				int ival = Float.floatToIntBits(fval);
				memory.putInt4(ival, offset);
			} else if(o instanceof Number) {
				double dval = ((Number)o).doubleValue();
				if(dval > Float.MAX_VALUE || dval < Float.MIN_VALUE) {
					throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_STRUCT_ELEMENT_OVERFLOW, structureElement.getSQLTypeName(), o.toString()));
				}
				float fval = ((Number)o).floatValue();
				int ival = Float.floatToIntBits(fval);
				memory.putInt4(ival, offset);
			} else {
				throwConversionError(o.getClass().getName());
			} 
		}
		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Float_C,
							 4,
							 0,
							 1,
							 offset);
		}
	}

	static class BooleanStructureElementTranslator extends StructMemberTranslator {

		public BooleanStructureElementTranslator(StructureElement structureElement, int index, boolean unicode) {
			super(structureElement, index, unicode);
		}

		public Object getObject(StructuredMem memory, int recordOffset) {
			byte b = memory.getInt1(offset+recordOffset);
			if(b == 0) {
				return new Boolean(false);
			} else {
				return new Boolean(true);
			}
		}
		
		public void putObject(StructuredMem memory, Object o) throws SQLException {
			if(o instanceof Boolean) {
				boolean bval = ((Boolean)o).booleanValue();
				if(bval) {
					memory.putInt1(1, offset);
				} else {
					memory.putInt1(0, offset);
				}
			} else {
				throwConversionError(o.getClass().getName());
			} 
		}
		
		public void addOutStreamDescriptor(AbstractABAPStreamGetval getval) throws SQLException {
			getval.addColumn(StreamHandle.Stream_OUT_C,
							 StreamHandle.StreamType_Int1_C,
							 1,
							 0,
							 1,
							 offset);
		}
	}
	
	public static StructMemberTranslator createStructMemberTranslator(DBProcParameterInfo paramInfo,
		int index,
		boolean unicode) 
		throws SQLException {
		DBProcParameterInfo.StructureElement s = paramInfo.getMember(index);
		if(s.typeName.equals("CHAR")) {
			if(s.codeType.equals("BYTE")) {
				return new ByteStructureElementTranslator(s, index, unicode);
			} else if(s.codeType.equals("ASCII")) {
				return new CharAsciiStructureElementTranslator(s, index, unicode);
			} 
		} else if(s.typeName.equals("WYDE")) {
			if(unicode) {
				return new WydeStructureElementTranslator(s, index, unicode);
			} else {
				return new CharAsciiStructureElementTranslator(s, index, unicode);
			}
		} else if(s.typeName.equals("SMALLINT")) {
			if(s.length == 5) {
				return new ShortStructureElementTranslator(s, index, unicode);
			}
		} else if(s.typeName.equals("INTEGER")) {
			if(s.length == 10) {
				return new IntStructureElementTranslator(s, index, unicode);
			} else if(s.length == 19) {
				return new LongStructureElementTranslator(s, index, unicode);
			}
		} else if(s.typeName.equals("FIXED")) {
			if(s.precision == 0) {
				if(s.length == 5) {
					return new ShortStructureElementTranslator(s, index, unicode);
				} else if(s.length == 10) {
					return new IntStructureElementTranslator(s, index, unicode);
				} else if(s.length == 19) {
					return new LongStructureElementTranslator(s, index, unicode);
				}
			}
		} else if(s.typeName.equals("FLOAT")) {
			if(s.length == 15) {
				return new DoubleStructureElementTranslator(s, index,unicode);			
			} else if(s.length == 6) {
				return new FloatStructureElementTranslator(s, index, unicode);
			}
		} else if(s.typeName.equals("BOOLEAN")) {
			return new BooleanStructureElementTranslator(s, index, unicode);
		} 
		throw new SQLExceptionSapDB(MessageTranslator.translate(MessageKey.ERROR_CONVERSION_STRUCTURETYPE, new Integer(index), s.getSQLTypeName()));
	}

	/**
	 * Creates the translators for a structure.
	 * @param info The extended parameter info.
	 * @param unicode Whether this is an unicode connection.
	 * @return The converter array.
	 */
	public static StructMemberTranslator[] createStructMemberTranslators(DBProcParameterInfo info, boolean unicode) 
		throws SQLException {
		
		StructMemberTranslator[] result = new StructMemberTranslator[info.getMemberCount()];
		
		for(int i=0; i<result.length; ++i) {
			result[i] = createStructMemberTranslator(info, i, unicode);
		}
		
		return result;
	}


}