/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.connector;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TreeMap;

/**
 * A JMXAddress is an URL in form of
 * <pre>
 * jmx:<protocol>://<host>:<port>/<path>;<property>
 * </pre>
 * used to identify the address of an adaptor.
 *
 * @version $Revision: 1.3 $
 */
public class JMXAddress implements Serializable, Cloneable
{
   private String address;
   private transient String m_protocol;
   private transient String m_host;
   private transient int m_port;
   private transient String m_path;
   private transient HashMap m_properties;

   public JMXAddress(String address) throws MalformedJMXAddressException
   {
      parse(address);
      this.address = toString();
   }

   public JMXAddress(String proto, String host, int port, String path) throws MalformedJMXAddressException
   {
      StringBuffer buffer = new StringBuffer("jmx:");
      buffer.append(proto).append("://").append(host);
      if (port > 0) buffer.append(":").append(port);
      buffer.append("/").append(path);

      parse(buffer.toString());

      this.address = toString();
   }

   /**
    * Protocol is in the form jrmp, iiop, https and so on.
    */
   public String getProtocol()
   {
      return m_protocol;
   }

   private void setProtocol(String protocol)
   {
      m_protocol = protocol;
   }

   /**
    * Returns the host part
    */
   public String getHost()
   {
      return m_host;
   }

   private void setHost(String host)
   {
      m_host = host;
   }

   /**
    * Returns the port
    */
   public int getPort()
   {
      return m_port;
   }

   private void setPort(int port)
   {
      m_port = port;
   }

   /**
    * Returns the path
    */
   public String getPath()
   {
      return m_path;
   }

   private void setPath(String path)
   {
      m_path = path;
   }

   /**
    * Returns the value of the specified property
    */
   public Object getProperty(String key)
   {
      return m_properties == null ? null : m_properties.get(key);
   }

   /**
    * Sets the specified property
    */
   public void putProperty(String key, Object value)
   {
      if (m_properties == null)
      {
         m_properties = new HashMap();
      }
      m_properties.put(key, value);
      // Refresh the value of the address
      this.address = toString();
   }

   /**
    * Returns a clone of the properties of this ServiceURL
    */
   public Map getProperties()
   {
      return m_properties == null ? null : (Map)m_properties.clone();
   }

   public int hashCode()
   {
      return address.hashCode();
   }

   public final boolean equals(Object obj)
   {
      if (obj == null)
      {
         return false;
      }
      if (obj == this)
      {
         return true;
      }

      try
      {
         return this.isEqual(obj) && ((JMXAddress)obj).isEqual(this);
      }
      catch (ClassCastException ignored)
      {
      }
      return false;
   }

   protected boolean isEqual(Object obj)
   {
      try
      {
         return address.equals(((JMXAddress)obj).address);
      }
      catch (ClassCastException ignored)
      {
      }
      return false;
   }

   public Object clone()
   {
      try
      {
         JMXAddress clone = (JMXAddress)super.clone();
         if (m_properties != null)
         {
            clone.m_properties = (HashMap)m_properties.clone();
         }
         return clone;
      }
      catch (CloneNotSupportedException ignored)
      {
         return null;
      }
   }

   public String toString()
   {
      StringBuffer buffer = new StringBuffer("jmx:");
      buffer.append(getProtocol()).append("://").append(getHost());
      int port = getPort();
      if (port > 0)
      {
         buffer.append(":").append(port);
      }
      buffer.append("/").append(getPath());
      if (m_properties.size() > 0)
      {
         TreeMap ordered = new TreeMap(m_properties);
         for (Iterator i = ordered.entrySet().iterator(); i.hasNext();)
         {
            buffer.append(";");
            Map.Entry entry = (Map.Entry)i.next();
            buffer.append(entry.getKey()).append("=").append(entry.getValue());
         }
      }
      return buffer.toString();
   }

   private void parse(String url) throws MalformedJMXAddressException
   {
      if (url == null)
      {
         throw new MalformedJMXAddressException("JMXAddress cannot be null");
      }

      if (m_properties == null)
      {
         m_properties = new HashMap();
      }

      String service = "jmx:";
      if (!url.startsWith(service))
      {
         throw new MalformedJMXAddressException("JMXAddress must start with '" + service + "'");
      }

      url = url.substring(service.length());

      // find the host
      String hostSeparator = "://";
      int index = url.indexOf(hostSeparator);
      if (index < 0)
      {
         throw new MalformedJMXAddressException("JMXAddress must have a host");
      }

      String protocol = url.substring(0, index).trim();
      String hostAndMore = url.substring(index + hostSeparator.length());

      if (protocol.length() == 0)
      {
         throw new MalformedJMXAddressException("JMXAddress must have a protocol");
      }
      setProtocol(protocol);

      index = hostAndMore.indexOf('/');
      if (index < 0)
      {
         // No path specified, look for properties
         int semicolon = hostAndMore.indexOf(';');
         if (semicolon < 0)
         {
            // No support for query part (protocol://host:port/path?query)
            parseHostAndPort(hostAndMore);
         }
         else
         {
            String hostAndPort = hostAndMore.substring(0, semicolon);
            parseHostAndPort(hostAndPort);
            String more = hostAndMore.substring(semicolon + 1);
            parseProperties(more);

         }
      }
      else
      {
         String hostAndPort = hostAndMore.substring(0, index);
         if (hostAndPort.length() == 0)
         {
            throw new MalformedJMXAddressException("JMXAddress must have a host");
         }
         parseHostAndPort(hostAndPort);

         String pathAndMore = hostAndMore.substring(index + 1);
         if (pathAndMore.length() > 0)
         {
            int semicolon = pathAndMore.indexOf(';');
            if (semicolon < 0)
            {
               // No support for query part (protocol://host:port/path?query)
               String path = pathAndMore;
               setPath(path);
            }
            else
            {
               String path = pathAndMore.substring(0, semicolon);
               setPath(path);
               String more = pathAndMore.substring(semicolon + 1);
               parseProperties(more);
            }
         }
      }
   }

   private void parseHostAndPort(String hostAndPort) throws MalformedJMXAddressException
   {
      int colon = hostAndPort.indexOf(':');
      if (colon == 0)
      {
         throw new MalformedJMXAddressException("JMXAddress must have a host");
      }
      if (colon > 0)
      {
         String host = hostAndPort.substring(0, colon);
         setHost(host);
         String portString = hostAndPort.substring(colon + 1);
         try
         {
            int port = Integer.parseInt(portString);
            setPort(port);
         }
         catch (NumberFormatException x)
         {
            throw new MalformedJMXAddressException("JMXAddress has an invalid port");
         }
      }
      else
      {
         if (hostAndPort.length() == 0)
         {
            throw new MalformedJMXAddressException("JMXAddress must have a host");
         }
         setHost(hostAndPort);
         setPort(-1);
      }
   }

   private void parseProperties(String properties) throws MalformedJMXAddressException
   {
      StringTokenizer tokenizer = new StringTokenizer(properties, ",");
      while (tokenizer.hasMoreTokens())
      {
         String token = tokenizer.nextToken().trim();
         int equal = token.indexOf('=');
         if (equal <= 0)
         {
            throw new MalformedJMXAddressException("JMXAddress has an invalid property");
         }
         String key = token.substring(0, equal);
         String value = token.substring(equal + 1);
         putProperty(key, value);
      }
   }
}
