// $Id: DemoFrame.java,v 1.19 2007/01/11 08:28:39 pietschy Exp $
// Copyright (c) 2000-2001 The Forge Group. All rights reserved.
/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: DemoFrame.java,v 1.19 2007/01/11 08:28:39 pietschy Exp $
 */
package org.pietschy.command.demo;

import org.pietschy.command.*;
import org.pietschy.command.delegate.*;
import org.pietschy.explicit.TableBuilder;

import javax.swing.*;
import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

public class
DemoFrame
extends JFrame
implements HoverListener
{
   static final String _ID_ = "$Id: DemoFrame.java,v 1.19 2007/01/11 08:28:39 pietschy Exp $";
   private JTabbedPane tabs;
   private PageSelectorGroup demoSelectorGroup;
   private CommandGroup mainMenuGroup;
   private CommandGroup toobarGroup;
   private JLabel statusBar;
   private DemoPanel demoPanel;
   private TableBuilder builder;


   /**
    * Constructs a new frame that is initially invisible.
    * <p>
    * This constructor sets the component's locale property to the value
    * returned by <code>JComponent.getDefaultLocale</code>.
    *
    * @throws HeadlessException if GraphicsEnvironment.isHeadless()
    *                           returns true.
    * @see java.awt.GraphicsEnvironment#isHeadless
    * @see Component#setSize
    * @see Component#setVisible
    * @see JComponent#getDefaultLocale
    */
   public DemoFrame()
   throws HeadlessException
   {
      super("GUI Commands Demo");
      ImageIcon image = new ImageIcon(getClass().getResource("images/logo_16x16.png"));
      setIconImage(image.getImage());
      addWindowListener(new WindowAdapter()
      {
         /**
          * Invoked when a window is in the process of being closed.
          * The close operation can be overridden at this point.
          */
         public void windowClosing(WindowEvent e)
         {
            CommandManager.defaultInstance().getCommand("menu.file.exit").execute();
         }
      });

      CommandManager.defaultInstance().addHoverListener(this);

      demoSelectorGroup = new PageSelectorGroup(CommandManager.defaultInstance(), "page.selectors");
      demoSelectorGroup.export();

      mainMenuGroup = CommandManager.defaultInstance().getGroup("menu.main");
      toobarGroup = CommandManager.defaultInstance().getGroup("main.toolbar");

      JMenuBar menuBar = mainMenuGroup.createMenuBar();
      setJMenuBar(menuBar);

      DelegatingCommand printCommand = new DelegatingCommand("print");
      printCommand.export();
      printCommand.trackDelegateIn("print-delegate", this);

      DelegateManager.setDelegateMediatorFactory(new DefaultDelegateMediatorFactory());

      tabs = new JTabbedPane();
      statusBar = new JLabel(" ");
      statusBar.setVerticalAlignment(SwingConstants.TOP);
      statusBar.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
      JPanel footer = new JPanel(new BorderLayout());
      footer.add(statusBar, BorderLayout.CENTER);
      footer.setBorder(BorderFactory.createEtchedBorder());
      footer.setPreferredSize(footer.getPreferredSize());
      footer.setMinimumSize(footer.getPreferredSize());
      footer.setMaximumSize(footer.getPreferredSize());

      JScrollPane sp = new JScrollPane(demoSelectorGroup.createButtonBar(), JScrollPane.VERTICAL_SCROLLBAR_ALWAYS, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

      builder = new TableBuilder();
      int row = 0;
      // column 0
      builder.add(sp, row, 0, 2, 1).fill();
      // column 1
      builder.add(toobarGroup.createToolBar(), row, 1).fillX();

      row++;
      builder.add(tabs, row, 1).fill();
      builder.row(row).paddingTop(0);

      row++;
      builder.add(footer, row, 0, 1, 2).fillX().ignoreColumnPadding(true);


      builder.setIgnorePaddingOnEdges(false);
      double padding = 4;
      builder.column(1).paddingLeft(padding);
      builder.row(0).paddingTop(padding);
      builder.row(2).paddingTop(padding);
      builder.column(0).paddingLeft(padding);
      builder.column(1).paddingRight(padding);

      builder.row(1).grow(1);
      builder.column(1).grow(1);

      JPanel panel = builder.getPanel();
      panel.setPreferredSize(new Dimension(720,520));
      setContentPane(panel);

      pack();

      setLocationRelativeTo(null);

      sp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
   }

   public void
   registerDemo(DemoPanel panel)
   {
      PageSelectCommand selector = panel.getSelector();
      demoSelectorGroup.add(selector);
   }

   /**
    * Loads the specified demo panel into the main frame.  This method will install the panels
    * menu and/or toolbar if provided.
    *
    * @param p the {@link DemoPanel} to be installed.
    */
   public void
   setDemoPanel(DemoPanel p)
   {
      if (demoPanel != null)
      {
         mainMenuGroup.visit(new ResetVisitor(ResetVisitor.SHALLOW));
         toobarGroup.reset();
      }

      demoPanel = p;

      tabs.removeAll();
      tabs.addTab(demoPanel.getName(), demoPanel);
      tabs.addTab("Command File", demoPanel.getCommandFileViewer());

      CommandGroup[] menuGroups = demoPanel.getMenuGroups();
      for (int i = 0; i < menuGroups.length; i++)
      {
         MatchVisitor visitor = new MatchVisitor(menuGroups[i], Face.MENU, MatchVisitor.SHALLOW);
         mainMenuGroup.visit(visitor);
         if (visitor.foundMatch())
            visitor.getMatchingGroup().addInline(menuGroups[i]);
         else
            mainMenuGroup.add(menuGroups[i]);
      }

      CommandGroup toolbar = demoPanel.getToolbarGroup();
      if (toolbar != null)
      {
         toobarGroup.addInline(toolbar);
      }

      // just trying out the focus tracking version...
      FocusTrackingDelegateMediator.getMediatorFor(this).setStaticContainer(demoPanel);

   }

   public void hoverStarted(HoverEvent e)
   {
      String description = e.getFace().getLongDescription();
      if (description == null)
         description = e.getFace().getDescription();

      statusBar.setText(description);
   }

   public void hoverEnded(HoverEvent e)
   {
      statusBar.setText(" ");
   }

}

