=begin
  locale.rb - Locale module

  Copyright (C) 2002-2006  Masao Mutoh

  You may redistribute it and/or modify it under the same
  license terms as Ruby.

  $Id: locale.rb,v 1.11 2006/03/15 16:34:05 mutoh Exp $
=end

require 'gettext/locale_object'

begin
  require 'locale_system'
rescue LoadError
  # This is used for installation process(from rake only)
  # And deprecated these constants.
  module Locale
    # Deprecated.
    CTYPE = 0
    # Deprecated.
    NUMERIC = 1
    # Deprecated.
    TIME = 2
    # Deprecated.
    COLLATE = 3
    # Deprecated.
    MONETARY = 4
    # Deprecated.
    MESSAGES = 5
    # Deprecated.
    ALL = 6
  end
end

if /cygwin|mingw|win32/ =~ RUBY_PLATFORM
  require 'gettext/locale_win32'
else
  require 'gettext/locale_posix'
end

# Locale module manages the locale informations of the application.
module Locale
  unless defined? CTYPE
    CTYPE = 0    #:nodoc:
    NUMERIC = 1  #:nodoc:
    TIME = 2     #:nodoc:
    COLLATE = 3  #:nodoc:
    MONETARY = 4 #:nodoc:
    MESSAGES = 5 #:nodoc:
    ALL = 6      #:nodoc:
  end

  @@default = nil
  @@current = nil

  module_function
  # Sets the default locale (Locale::Object).
  #
  # * locale: the default locale
  # * Returns: self.
  def set_default(locale)
    @@default = locale
    if @@default
      @@default.charset ||= @@locale_system_module.get_charset(locale)
    end
    self
  end
  # Same as Locale.set_default.
  #
  # * locale: the default locale (Locale::Object).
  # * Returns: locale.
  def default=(locale)
    set_default(locale)
    @@default
  end

  # Gets the system locale.
  # * Returns: the system locale (Locale::Object).
  def system
    @@locale_system_module.system
  end

  # Gets the default locale.
  #
  # If the default locale not set, this returns system locale.
  # * Returns: the default locale (Locale::Object).
  def default
    @@default ? @@default : system
  end

  # Gets the current locale (Locale::Object).
  #
  # If the current locale is not set, this returns default locale. 
  # * Returns: the current locale (Locale::Object).
  def current
    @@current = default unless @@current
    @@current
  end

  # Sets a locale as the current locale.
  #
  # This returns the current Locale::Object.
  # * lang: Locale::Object or locale name(String), or language name.
  # * country: the country code(String)
  # * charset: the charset(override the charset even if the locale name has charset).
  # * Returns: self
  #
  #    Locale.set_current("ja_JP.eucJP")
  #    Locale.set_current("ja", "JP")
  #    Locale.set_current("ja", "JP", "eucJP")
  #    Locale.set_current("ja", nil, "eucJP")
  #    Locale.set_current(Locale::Object.new("ja", "JP", "eucJP"))
  def set_current(lang, country = nil, charset = nil)
    if lang == nil
      @@current = nil
    else
      if lang.kind_of? Locale::Object
	@@current = lang
      else
	@@current = Locale::Object.new(lang, country, charset)
      end
      @@current.charset ||= @@locale_system_module.get_charset(@@current)
    end
    self
  end

  # Sets a current locale. This is a single argument version of Locale.set_current.
  #
  # * lang: the Locale::Object
  # * Returns: the current locale (Locale::Object).
  #
  #    Locale.current = "ja_JP.eucJP"
  #    Locale.current = Locale::Object.new("ja", "JP", "eucJP")
  def current=(lang)
    set_current(lang)
    @@current
  end

  # call-seq:
  #   set(lang)
  #   set(lang, country = nil, charset = nil)
  #
  # * lang: language as String, or Locale::Object
  # * country: country as String or nil
  # * charset: charset as String or nil
  # * Returns: a Locale::Object.
  #
  # Sets a default locale. This function is an alias of Locale.set_default.
  #
  # *Notice*: Locale.set(lctype, locale) is deprecated.
  def set(lang, country = nil, charset = nil)
    if lang.kind_of? Numeric
      warn "Locale.set(lctype, locale) is deprecated. Use Locale.set(locale) instead." 
      # Locale.set(Locale::CTYPE, "ja_JP.eucJP") "country" means locale.
      lang = country
      country = nil
      charset = nil
    end

    set_current(nil)
    if lang.kind_of? String
      set_default(Locale::Object.new(lang, country, charset))
    elsif lang.kind_of? Locale::Object
      set_default(lang)
    else
      set_default(nil)
    end
    @@default
  end

  # call-seq:
  #   current
  #   get
  #   get(lctype = nil)  (Deprecated)
  #
  # * Returns: the current locale (Locale::Object).
  #
  # *Notice*: lctype is deprecated. Use this with no parameter instead.
  def get(lctype = nil)
    warn "Locale.get(lctype) is deprecated. Use Locale.get (noparam) instead." if lctype
    @@current = default unless @@current
    @@current
  end

  # *Deprecated*. Use Locale.set or Locale.set_current instead.
  def setlocale(lctype, loc)
    warn "Deprecated. Use Locale#set instead."
    set(lctype, loc)
  end

  # Same as charset. Gets the charset of the current locale.
  # * Returns: the charset of the current locale (String)
  def codeset
    current.charset
  end

  # Gets the charset of the current locale.
  # * Returns: the charset of the current locale (String)
  def charset
    codeset
  end 
  # Same as codeset. Returns the charset of the current locale. 
  # * Returns: the charset of the current locale (String)
  def current_charset
    codeset
  end

  # Clear default/current locale.
  # * Returns: self
  def clear
    set(nil)
    set_current(nil)
    self
  end
end
