/** -*- C++ -*-
   @file cache/apt/index.tcc
   @author Peter Rockai <me@mornfall.net>
*/

#include <wibble/range.h>

#include <apt-pkg/strutl.h> // for stringcasecmp
#include <apt-pkg/configuration.h> // for _config

// for cache generation
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/sourcelist.h>

#include <ept/cache/apt/index.h>
#include <ept/path.h>
#include <ept/error.h>

#ifndef EPT_CACHE_APT_INDEX_TCC
#define EPT_CACHE_APT_INDEX_TCC
namespace ept {
namespace t {
namespace cache {
namespace apt {

template< typename P >
time_t Index< P >::currentTimestamp() {
    return Path::timestamp( _config->FindFile( "Dir::Cache::pkgcache" ) );
}

template< typename P >
void Index< P >::reload()
{
    close();
    open();
}

template< typename P >
void Index< P >::close()
{
    delete m_aptCache;
    m_aptCache = 0;
    m_sorted.clear();
    m_indirector.invalidate();
}

template< typename P >
void Index< P >::open()
{
    pkgSourceList list;
    if ( list.ReadMainList() == false ) {
        _error->DumpErrors();
        throw wibble::exception::System( "The list of sources could not be read." );
    }

    MMap *m = 0;
    OpProgress dummy;
    bool Res = pkgMakeStatusCache( list, dummy, &m, true ); // true -> allow mem
    // progress().Done();
    if ( !Res ) {
        std::cerr << "The package lists or status file could not be parsed or opened." << std::endl;
        throw wibble::exception::System(
            "The package lists or status file could not be parsed or opened." );
    }

    m_aptCache = new pkgCache( m, true );
    checkGlobalError( "Failed building cache" );
    m_timestamp = currentTimestamp();

    m_ondiskToPointer.resize( packageCount() + 1, 0 );
    for ( pkgCache::PkgIterator p = aptCache().PkgBegin(); p != aptCache().PkgEnd(); ++ p ) {
        m_ondiskToPointer[ p->ID ] = p;
    }
    m_ondiskToPointer[ packageCount() ] = pkgPtr();
}

template< typename T >
bool lessByName( const T &e1, const T &e2 )
{
    return e1.name() < e2.name();
}

template< typename P >
Index< P >::Index( Aggregator &ag )
    : m_aggregator( ag ), m_indirector( ag ), m_sorted( lessByName< Package > )
{
    open();
    m_hashIndex.resize( packageCount(), -1 ); // roughly
}

template< typename P >
wibble::Range< typename Index< P >::Package > Index< P >::sorted()
{
    if ( m_sorted.empty() ) {
        std::cerr << "Index sorting packages..." << std::endl;
        /* ownerCache()->progress().OverallProgress(
           0, packageCount(), 100, gettext( "Sorting" ) ); */
        int done = 0;
        iterator e = end();
        for ( iterator i = begin(); i != e; ++i ) {
            done ++;
            m_sorted.insert( m_sorted.begin(), *i );
            /* if ( done % 100 == 0 )
               ownerCache()->progress().Progress( done ); */
            // ownerCache()->progress().Done();
        }
        std::cerr << "sorted " << done << " packages..." << std::endl;
    }
    return wibble::range( m_sorted );
}

template< typename P >
int Index< P >::hashSize() const
{
    return sizeof (aptCache().HeaderP->HashTable) / sizeof (aptCache().HeaderP->HashTable[0]);
}

template< typename P >
void Index< P >::advancePackageInCache( Package &p )
{
    const Aggregator &packages = p.pointer().aggregator();
    const Index &idx = packages.index();

    // p.checkValid();
    pkgCache::Package *pkg = p.pointer().package();
    long hidx = p.pointer().hashIndex();

    // Follow the current links
    if ( pkg != idx.pkgPtr() ) {
        pkg = idx.pkgPtr( pkg->NextPackage );
    }

    // Follow the hash table
    while ( pkg == idx.pkgPtr() && ( hidx + 1 ) < idx.hashSize() )
    {
        ++ hidx;
        pkg = idx.pkgPtr( idx.hashOffset( hidx ) );
    }

    // we would wrap around, put out an invalid entity
    if ( pkg == idx.pkgPtr() )
        p = Package( PackagePointer( idx.m_aggregator, 0 ) );
    else
        p = idx.createPackage( pkg, hidx );
}

template< typename P >
static bool lastPackageInCache( const typename Index< P >::Package &p ) {
    return p == typename Index< P >::Package();
}

template< typename P >
typename Index< P >::PackageRange Index< P >::range() {
    Package f = createPackage( pkgPtr(), -1 );
    // std::cerr << "Index::range(): hash size = " << hashSize() << std::endl;
    advancePackageInCache( f );
    // std::cerr << "Index::range(): first pkg hash = " << f.pointer().hashIndex() << std::endl;
    return wibble::generatedRange( f, advancePackageInCache, lastPackageInCache< P > );
}

template< typename P >
pkgCache::Package *Index< P >::aptPackageByName( const std::string& n ) const
{
    const char *Name = n.c_str();
    pkgCache::Package *Pkg = pkgPtr(
        aptCache().HeaderP->HashTable[ aptCache().Hash( Name ) ] );
    for (; Pkg != aptCache().PkgP; Pkg = aptCache().PkgP + Pkg->NextPackage)
    {
        if (Pkg->Name != 0 && aptCache().StrP[Pkg->Name] == Name[0] &&
            ::stringcasecmp(Name,aptCache().StrP + Pkg->Name) == 0)
            return Pkg; // createPackage( Pkg, 0 );
    }
    return 0;
}

template< typename P >
typename Index< P >::Package Index< P >::packageByName( const std::string &n ) const {
    pkgCache::Package *p = aptPackageByName( n );
    if ( p == 0 )
        return Package();
    return createPackage( p, -1 );
}

template< typename P >
typename Index< P >::Package Index< P >::createPackage( pkgCache::Package *p, long h ) const
{
    int id = m_indirector.ondiskToRuntime( p == pkgPtr() ? packageCount() : p->ID );
    assert( id >= 0 );
    if ( m_hashIndex.size() <= static_cast< unsigned >( id ) )
        m_hashIndex.resize( id + 1, -1 );
    if ( h != -1 )
        m_hashIndex[ id ] = h;
    return Package( PackagePointer( m_aggregator, id ) );
}

template< typename P >
typename Index< P >::Package Index< P >::packageForVersion( Version v ) const {
    return createPackage( pkgPtr( v.pointer().version()->ParentPkg ), -1 );
    // return Version( &cache(), p.pointer().package()->CurrentVersion );
}

}
}
}
}
#endif
