/* 
 **  mod_layout.c -- Apache layout module
 **
 **  Copyright 2007 Brian Aker brian@tangent.org
 **  
 ** Please see "LICENSE" file that was included in the distribution.
 **  
*/

#include "mod_layout.h"

module AP_MODULE_DECLARE_DATA layout_module;

static void
set_config_defaults(apr_pool_t *p, layout_conf *cfg)
{
  memset(cfg, 0, sizeof(layout_conf));
  cfg->notes= UNSET;
  cfg->comment= UNSET;
  cfg->append_header= UNSET;
  cfg->append_footer= UNSET;
  cfg->display_origin= ON;
  cfg->header_enabled= UNSET;
  cfg->footer_enabled= UNSET;
  cfg->begin_tag= LAYOUT_BEGINTAG;
  cfg->end_tag= LAYOUT_ENDTAG;
  cfg->time_format= LAYOUT_TIMEFORMAT;
}

static void *create_dir_mconfig(apr_pool_t *p, char *dir) 
{
  layout_conf *cfg;

  cfg= apr_pcalloc(p, sizeof(layout_conf));
  set_config_defaults(p, cfg);
  cfg->dir= apr_pstrdup(p, dir);

  return (void *) cfg;
}

static void *merge_dir_mconfig(apr_pool_t *p, void *origin, void *new) 
{
  layout_conf *cfg= apr_pcalloc(p, sizeof(layout_conf));
  set_config_defaults(p, cfg);
  layout_conf *cfg_origin= (layout_conf *)origin;
  layout_conf *cfg_new= (layout_conf *)new;
  cfg->dir= apr_pstrdup(p, cfg_new->dir);

  if (strcmp(cfg_new->time_format, LAYOUT_TIMEFORMAT))
    cfg->time_format= apr_pstrdup(p, cfg_new->time_format);
  else if (strcmp(cfg_origin->time_format, LAYOUT_TIMEFORMAT))
    cfg->time_format= apr_pstrdup(p, cfg_origin->time_format);

  if (strcmp(cfg_new->begin_tag, LAYOUT_BEGINTAG))
    cfg->begin_tag= apr_pstrdup(p, cfg_new->begin_tag);
  else if (strcmp(cfg_origin->begin_tag, LAYOUT_BEGINTAG))
    cfg->begin_tag= apr_pstrdup(p, cfg_origin->begin_tag);

  if (strcmp(cfg_new->end_tag, LAYOUT_ENDTAG))
    cfg->end_tag= apr_pstrdup(p, cfg_new->end_tag);
  else if (strcmp(cfg_origin->end_tag, LAYOUT_ENDTAG))
    cfg->end_tag= apr_pstrdup(p, cfg_origin->end_tag);

  cfg->notes= (cfg_new->notes == UNSET) ? cfg_origin->notes : cfg_new->notes;
  cfg->comment= (cfg_new->comment == UNSET) ? cfg_origin->comment : cfg_new->comment;
  cfg->display_origin= cfg_new->display_origin;

  cfg->append_header= (cfg_new->append_header == UNSET) ? cfg_origin->append_header : cfg_new->append_header;
  cfg->append_footer= (cfg_new->append_footer == UNSET) ? cfg_origin->append_footer : cfg_new->append_footer;

  if (isOn(cfg->append_header) || isOn(cfg->append_footer))
  {
    if (isOn(cfg->append_header) && isOn(cfg->append_footer))
    {
      if (cfg_origin->layouts)
        cfg->layouts= apr_array_append(p, cfg_origin->layouts, cfg_new->layouts);
      else
        cfg->layouts= cfg_new->layouts;

      cfg->header_enabled= cfg_new->header_enabled ? cfg_new->header_enabled : cfg_origin->header_enabled;
      cfg->footer_enabled= cfg_new->footer_enabled ? cfg_new->footer_enabled : cfg_origin->header_enabled;
    } 
    else if (isOn(cfg->append_header)) 
    {
      cfg->header_enabled= cfg_new->header_enabled ? cfg_new->header_enabled : cfg_origin->header_enabled;
      cfg->footer_enabled= cfg_new->footer_enabled;
      cfg->layouts= layout_array_push_kind(p, cfg_origin->layouts, cfg_new->layouts, HEADER);
      /* We now assume just append_footer */
    } 
    else  
    {
      cfg->header_enabled= cfg_new->header_enabled;
      cfg->footer_enabled= cfg_new->footer_enabled ? cfg_new->footer_enabled : cfg_origin->header_enabled;
      cfg->layouts= layout_array_push_kind(p, cfg_origin->layouts, cfg_new->layouts, FOOTER);
    }
  } 
  else 
  {
    if (cfg_new->layouts)
    {
      cfg->layouts= cfg_new->layouts;
      cfg->header_enabled= cfg_new->header_enabled;
      cfg->footer_enabled= cfg_new->footer_enabled;
    } 
    else 
    {
      cfg->layouts= cfg_origin->layouts;
      cfg->header_enabled= cfg_origin->header_enabled;
      cfg->footer_enabled= cfg_origin->footer_enabled;
    }
  }

  if (cfg_origin->uris_ignore) 
  {
    if (cfg_new->uris_ignore) 
      cfg->uris_ignore= apr_table_overlay(p, cfg_new->uris_ignore, cfg_origin->uris_ignore);
    else 
      cfg->uris_ignore= cfg_origin->uris_ignore;
  } 
  else 
    cfg->uris_ignore= cfg_new->uris_ignore;

  if (cfg_origin->uris_ignore_header) 
  {
    if (cfg_new->uris_ignore_header) 
      cfg->uris_ignore_header= apr_table_overlay(p, cfg_new->uris_ignore_header, cfg_origin->uris_ignore_header);
    else
      cfg->uris_ignore_header= cfg_origin->uris_ignore_header;
  } 
  else
    cfg->uris_ignore_header= cfg_new->uris_ignore_header;

  if (cfg_origin->uris_ignore_footer) 
  {
    if (cfg_new->uris_ignore_footer)
      cfg->uris_ignore_footer= apr_table_overlay(p, cfg_new->uris_ignore_footer, cfg_origin->uris_ignore_footer);
    else
      cfg->uris_ignore_footer= cfg_origin->uris_ignore_footer;
  }
  else
    cfg->uris_ignore_footer= cfg_new->uris_ignore_footer;

  if (cfg_origin->tag_ignore) 
  {
    if (cfg_new->tag_ignore)
      cfg->tag_ignore= apr_table_overlay(p, cfg_new->tag_ignore, cfg_origin->tag_ignore);
    else
      cfg->tag_ignore= cfg_origin->tag_ignore;
  } 
  else
    cfg->tag_ignore= cfg_new->tag_ignore;

  if (cfg_origin->tag_ignore_footer) 
  {
    if (cfg_new->tag_ignore_footer)
      cfg->tag_ignore_footer= apr_table_overlay(p, cfg_new->tag_ignore_footer, cfg_origin->tag_ignore_footer);
    else
      cfg->tag_ignore_footer= cfg_origin->tag_ignore_footer;
  } 
  else
    cfg->tag_ignore_footer= cfg_new->tag_ignore_footer;

  if (cfg_origin->tag_ignore_header) 
  {
    if (cfg_new->tag_ignore_header)
      cfg->tag_ignore_header= apr_table_overlay(p, cfg_new->tag_ignore_header, cfg_origin->tag_ignore_header);
    else
      cfg->tag_ignore_header= cfg_origin->tag_ignore_header;
  } 
  else
    cfg->tag_ignore_header= cfg_new->tag_ignore_header;

  return (void *) cfg;
}

static apr_status_t layout_filter(ap_filter_t *f, apr_bucket_brigade *b) 
{
  request_rec *r= f->r;
  layout_filter_struct *ctx= f->ctx;
  apr_bucket *e;
  const char *str;
  apr_size_t len;
  layout_request *info= NULL;
  layout_conf *cfg;
  int start_position= 0;


  if (r->main) 
  {
    ap_remove_output_filter(f);
    return ap_pass_brigade(f->next, b);
  }

  cfg= ap_get_module_config(r->per_dir_config, &layout_module);

  WATCHPOINT_STRING(r->content_type);
  WATCHPOINT_STRING(r->handler);

  apr_table_setn(r->headers_out, "X-Powered-By", "ModLayout/"VERSION);

  /* If no layouts were specified, we skip */
  if (cfg->layouts == NULL) 
  {
    ap_remove_output_filter(f);
    return ap_pass_brigade(f->next, b);
  }

  /* Now lets look at the ignore logic */
  /* This is where we finally decide what is going to happen */
  if (cfg->uris_ignore && table_find(cfg->uris_ignore, r->uri))
  {
    ap_remove_output_filter(f);
    return ap_pass_brigade(f->next, b);
  }

  info= create_layout_request(r, cfg);

  if (isOff(info->header)  && isOff(info->footer))
  {
    ap_remove_output_filter(f);
    return ap_pass_brigade(f->next, b);
  }

  if (ctx == NULL) 
  {
    f->ctx= ctx= apr_pcalloc(f->r->pool, sizeof(*ctx));
    ctx->b= apr_brigade_create(f->r->pool, r->connection->bucket_alloc);
    ctx->output= NULL;
  }

  apr_table_unset(f->r->headers_out, "Content-Length");
  apr_table_unset(f->r->headers_out, "ETag");

  for (e = APR_BRIGADE_FIRST(b);
       e != APR_BRIGADE_SENTINEL(b);
       e = APR_BUCKET_NEXT(e)) 
  {
    if (APR_BUCKET_IS_EOS(e) || APR_BUCKET_IS_FLUSH  (e)) 
    {
      info->f= f->next;
      info->b= ctx->b;
      if (isOn(info->origin))  
      {
        /* This code right here could be replaced and this would all be faster */
        /* We have to make two passes through to make sure the headers always happen */
        if (info->header && (string_search(r, ctx->output, cfg->begin_tag, start_position, 0) == -1)) 
          layout_kind(r, cfg, info, HEADER);
        parser_put(r, cfg, info,  ctx->output, start_position);
        if (info->footer && (string_search(r, ctx->output, cfg->end_tag, start_position, 0) == -1))
          layout_kind(r, cfg, info, FOOTER);
      } 
      else 
      {
        layout_kind(r, cfg, info, HEADER);
        if (isOn(cfg->notes))
          update_info(r->notes, info);
        layout_kind(r, cfg, info, LAYOUT);
        if (isOn(cfg->notes))
          update_info(r->notes, info);
        layout_kind(r, cfg, info, FOOTER);
      }

      APR_BUCKET_REMOVE(e);
      APR_BRIGADE_INSERT_TAIL(ctx->b, e);

      return ap_pass_brigade(f->next, ctx->b);
    }

    if (APR_BUCKET_IS_FLUSH(e)) 
      continue;
    apr_bucket_read(e, &str, &len, APR_NONBLOCK_READ);

    if (ctx->output)
      ctx->output= apr_psprintf(r->pool,"%s%.*s", ctx->output, (int)len, str);
    else
      ctx->output= apr_pstrndup(r->pool, str, len);
  }

  apr_brigade_destroy(b);
  return APR_SUCCESS;
}

static const char * add_layout_pattern(cmd_parms * cmd, void *mconfig, const char *pattern, const char *layout, const char *mode) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  struct stat sbuf;
  const char *temp;
  layout_string *directive;


  directive= apr_pcalloc (cmd->pool, sizeof (layout_string));

  if (ap_ind(layout, ' ') != -1) 
  {
    directive->comment= apr_pstrdup (cmd->pool, "Static Content");
    directive->string= apr_pstrdup (cmd->pool, layout);
    directive->type= 1;
  } 
  else if (stat(layout, &sbuf) == 0)
  {
    if ((temp= layout_add_file(cmd, layout)) == 0) 
      return NULL;
    directive->comment= apr_pstrdup (cmd->pool, layout);
    directive->string= apr_pstrdup (cmd->pool, temp);
    directive->type= 1;
  } 
  else 
  {
    directive->comment= apr_pstrdup (cmd->pool, layout);
    directive->string= apr_pstrdup (cmd->pool, layout);
    directive->type= 0;
  }
  directive->pattern= apr_pstrdup (cmd->pool, pattern);

  if (cfg->layouts == 0) 
    cfg->layouts= apr_array_make (cmd->pool, 1, sizeof (layout_string *));

  if (strcasecmp(cmd->cmd->name, "LayoutHeader") == 0) 
  {
    cfg->header_enabled= ON;
    directive->kind= HEADER;
  } 
  else if (strcasecmp(cmd->cmd->name, "LayoutFooter") == 0) 
  {
    cfg->footer_enabled= ON;
    directive->kind= FOOTER;
  } 
  else 
  {
    directive->kind= LAYOUT;
  }

  if (mode) 
  {
    if (strcasecmp(mode, "append") == 0) 
      directive->append= APPEND;
    else if (strcasecmp(mode, "prepend") == 0) 
      directive->append= PREPEND;
    else if(strcasecmp(mode, "replace") == 0) 
      directive->append= REPLACE;
    else 
    {
      ap_log_error(APLOG_MARK, APLOG_WARNING, 0, cmd->server,
                   "The type %s is not valid for %s ",mode, cmd->cmd->name);
      directive->append= REPLACE;
    }
  } 
  else 
    directive->append= REPLACE;

   *(layout_string **) apr_array_push (cfg->layouts)= (layout_string *) directive;

  return NULL;
}

static const char * add_layout(cmd_parms * cmd, void *mconfig, const char *layout) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (strcasecmp(cmd->cmd->name, "LayoutHeader") == 0) 
    (void)add_layout_pattern(cmd, mconfig, (char *)cfg->begin_tag, layout, "append");
  else if (strcasecmp(cmd->cmd->name, "LayoutFooter") == 0)
    (void)add_layout_pattern(cmd, mconfig, (char *)cfg->end_tag, layout, "prepend");
  return NULL;
}


static const char * ignore_uri(cmd_parms * cmd, void *mconfig, const char *uri) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (cfg->uris_ignore == 0)
    cfg->uris_ignore= apr_table_make(cmd->pool, 1);

  apr_table_setn(cfg->uris_ignore, uri, "1");

  return NULL;
}

static const char * ignore_header_uri(cmd_parms * cmd, void *mconfig, const char *uri) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (cfg->uris_ignore_header == 0)
    cfg->uris_ignore_header= apr_table_make(cmd->pool, 1);

  apr_table_setn(cfg->uris_ignore_header, uri, "1");

  return NULL;
}

static const char * ignore_footer_uri(cmd_parms * cmd, void *mconfig, const char *uri) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (cfg->uris_ignore_footer == 0)
    cfg->uris_ignore_footer= apr_table_make(cmd->pool, 1);

  apr_table_setn(cfg->uris_ignore_footer, uri, "1");

  return NULL;
}

static const char * tag_ignore_add(cmd_parms * cmd, void *mconfig, const char *type) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (cfg->tag_ignore == 0)
    cfg->tag_ignore= apr_table_make(cmd->pool, 1);

  apr_table_setn(cfg->tag_ignore, type, "1");

  return NULL;
}

static const char * tag_ignore_footer_add(cmd_parms * cmd, void *mconfig, const char *type) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (cfg->tag_ignore_footer == 0)
    cfg->tag_ignore_footer= apr_table_make(cmd->pool, 1);

  apr_table_setn(cfg->tag_ignore_footer, type, "1");

  return NULL;
}

static const char * tag_ignore_header_add(cmd_parms * cmd, void *mconfig, const char *type) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  if (cfg->tag_ignore_header == 0)
    cfg->tag_ignore_header= apr_table_make(cmd->pool, 1);

  apr_table_setn(cfg->tag_ignore_header, type, "1");

  return NULL;
}

#ifdef NOT_USED
static const char * footer_off(cmd_parms * cmd, void *mconfig) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  cfg->footer_enabled= OFF;

  return NULL;
}

static const char * header_off(cmd_parms * cmd, void *mconfig) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  cfg->header_enabled= OFF;

  return NULL;
}
#endif

static const char * append_layouts(cmd_parms * cmd, void *mconfig, int flag) 
{
  layout_conf *cfg= (layout_conf *) mconfig;
  cfg->append_header= flag;
  cfg->append_footer= flag;

  return NULL;
}

/* Just here for debugging purposes */
static int layout_fixup(request_rec *r)
{
  WATCHPOINT_STRING(r->content_type);
  WATCHPOINT_STRING(r->handler);

  return DECLINED;
}

static void layout_register_hooks(apr_pool_t *p) 
{
  ap_register_output_filter("LAYOUT", layout_filter, NULL, AP_FTYPE_CONTENT_SET);
#ifdef HAVE_DEBUG
  ap_hook_fixups(layout_fixup, NULL, NULL, APR_HOOK_LAST);
#endif
}

static const command_rec layout_cmds[]= {
  AP_INIT_TAKE23("Layout", add_layout_pattern, NULL, OR_ALL, Layout),
  AP_INIT_TAKE1("LayoutHeader", add_layout, NULL, OR_ALL, LayoutHeader),
  AP_INIT_TAKE1("LayoutFooter", add_layout, NULL, OR_ALL, LayoutFooter),
  AP_INIT_FLAG("LayoutAppend", append_layouts, NULL, OR_ALL, LayoutAppend),
  AP_INIT_TAKE1("LayoutIgnoreURI", ignore_uri, NULL, OR_ALL, LayoutIgnoreURI),
  AP_INIT_TAKE1("LayoutIgnoreHeaderURI", ignore_header_uri, NULL, OR_ALL, LayoutIgnoreHeaderURI),
  AP_INIT_TAKE1("LayoutIgnoreFooterURI", ignore_footer_uri, NULL, OR_ALL, LayoutIgnoreFooterURI),
  AP_INIT_FLAG("LayoutComment", ap_set_flag_slot, (void *) APR_OFFSETOF(layout_conf, comment), OR_ALL, LayoutComment),
  AP_INIT_FLAG("LayoutDisplayOriginal", ap_set_flag_slot, (void *) APR_OFFSETOF(layout_conf, display_origin), OR_ALL, LayoutDisplayOriginal),
  AP_INIT_TAKE1("LayoutTimeFormat", ap_set_string_slot, (void *) APR_OFFSETOF(layout_conf, time_format), OR_ALL, LayoutTimeFormat),
  AP_INIT_TAKE1("LayoutIgnoreTag", tag_ignore_add, NULL, OR_ALL, LayoutIgnoreTag),
  AP_INIT_TAKE1("LayoutIgnoreTagFooter", tag_ignore_footer_add, NULL, OR_ALL, LayoutIgnoreTagFooter),
  AP_INIT_TAKE1("LayoutIgnoreTagHeader", tag_ignore_header_add, NULL, OR_ALL, LayoutIgnoreTagHeader),
  AP_INIT_FLAG("LayoutNotes", ap_set_flag_slot, (void *) APR_OFFSETOF(layout_conf, notes), OR_ALL, LayoutNotes),
  {NULL}
};

/* Dispatch list for API hooks */
module AP_MODULE_DECLARE_DATA layout_module= {
  STANDARD20_MODULE_STUFF, 
  create_dir_mconfig,    /* create per-dir    config structures */
  merge_dir_mconfig,     /* merge  per-dir    config structures */
  NULL,                  /* create per-server config structures */
  NULL,                  /* merge  per-server config structures */
  layout_cmds,           /* table of config file commands       */
  layout_register_hooks  /* register hooks                      */
};
