/*
 * Minimalist Kernel Debugger - Architecture independent stack traceback
 *
 * Copyright (C) 1999 Silicon Graphics, Inc.
 * Copyright (C) Scott Lurndal (slurn@engr.sgi.com)
 * Copyright (C) Scott Foehner (sfoehner@engr.sgi.com)
 * Copyright (C) Srinivasa Thirumalachar (sprasad@engr.sgi.com)
 *
 * See the file LIA-COPYRIGHT for additional information.
 *
 * Written March 1999 by Scott Lurndal at Silicon Graphics, Inc.
 *
 * Modifications from:
 *      Richard Bass                    1999/07/20
 *              Many bug fixes and enhancements.
 *      Scott Foehner
 *              Port to ia64
 *      Srinivasa Thirumalachar
 *              RSE support for ia64
 *	Masahiro Adegawa                1999/12/01
 *		'sr' command, active flag in 'ps'
 *	Scott Lurndal			1999/12/12
 *		Significantly restructure for linux2.3
 *	Keith Owens			2000/05/23
 *		KDB v1.2
 *	Keith Owens			2000/09/16
 *		KDB v1.4
 *		Env BTAPROMPT.
 *
 */

#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/kdb.h>
#include <linux/kdbprivate.h>
#include <asm/system.h>


/*
 * kdb_bt
 *
 *	This function implements the 'bt' command.  Print a stack
 *	traceback.
 *
 *	bt [<address-expression>]   (addr-exp is for alternate stacks)
 *	btp <pid>		     (Kernel stack for <pid>)
 *
 * 	address expression refers to a return address on the stack.  It
 *	is expected to be preceeded by a frame pointer.
 *
 * Inputs:
 *	argc	argument count
 *	argv	argument vector
 *	envp	environment vector
 *	ef	registers at time kdb was entered.
 * Outputs:
 *	None.
 * Returns:
 *	zero for success, a kdb diagnostic if error
 * Locking:
 *	none.
 * Remarks:
 *	Backtrack works best when the code uses frame pointers.  But
 *	even without frame pointers we should get a reasonable trace.
 *
 *	mds comes in handy when examining the stack to do a manual
 *	traceback.
 */

int
kdb_bt(int argc, const char **argv, const char **envp, kdb_eframe_t ef)
{
	int	diag;
	int	argcount = 5;
	int	btaprompt = 1;
	char	buffer[80];
	int 	nextarg;
	unsigned long addr;
	long	offset;

	kdbgetintenv("BTARGS", &argcount);	/* Arguments to print */
	kdbgetintenv("BTAPROMPT", &btaprompt);	/* Prompt after each proc in bta */

	if (strcmp(argv[0], "bta") == 0) {
		struct task_struct *p;

		for_each_task(p) {
			kdb_printf("Stack traceback for pid %d\n", p->pid);

			diag = kdba_bt_process(p, argcount);

			if (btaprompt) {
				kdb_getstr(buffer, sizeof(buffer),
					   "Enter <q> to end, <cr> to continue:");

				if (buffer[0] == 'q') {
					return 0;
				}
			}
		}
	} else if (strcmp(argv[0], "btp") == 0) {
		struct task_struct *p;
		unsigned long	   pid;
		
		if (argc < 1)
			return KDB_ARGCOUNT;

		diag = kdbgetularg((char *)argv[1], &pid);
		if (diag)
			return diag;

		for_each_task(p) {
			if (p->pid == (pid_t)pid) {
				return kdba_bt_process(p, argcount);
			}
		}

		kdb_printf("No process with pid == %ld found\n", pid);
		return 0;
	} else {
		if (argc) {
			nextarg = 1;
			diag = kdbgetaddrarg(argc, argv, &nextarg, &addr,
					     &offset, NULL, ef);
			if (diag)
				return diag;

			return kdba_bt_stack(ef, &addr, argcount, current);
		} else {
			return kdba_bt_stack(ef, NULL, argcount, current);
		}
	}

	/* NOTREACHED */
	return 0;
}
