/*
The DsTool program is the property of:
 
                             Cornell University 
                        Center of Applied Mathematics 
                              Ithaca, NY 14853
                      dstool_bugs@macomb.tn.cornell.edu
 
and may be used, modified and distributed freely, subject to the following
restrictions:
 
       Any product which incorporates source code from the DsTool
       program or utilities, in whole or in part, is distributed
       with a copy of that source code, including this notice. You
       must give the recipients all the rights that you have with
       respect to the use of this software. Modifications of the
       software must carry prominent notices stating who changed
       the files and the date of any change.
 
DsTool is distributed in the hope that it will be useful, but WITHOUT ANY 
WARRANTY; without even the implied warranty of FITNESS FOR A PARTICULAR PURPOSE.
The software is provided as is without any obligation on the part of Cornell 
faculty, staff or students to assist in its use, correction, modification or
enhancement.
*/

/* 
 * fixed_top_def.c
 */

#include <model_headers.h>

/* spinning top with one pont fixed */
/*
   This vector field describes the motion of a spinning top
   with the base fixed.

   Here is the Lagrangian formulation:

   T = I1 (w1^2 + w2^2) / 2 + I3 w3^2 / 2
   V = Mgl cos(theta)
   L = T - V

     where

     w1 = d_phi sin(theta) sin(phi) + d_theta cos(phi)
     w2 = d_phi sin(theta) cos(phi) - d_theta sin(phi)
     w3 = d_psi + d_phi cos(theta)

     and d_VAR is shorthand for d(VAR)/dt (the time derivative),
     and phi, theta, psi are the Euler angles for rigid body rotation.

   Lagrange's equations of motion give:

     The generalized momenta (P_VAR = partial_d(L)/partial_d(d_VAR)):

     P_theta = I1 d_theta
     P_phi = I1 d_phi sin(theta)^2 + I3 (D_psi + d_phi cos(theta)) cos(theta)
     P_psi = I3 (d_psi+d_phi cos(theta)

     and evolution (d_P_VAR = partial_D(L)/partial_d(VAR)):

     d_P_theta = I1 d_phi^2 sin(2*theta)/2 
                 - I3 (d_psi+d_phi cos(theta)) * d_phi * sin(theta)
                 + Mgl sin(theta)
     d_P_phi = 0
     d_P_psi = 0

   So P_phi abd P_psi are constant and thus parameters in the problem.

   We can solve for d_VAR in terms on the generalized momenta:

     d_theta = P_theta/I1
     d_phi = (P_phi - cos(theta) P_psi) / (I1 sin(theta)^2)
     d_psi = P_psi/I3 - cos(theta) * (P_phi - cos(theta) P_psi) / 
                                                 (I1 sin(theta)^2)

	Translation table:

		x[0] <--> theta
		x[1] <--> phi
		x[2] <--> psi
		x[3] <--> P_theta
		x[4] <--> t
		p[0] <--> I1            (moment of inertia)
		p[1] <--> I3            (moment of inertia)
		p[2] <--> Mgl           (Mass * gravity * l)
		p[3] <--> P_phi
		p[4] <--> P_psi

*/
/* ------------------------------------------------------------------------
   proc used to define the vector field 
   ------------------------------------------------------------------------ */
int 
fixed_top(double* f, double* x, double* p )
{
  double cos_theta = cos(x[0]);
  double sin_theta = sin(x[0]);

  f[0] = x[3]/p[0];
  f[1] = (p[3]-cos_theta*p[4])/(p[0]*sin_theta*sin_theta);
  f[2] = p[4]/p[1] - cos_theta * f[1];
  f[3] = (p[0]-p[1])*f[1]*f[1]*sin(2.0*x[0])/2.0
    -p[1]*f[1]*f[2]*sin_theta
      +p[2]*sin_theta;
  return(0);

}

/* ------------------------------------------------------------------------
   proc used to define functions of the variable, parameters or derivatives
   ------------------------------------------------------------------------ */
int 
fixed_top_func( double* f, double* x, double* p )
{
  f[0] = 1.5 * sin(x[0]) * cos(x[1]);
  f[1] = 1.5 * sin(x[0]) * sin(x[1]);
  f[2] = 1.5 * cos(x[0]);

  return 0;
}
/* ------------------------------------------------------------------------
   proc used to define jacobian
   ------------------------------------------------------------------------ */
int 
fixed_top_jac( double** m, double* x, double* p )
{
  return 0;
}

/* ---------------------------------
  proc for geomview
  ---------------------------------  */
int 
fixed_top_gv( double* f, double* x, double* p )
{
  /* from Euler angles to coordinate change from space to body */
  /* for geomview we need the transpose of the inverse, but */
  /* this matrix is orthogonal so we are done!!! */
  double cos_theta = cos(x[0]);
  double sin_theta = sin(x[0]);
  double cos_phi = cos(x[1]);
  double sin_phi = sin(x[1]);
  double cos_psi = cos(x[2]);
  double sin_psi = sin(x[2]);

  f[0] = cos_psi*cos_phi - cos_theta*sin_phi*sin_psi;
  f[1] = cos_psi*sin_phi + cos_theta*cos_phi*sin_psi;
  f[2] = sin_psi*sin_theta;
  f[3] = 0;
  f[4] = -sin_psi*cos_phi - cos_theta*sin_phi*cos_psi;
  f[5] = -sin_psi*sin_phi + cos_theta*cos_phi*cos_psi;
  f[6] = cos_psi*sin_theta;
  f[7] = 0;
  f[8] = sin_theta*sin_phi;
  f[9] = -sin_theta*cos_phi;
  f[10] = cos_theta;
  f[11] = 0;
  f[12] = f[13] = f[14] = 0;
  f[15] = 1.0;

  {
    double deform = 1.0+0.1*cos(x[0]);
    int i;
    for (i=0; i<8; i++)
      f[i] = deform*f[i];
  }

  return 0;  
}

/* ------------------------------------------------------------------------
   proc to define the default data for the dynamical system
   Note: You may change the entries for each variable but
	 DO NOT change the list of items.  If a variable is
	 unused, NULL or zero the entry, as appropriate.
   ------------------------------------------------------------------------ */
int 
fixed_top_init(void)
{

  /* define the dynamical system in this segment 
     ------------------------------------------------------------------------------------------- */
  int	 n_varb=4;					  /* dim of the phase spase */
  static char	*variable_names[]={"theta","phi","psi","P_theta"};       	  /* list of phase varb names */
  static double	variables[]={0.3,0.0,0.0,0.0};			  /* initial conditions for the variables */
  static double	variable_min[]={0,-PI,-PI,0};		  /* default varb min for display */
  static double	variable_max[]={HALFPI,PI,PI,0};		  /* default varb max for display */
  static char   *indep_varb_name="time";		  /* name of indep variable  */
  double indep_varb_min=0.;				  /* default indep varb min r display */
  double indep_varb_max=100.;				  /* default indep varb max r display */

  int	 n_param=5;					  /* dim of the parameter space */
  static char	*parameter_names[]={"I1","I3","Mgl", "P_phi", "P_psi"};  /* list of param names */
  static double	parameters[]={1.0,1.0,10.,9.55,10.};		  /* initial parameter values */
  static double	parameter_min[]={0.,0.,0.,0.,0.};		  /* default param min for display */
  static double	parameter_max[]={2.,2.,15.,15.,15.};		  /* default param max for display */

  int	 n_funct=3;					  /* number of user-defined functions */
  static char	*funct_names[]={"x","y","z"};		  /* list of param names */
  static double	funct_min[]={-1.5,-1.5,-1.5};			  /* default varb max for display */
  static double	funct_max[]={1.5,1.5,1.5};			  /* default varb max for display */

  int	 manifold_type = PERIODIC;			  /* EUCLIDEAN or PERIODIC (periodic variables) */
  static int	periodic_varb[] = {TRUE, TRUE, TRUE, FALSE, FALSE, FALSE};	  /* if PERIODIC, which variables are periodic */
  static double period_start[] = {-PI, -PI, -PI, 0., 0., 0.};		  /* if PERIODIC, beginning of fundamental domain */
  static double period_end[] = {PI, PI, PI, 0., 0., 0.};		  /* if PERIODIC, end of fundamental domain */

  int 	 mapping_toggle=FALSE;				  /* is this a map? */
  int	 inverse_toggle=FALSE;				  /* if so, is inverse FALSE, APPROX_INV, or EXPLICIT_INV?*/

  /*  In this section, input NULL or the name of the function which contains... */
  int           (*def_name)()=fixed_top;		  /* the eqns of motion */
  int           (*jac_name)()=fixed_top_jac;		  /* the jacobian (deriv w.r.t. space) */
  int           (*aux_func_name)()=fixed_top_func;	  /* the auxiliary functions */
  int           (*inv_name)()=NULL;			  /* the inverse or approx inverse */
  int           (*dfdt_name)()=NULL;			  /* the deriv w.r.t time */
  int           (*dfdparam_name)()=NULL;		  /* the derivs w.r.t. parameters */

#define GV
  int (*gv_funct)()=fixed_top_gv;   /* transformations for geomview */
  int (*gv_custom_funct)()=NULL;    /* nontransformation geomview commands */
  int gv_n=1;                       /* number of transformations */
  static char *gv_filename = "top.oogl";

  c_filename = __FILE__;  /* display this file for help with this system */

/* ------------------ end of dynamical system definition ------------------ */
#include <ds_define.c>

  return 0;
}

