(****************************************************************************)
(*  Copyright © 2012-2014 Mehdi Dogguy <mehdi@debian.org>                   *)
(*                                                                          *)
(*  This file is part of Dochelp.                                           *)
(*                                                                          *)
(*  Dochelp is free software: you can redistribute it and/or modify it      *)
(*  under the terms of the GNU General Public License as published by the   *)
(*  Free Software Foundation, either version 3 of the License, or (at your  *)
(*  option) any later version.                                              *)
(*                                                                          *)
(*  Dochelp is distributed in the hope that it will be useful, but WITHOUT  *)
(*  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or   *)
(*  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *)
(*  for more details.                                                       *)
(*                                                                          *)
(*  You should have received a copy of the GNU General Public License       *)
(*  along with Dochelp.  If not, see <http://www.gnu.org/licenses/>.        *)
(****************************************************************************)

open Utils

let version = "0.1"

type action =
  | List
  | Search of string
  | Info of string
  | Cat of string
  | Open of string
  | Update
  | Version

let perform = ref List

let spec = Arg.align [
  "--list",
  Arg.Unit (fun () -> perform := List),
  " List available documents";

  "--search",
  Arg.String (fun regexp -> perform := Search regexp),
  "regexp Look for a matching document";

  "--info",
  Arg.String (fun docid -> perform := Info docid),
  "docid Print available information for a document";

  "--cat",
  Arg.String (fun docid -> perform := Cat docid),
  "docid Dump the content of a document";

  "--open",
  Arg.String (fun docid -> perform := Open docid),
  "docid Open a document";

  "--update",
  Arg.Unit (fun () -> perform := Update),
  " Update global indexes";

  "--base-doc",
  Arg.Set_string doc_base_dir,
  Printf.sprintf " Specify doc-base dir (default: %s)" !doc_base_dir;

  "--target",
  Arg.Set_string Html.target,
  Printf.sprintf
    " Specify path of the generated html page (default: %s)"
    !Html.target;

  "--online",
  Arg.Set Html.online,
  Printf.sprintf
    " Use relative paths in generated HTML page (default: %b)"
    !Html.online;

  "--version",
  Arg.Unit (fun () -> perform := Version),
  " Show version number";
]

let () = Arg.parse spec (fun _ -> ()) ""

let () (* main *) =
  match !perform with
  | Version ->
      print_endline version

  | List ->
      iter_files begin fun file filename ->
        let tag = if is_valid_doc_base_file filename
          then "I"
          else "E"
        in
        Printf.printf "%s: %s\n" tag file
      end;
      Printf.printf "%!"

  | Search regexp ->
      let rex = Pcre.regexp regexp in
      iter_files begin fun file filename ->
        if regexp_match rex filename then
          Printf.printf "I: %s\n" file
      end;
      Printf.printf "%!"

  | Cat docid ->
      let path =
        if Sys.file_exists docid then
          docid
        else
          !doc_base_dir // docid
      in
      let code = Sys.command (Printf.sprintf "cat %s" path) in
      exit code

  | Info docid ->
      let path =
        if Sys.file_exists docid then
          docid
        else
          !doc_base_dir // docid
      in
      let ic = open_in path in
      let lexbuf = Lexing.from_channel ic in
      Document.print (Document.read path lexbuf)

  | Update -> Html.process ()

  | _ -> Printf.eprintf "Not implemented yet!\n%!"
