/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

/*
 * Created on Jan 20, 2006
 */
package org.jboss.test.remoting.transport.multiplex;

import java.io.IOException;

import javax.management.MBeanServer;

import org.jboss.logging.Logger;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.PortUtil;
import org.jboss.remoting.transport.multiplex.MultiplexServerInvoker;
import org.jboss.remoting.transport.multiplex.utility.AddressPair;
import org.jboss.test.remoting.transport.multiplex.utility.SuccessCountingTestCase;


/**
 * A InvokerGroupTestCase.

 * @author <a href="mailto:r.sigal@computer.org">Ron Sigal</a>
 * @version $Revision: 1069 $
 * <p>
 * Copyright (c) 2005
 * </p>
 */

public class InvokerGroupTestCase extends SuccessCountingTestCase
{
   private static final Logger log = Logger.getLogger(InvokerGroupTestCase.class);
   
   static int connectPort;
   static String serverLocatorURI;
   static InvokerLocator serverLocator;;
   static Connector connector;

   public void setUp() throws Exception
   {
      super.setUp();
      
      if (connector == null)
      {
         try
         {
            connectPort = PortUtil.findFreePort("localhost");
            serverLocatorURI = "multiplex://localhost:" + connectPort;
            serverLocator = new InvokerLocator(serverLocatorURI);
            connector = new Connector(serverLocator.getLocatorURI());
            connector.create();
            connector.addInvocationHandler("test", new SimpleServerInvocationHandler());
            connector.start();
         }
         catch (Exception e)
         {
            log.error(e);
            e.printStackTrace();
         }
      }
      else
      {
         Thread.sleep(1000);
      }
   }
   
   
   public void testClientStartsClient2WithIdServer1ServerCloses()
   {
      log.info("entering testClientStartsClientStartsClient2WithIdServer1ServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?clientMultiplexId=id&multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort + "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClientStartsClient2WithIdServer1ServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient2WithIdServer1ClientCloses()
   {
      log.info("entering testClientStartsClient2WithIdServer1ClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?clientMultiplexId=id&multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort + "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithIdServer1ClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient2WithIdServer2WithoutIdServerCloses()
   {
      log.info("entering testClientStartsClient2WithIdServer2WithoutIdServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?clientMultiplexId=id&multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithIdServer2WithoutIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient2WithIdServer2WithoutIdClientCloses()
   {
      log.info("entering testClientStartsClient2WithIdServer2WithoutIdServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?clientMultiplexId=id&multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithIdServer2WithoutIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
 
   
   public void testClientStartsClient2WithoutIdServer2WithIdServerCloses()
   {
      log.info("entering testClientStartsClient2WithoutIdServer2WithIdServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithoutIdServer2WithIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient2WithoutIdServer2WithIdClientCloses()
   {
      log.info("entering testClientStartsClient2WithoutIdServer2WithIdClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithoutIdServer2WithIdClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient2WithoutIdServer2WithoutIdServerCloses()
   {
      log.info("entering testClientStartsClient2WithoutIdServer2WithOutIdServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithoutIdServer2WithOutIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient2WithoutIdServer2Without2ClientCloses()
   {
      log.info("entering testClientStartsClient2WithoutIdServer2Without2ClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient2WithoutIdServer2Without2ClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
  

   public void testClientStartsClient3Server1ServerCloses()
   {
      log.info("entering testClientStartsClient3Server1ServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?clientMultiplexId=id";
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();
         
         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient3Server1ServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testClientStartsClient3Server1ClientCloses()
   {
      log.info("entering testClientStartsClient3Server1ClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String params = "/?clientMultiplexId=id";
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testClientStartsClient3Server1ClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   

   public void testServerStartsServer2WithIdClient1ServerCloses()
   {
      log.info("entering testServerStartsServer2WithIdClient1ServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?clientMultiplexId=id";
         String locatorURI =  serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithIdClient1ServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testServerStartsServer2WithIdClient1ClientCloses()
   {
      log.info("entering testServerStartsServer2WithIdClient1ClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?clientMultiplexId=id";
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithIdClient1ClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testServerStartsServer2WithIdClient2WithoutIdServerCloses()
   {
      log.info("entering testServerStartsServer2WithIdClient2WithoutIdServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithIdClient2WithoutIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testServerStartsServer2WithIdClient2WithoutIdClientCloses()
   {
      log.info("entering testServerStartsServer2WithIdClient2WithoutIdClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithIdClient2WithoutIdClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
 
   
   public void testServerStartsServer2WithoutIdClient2WithIdServerCloses()
   {
      log.info("entering testServerStartsServer2WithoutIdClient2WithIdServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?clientMultiplexId=id&multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithoutIdClient2WithIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testServerStartsServer2WithoutIdClient2WithIdClientCloses()
   {
      log.info("entering testServerStartsServer2WithoutIdClient2WithIdClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?clientMultiplexId=id&multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithoutIdClient2WithIdClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testServerStartsServer2WithoutIdClient2WithoutIdServerCloses()
   {
      log.info("entering testServerStartsServer2WithoutIdClient2WithoutIdClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithoutIdClient2WithoutIdServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testServerStartsServer2WithoutIdClient2WithoutIdClientCloses()
   {
      log.info("entering testServerStartsServer2WithoutIdClient2WithoutIdClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?multiplexBindHost=localhost&multiplexBindPort=" + freePort;
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer2WithoutIdClient2WithoutIdClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
  

   public void testServerStartsServer3Client1ServerCloses()
   {
      log.info("entering testServerStartsServer3Client1ServerCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertFalse(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?clientMultiplexId=id";
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer3Client1ServerCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }   
   

   public void testServerStartsServer3Client1ClientCloses()
   {
      log.info("entering testServerStartsServer3Client1ClientCloses()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertFalse(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         String params = "/?clientMultiplexId=id";
         String locatorURI = serverLocatorURI + params;
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Client client = new Client(locatorClient);
         client.connect();
         
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         client.disconnect();
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerStartsServer3Client1ClientCloses() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
  
   
   public void testMultipleClientsSameLocators()
   {
      log.info("entering testMultipleClientsSameLocators()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertFalse(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         int CLIENT_COUNT = 5;
         Client[] clients = new Client[CLIENT_COUNT];
         String locatorURI = serverLocatorURI + "/?clientMultiplexId=id";
         InvokerLocator locatorClient = new InvokerLocator(locatorURI);
         Object o = MultiplexServerInvoker.getSocketGroupMap().get("id");
         MultiplexServerInvoker.SocketGroupInfo socketGroupInfo = (MultiplexServerInvoker.SocketGroupInfo) o;
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            clients[i] = new Client(locatorClient);
            clients[i].connect();
            log.info("started Client: " + i);
            assertTrue(socketGroupInfo.getClientInvokers().size() == 1);
         }
         
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         
         for (int i = 0; i < CLIENT_COUNT - 1; i++)
         {
            clients[i].disconnect();
            assertTrue(socketGroupInfo.getClientInvokers().size() == 1);
         }
         
         clients[CLIENT_COUNT - 1].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 0);
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testMultipleClientsSameLocators() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testMultipleClientsDifferentLocatorsServerStarts()
   {
      log.info("entering testMultipleClientsDifferentLocatorsServerStarts()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertFalse(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         int CLIENT_COUNT = 5;
         Client[] clients = new Client[CLIENT_COUNT];
         Object o = MultiplexServerInvoker.getSocketGroupMap().get("id");
         MultiplexServerInvoker.SocketGroupInfo socketGroupInfo = (MultiplexServerInvoker.SocketGroupInfo) o;
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            String locatorURI = serverLocatorURI + "/?clientMultiplexId=id&blah=" + i;
            InvokerLocator locatorClient = new InvokerLocator(locatorURI);
            clients[i] = new Client(locatorClient);
            clients[i].connect();
            assertTrue(socketGroupInfo.getClientInvokers().size() == i + 1);
         }
         
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            clients[i].disconnect();
            assertTrue(socketGroupInfo.getClientInvokers().size() == CLIENT_COUNT - i - 1);
         }
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testMultipleClientsDifferentLocatorsServerStarts() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }

   
   public void testMultipleClientsDifferentLocatorsDontWaitOnServer()
   {
      log.info("entering testMultipleClientsDifferentLocatorsDontWaitOnServer()");
      
      try
      {  
         int CLIENT_COUNT = 5;
         Client[] clients = new Client[CLIENT_COUNT];
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            String locatorURI = serverLocatorURI + "/?clientMultiplexId=id&blah=" + i;
            InvokerLocator locatorClient = new InvokerLocator(locatorURI);
            clients[i] = new Client(locatorClient);
            clients[i].connect();
            
            try
            {
               clients[i].invoke(new Integer(i));
               fail();
            }
            catch (IOException e)
            {
               assertTrue(e.getMessage().equals("connection to server has not been made"));
            }
            catch (Throwable e)
            {
               log.error(e);
               fail();
            }
         }
         
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
 
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            clients[i].disconnect();
         }
         
         callbackConnector.stop();
         
         log.info("testMultipleClientsDifferentLocatorsDontWaitOnServer() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testMultipleClientsDifferentLocatorsWaitOnServer()
   {
      log.info("entering testMultipleClientsDifferentLocatorsWaitOnServer()");
      
      try
      {  
         int CLIENT_COUNT = 5;
         Client[] clients = new Client[CLIENT_COUNT];
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            String locatorURI = serverLocatorURI + "/?clientMultiplexId=id&blah=" + i;
            InvokerLocator locatorClient = new InvokerLocator(locatorURI);
            clients[i] = new Client(locatorClient);
            clients[i].connect();
         }
         
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            try
            {
               Integer payload = new Integer(i);
               assertEquals(clients[i].invoke(payload), payload);
            }
            catch (Throwable e)
            {
               log.error(e);
               e.printStackTrace();
               fail();
            }
         }
         
         for (int i = 0; i < CLIENT_COUNT; i++)
         {
            clients[i].disconnect();
         }
         
         callbackConnector.stop();
         
         log.info("testMultipleClientsDifferentLocatorsWaitOnServer() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testMultipleClientsMixedLocators()
   {
      log.info("entering testMultipleClientsMixedLocators()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id";
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertFalse(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         Client[] clients = new Client[5];
         Object o = MultiplexServerInvoker.getSocketGroupMap().get("id");
         MultiplexServerInvoker.SocketGroupInfo socketGroupInfo = (MultiplexServerInvoker.SocketGroupInfo) o;
         
         String basicLocatorURI = serverLocatorURI + "/?clientMultiplexId=id";
         InvokerLocator locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=1");
         clients[0] = new Client(locatorClient);
         clients[0].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 1);
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=2");
         clients[1] = new Client(locatorClient);
         clients[1].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=1");
         clients[2] = new Client(locatorClient);
         clients[2].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=3");
         clients[3] = new Client(locatorClient);
         clients[3].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 3);
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=1");
         clients[4] = new Client(locatorClient);
         clients[4].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 3);
         
         clients[0].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 3);
         
         clients[1].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         
         clients[2].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         
         clients[3].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 1);
         
         clients[4].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 0);

         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testMultipleClientsMixedLocators() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   public void testMultipleClientsMixedLocatorsNoServer()
   {
      log.info("entering testMultipleClientsMixedLocatorsNoServer()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         Client[] clients = new Client[5];
         
         String basicLocatorURI = serverLocatorURI;
         basicLocatorURI += "/?clientMultiplexId=id&muiltiplexBindHost=localhost&multiplexBindPort=" + freePort;
         InvokerLocator locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=1");
         clients[0] = new Client(locatorClient);
         clients[0].connect();
         Object o = MultiplexServerInvoker.getSocketGroupMap().get("id");
         MultiplexServerInvoker.SocketGroupInfo socketGroupInfo = (MultiplexServerInvoker.SocketGroupInfo) o;
         assertTrue(socketGroupInfo.getClientInvokers().size() == 1);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=2");
         clients[1] = new Client(locatorClient);
         clients[1].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=1");
         clients[2] = new Client(locatorClient);
         clients[2].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=3");
         clients[3] = new Client(locatorClient);
         clients[3].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 3);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         locatorClient = new InvokerLocator(basicLocatorURI + "&invoker=1");
         clients[4] = new Client(locatorClient);
         clients[4].connect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 3);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         clients[0].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 3);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         clients[1].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         clients[2].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 2);
         System.out.println(socketGroupInfo.getClientInvokers());

         clients[3].disconnect();
         assertTrue(socketGroupInfo.getClientInvokers().size() == 1);
         System.out.println(socketGroupInfo.getClientInvokers());
         
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
                  
         clients[4].disconnect();
         log.info("client invokers: " + socketGroupInfo.getClientInvokers().size());
         System.out.println("client invokers: " + socketGroupInfo.getClientInvokers().size());
         System.out.println(socketGroupInfo.getClientInvokers());
         assertTrue(socketGroupInfo.getClientInvokers().size() == 0);
         
         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));  
       
         log.info("testMultipleClientsMixedLocatorsNoServer() PASSES");
         
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }

   
   public void testServerNoClients()
   {
      log.info("entering testServerNoClients()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector = new Connector(callbackURI);
         callbackConnector.start();

         assertTrue(callbackConnector.isStarted());
         assertTrue(callbackConnector.getServerInvoker().isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         callbackConnector.stop();

         assertFalse(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         assertFalse(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         log.info("testServerNoClients() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
 
   
   public void testMultipleServers()
   {
      log.info("entering testMultipleServers()");
      
      try
      {
         int freePort = PortUtil.findFreePort("localhost");
         String callbackURI = "multiplex://localhost:" + freePort;
         callbackURI += "/?serverMultiplexId=id&multiplexConnectHost=localhost&multiplexConnectPort=" + connectPort;
         Connector callbackConnector1 = new Connector(callbackURI);
         
         callbackConnector1.start();
         assertTrue(callbackConnector1.isStarted());
         assertTrue(MultiplexServerInvoker.getSocketGroupMap().containsKey("id"));
         AddressPair addressPair = new AddressPair("localhost", connectPort, "localhost", freePort);
         assertTrue(MultiplexServerInvoker.getAddressPairMap().containsKey(addressPair));
         
         try
         {
            Connector callbackConnector2  = new Connector(callbackURI + "&blah");
            callbackConnector2.start();
            fail();
         }
         catch (Exception e)
         {
            assertTrue(e.getCause().getMessage().startsWith("Second MultiplexServerInvoker attempting to join invoker group: "));
         }
         
         callbackConnector1.stop();
         
         log.info("testMultipleServers() PASSES");
         OKCounter++;
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
      }
      finally
      {
         verifySocketGroupInfoEmpty();
      }
   }
   
   
   protected boolean verifySocketGroupInfoEmpty()
   {
      for (int i = 0; i < 5; i++)
      {
         if (MultiplexServerInvoker.getSocketGroupMap().isEmpty())
            break;
      }
      
      if (MultiplexServerInvoker.getSocketGroupMap().isEmpty())
         return true;
      
      MultiplexServerInvoker.getSocketGroupMap().clear();
      return false;
   }
   
   
   class SimpleServerInvocationHandler implements ServerInvocationHandler
   {
      InvokerCallbackHandler handler;
      
      public void addListener(InvokerCallbackHandler callbackHandler)
      {
      }

      public Object invoke(InvocationRequest invocation) throws Throwable
      {
         return invocation.getParameter();
      }

      public void removeListener(InvokerCallbackHandler callbackHandler)
      {
      }

      public void setInvoker(ServerInvoker invoker)
      {
      }

      public void setMBeanServer(MBeanServer server)
      { 
      }
   }
}
