/* Integrity.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.other;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.grinvin.graphs.Edge;
import org.grinvin.graphs.GraphBundleView;
import org.grinvin.graphs.GraphView;
import org.grinvin.graphs.Vertex;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * Computes the integrity of a graph
 */
public class Integrity extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        int nrOfVertices = bundle.getGraph().getNumberOfVertices();
        int nrOfEdges = bundle.getGraph().getNumberOfEdges();
        int[][] adjlist = bundle.adjacencyList();
        if(nrOfVertices==0)
            return new IntegerValue(0, this);
        
        //for(int i = 0; i<adjlist.length; i++)
        //    System.out.println(Arrays.toString(adjlist[i]));
        
        int best = nrOfVertices < nrOfEdges + 1 ? nrOfVertices : nrOfEdges + 1;
        //upperbounds:
        //|V|
        //|E|+1 (because vertex cover <= |E|)
        //vertex cover + 1
        
        int approxVertexCover = approximateVertexCover(bundle.getGraph());
        if(approxVertexCover + 1 < best)
            best = approxVertexCover + 1;
        
        int lowerBound = 0;
        //lowerbounds:
        //min.degree + 1
        //cliquenumber
        //min_{t=1,...,p}(max(t,d_t+1))
        //chromatic number
        
        int[] degrees = sortDegrees(adjlist);
        
        if(degrees[nrOfVertices-1] == nrOfVertices-1)
            //complete graph
            return new IntegerValue(nrOfVertices, this);

        int tempMin = nrOfVertices;
        for(int i = 0; i<nrOfVertices; i++){
            int max = i + 1 < degrees[i]+1 ? degrees[i]+1 : i+1;
            if(tempMin>max)
                tempMin = max;
        }
            
        if(lowerBound < tempMin)
            lowerBound = tempMin;
        
        //System.out.println("First lowerbound: " + lowerBound);
        
        if(lowerBound < degrees[nrOfVertices-1])
            lowerBound = degrees[nrOfVertices-1];
        
        //System.out.println("Second lowerbound: " + lowerBound);
        
        
        boolean[] removedSet = new boolean[nrOfVertices];
        Arrays.fill(removedSet, false);
        
        for(int i = 0; i<nrOfVertices; i++){
            removedSet[i]=true;
            best = integrity(best, 1, lowerBound, removedSet, i, adjlist);
            removedSet[i]=false;
        }

        return new IntegerValue(best, this);
    }
    
    public int integrity(int best, int orderOfSet, int lowerBound, boolean[] removedSet, int lastAddedToSet, int[][] adjacencyList){
        int largestComponent = largestComponentOrder(adjacencyList, removedSet, orderOfSet);
        //System.out.println(Arrays.toString(removedSet));
        //System.out.println("Largest component: " + largestComponent);
        if(best>orderOfSet + largestComponent)
            best = orderOfSet + largestComponent;
        //System.out.println("Best: " + best);
        
        if(orderOfSet >= best - 1)
            return best;
        if(best == lowerBound)
            return best;
        
        for(int i = lastAddedToSet + 1; i<adjacencyList.length; i++){
            removedSet[i]=true;
            best = integrity(best, orderOfSet + 1, lowerBound, removedSet, i, adjacencyList);
            removedSet[i]=false;
        }
        
        return best;
    }
      
    /**
     * returns the order of the largest component in the graph
     */
    private int largestComponentOrder(int[][] adjacencyList, boolean[] removedSet, int orderOfSet){
        boolean[] vertices = new boolean[adjacencyList.length];
        System.arraycopy(removedSet, 0, vertices, 0, vertices.length);
        int maximum = 0;
        int currentCounted = 0;
        for(int vertex = 0; vertex < adjacencyList.length; vertex++)
            if(!vertices[vertex]){
                vertices[vertex]=true;
                int componentSize = checkComponent(adjacencyList, vertices, vertex, 1);
                currentCounted += componentSize;
                if(maximum < componentSize)
                    maximum = componentSize;
                if(maximum>vertices.length-currentCounted-orderOfSet)
                    return maximum;
            }        
        return maximum;
    }
    
    /**
     * check a component and calculate it's size during.
     */
    private int checkComponent(int[][] adjacencyList, boolean[] vertices, int currentVertex, int componentSize) {
        for(int i=0; i < adjacencyList[currentVertex].length; i++){
            if(!vertices[adjacencyList[currentVertex][i]]){
                if(!vertices[adjacencyList[currentVertex][i]]){
                    vertices[adjacencyList[currentVertex][i]]=true;
                    componentSize = checkComponent(adjacencyList, vertices, adjacencyList[currentVertex][i], componentSize + 1);
                }
            }
        }
        return componentSize;
    }
    
    /**
     * sort degrees
     */
    private int[] sortDegrees(int[][] adjlist) {
        int[] degrees = new int[adjlist.length];
        for(int i=0; i < adjlist.length; i++)
            degrees[i] = adjlist[i].length;
        
        for(int i=0; i < degrees.length; i++)
            for(int j=degrees.length-1; j>i; j--)
                if(adjlist[degrees[j]].length > adjlist[degrees[j-1]].length){
                    degrees[j] = degrees[j] + degrees[j-1];
                    degrees[j-1] = degrees[j] - degrees[j-1];
                    degrees[j] = degrees[j] - degrees[j-1];
                }
        return degrees;
    }
    
    private int approximateVertexCover(GraphView graph){
        List<Edge> edges = new ArrayList<Edge>();
        for(Edge e : graph.edges())
            edges.add(e);
        int coverSize = 0;
        while(!edges.isEmpty()){
            Edge e = edges.get(0);
            coverSize += 2;
            Vertex v1 = e.getFirstEndpoint();
            Vertex v2 = e.getSecondEndpoint();
            for(int i = 0; i<edges.size(); i++)
                if(edges.get(i).getFirstEndpoint().equals(v1) || edges.get(i).getFirstEndpoint().equals(v1))
                    edges.remove(i--);
        }        
        return coverSize;
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.Integrity";
    }
    
}
