/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: cursor.c,v 1.4 94/07/17 15:39:29 nau Exp $";

/* cursor stuff
 */

#include "global.h"

#include "cursor.h"
#include "data.h"
#include "draw.h"
#include "error.h"

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
#define	SAVE_STACK_DEPTH	10			/* deep enough for what we use it for */

static	Boolean	SaveStack[SAVE_STACK_DEPTH];/* used by HideCursor() and RestoreCursor() */
static	int		StackPosition = 0;

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	DrawCursor(void);
static	void	DrawMarkedLine(void);
static	void	DrawMarkedElement(void);
static	void	DrawMarkedBlock(void);
static	void	DrawLineStack(void);
static	void	DrawCursorElements(void);
static	void	ClipCursor(Position, Position);

/* ---------------------------------------------------------------------------
 * draws the cursor 
 * don't transform MAX_COORD to screen coordinats, it is
 * already the maximum of screen- and pcb-coordinates
 */
static void DrawCursor(void)
{
	XDrawLine(Dpy, Output.OutputWindow, MyCursor.GC,
		TO_SCREEN(MyCursor.X), MIN_COORD, TO_SCREEN(MyCursor.X), MAX_COORD);
	XDrawLine(Dpy, Output.OutputWindow, MyCursor.GC,
		MIN_COORD, TO_SCREEN(MyCursor.Y), MAX_COORD, TO_SCREEN(MyCursor.Y));
}

/* ---------------------------------------------------------------------------
 * draws the marked line 
 */
static void DrawMarkedLine(void)
{
	XDrawLine(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
		TO_SCREEN(MyCursor.MarkedLine.X1), TO_SCREEN(MyCursor.MarkedLine.Y1),
		TO_SCREEN(MyCursor.MarkedLine.X2), TO_SCREEN(MyCursor.MarkedLine.Y2));
}

/* ---------------------------------------------------------------------------
 * draws the lines from a loaded element which is to be merged in
 */
static void DrawMarkedElement(void)
{
	ElementTypePtr	element = &MyCursor.MarkedElement.Element;
	LineTypePtr		line = element->Line;
	ArcTypePtr		arc = element->Arc;
	MarkTypePtr		mark = element->Mark;
	Cardinal		n;

	for (n = element->LineN; n; n--, line++)
		XDrawLine(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
			TO_SCREEN(MyCursor.X +line->X1), TO_SCREEN(MyCursor.Y +line->Y1),
			TO_SCREEN(MyCursor.X +line->X2), TO_SCREEN(MyCursor.Y +line->Y2));

		/* arc coordinates and angles have to be converted to X11 notation */
	for (n = element->ArcN; n; n--, arc++)
		XDrawArc(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
			TO_SCREEN(MyCursor.X +arc->X -arc->Width),
			TO_SCREEN(MyCursor.Y +arc->Y -arc->Height),
			TO_SCREEN(2*arc->Width), TO_SCREEN(2*arc->Height),
			(arc->StartAngle +180) *64, arc->Delta *64);

	for (n = element->MarkN; n; n--, mark++)
	{
		XDrawLine(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
			TO_SCREEN(MyCursor.X +mark->X -MARK_SIZE/2), TO_SCREEN(MyCursor.Y +mark->Y),
			TO_SCREEN(MyCursor.X +mark->X +MARK_SIZE/2), TO_SCREEN(MyCursor.Y +mark->Y));
		XDrawLine(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
			TO_SCREEN(MyCursor.X +mark->X), TO_SCREEN(MyCursor.Y +mark->Y -MARK_SIZE/2),
			TO_SCREEN(MyCursor.X +mark->X), TO_SCREEN(MyCursor.Y +mark->Y +MARK_SIZE/2));
	}
}

/* ---------------------------------------------------------------------------
 * draws all objects in on linestack
 */
static void DrawLineStack(void)
{
	LineTypePtr	line = MyCursor.LineStack.Line;
	Cardinal	n = MyCursor.LineStack.LineN;

	for (; n; n--, line++)
		XDrawLine(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
			TO_SCREEN(MyCursor.X +line->X1 -MyCursor.LineStack.OriginX),
			TO_SCREEN(MyCursor.Y +line->Y1 -MyCursor.LineStack.OriginY),
			TO_SCREEN(MyCursor.X +line->X2 -MyCursor.LineStack.OriginX),
			TO_SCREEN(MyCursor.Y +line->Y2 -MyCursor.LineStack.OriginY));
}

/* ---------------------------------------------------------------------------
 * draws the marked block in a special way
 */
static void DrawMarkedBlock(void)
{
	Position    x1, y1,     /* upper left corner */
				x2, y2;     /* lower right corner */

	x1 = MIN(MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.X2);
	y1 = MIN(MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.Y2);
	x2 = MAX(MyCursor.MarkedBlock.X1, MyCursor.MarkedBlock.X2);
	y2 = MAX(MyCursor.MarkedBlock.Y1, MyCursor.MarkedBlock.Y2);
	XDrawRectangle(Dpy, Output.OutputWindow, MyCursor.MarkedGC,
		TO_SCREEN(x1), TO_SCREEN(y1), TO_SCREEN(x2-x1), TO_SCREEN(y2-y1));
}

/* ---------------------------------------------------------------------------
 * draws additional stuff that follows the cursor
 */
static void DrawCursorElements(void)
{
	DrawCursor();
	switch (MyCursor.Mode)
	{
		case MODE_LINE:
				/* draw only if starting point is set */
			if (MyCursor.MarkedLine.State != MARKED_STATE_RESET)
				DrawMarkedLine();
			break;

		case MODE_BLOCK:
			if (MyCursor.MarkedBlock.State != MARKED_STATE_RESET)
				DrawMarkedBlock();
			break;

		case MODE_ELEMENT:
			DrawMarkedElement();
			break;

		case MODE_LINESTACK:
			DrawLineStack();
			break;
	}
}

/* ---------------------------------------------------------------------------
 * switches cursor on
 */
void CursorOn(void)
{
	if (!MyCursor.On)
	{
		MyCursor.On = True;
		DrawCursorElements();
	}
}

/* ---------------------------------------------------------------------------
 * switches cursor off
 */
void CursorOff(void)
{
	if (MyCursor.On)
	{
		MyCursor.On = False;
		DrawCursorElements();
	}
}

/* ---------------------------------------------------------------------------
 * saves cursor state (on/off) and hides him
 */
void HideCursor(void)
{
	SaveStack[StackPosition++] = MyCursor.On;
	if (StackPosition >= SAVE_STACK_DEPTH)
		StackPosition--;
	CursorOff();
}

/* ---------------------------------------------------------------------------
 * restores last cursor state
 */
void RestoreCursor(void)
{
	if (StackPosition)
	{
		if (SaveStack[--StackPosition])
			CursorOn();
		else
			CursorOff();
	}
}

/* ---------------------------------------------------------------------------
 * clips cursor position, sets the new position to x,y or restores the old one
 */
static void ClipCursor(Position x, Position y)
{
	Position	x1, y1, x2, y2;
	
		/* get PCB coordinates from screen */
	x1 = TO_PCB(Output.OffsetX);
	y1 =  TO_PCB(Output.OffsetY);
	x2 = TO_PCB(Output.OffsetX +Output.Width -1);
	y2 = TO_PCB(Output.OffsetY +Output.Height -1);

		/* set new position if possible */
	MyCursor.X = (x < x1 || x > x2) ? MyCursor.X : x;
	MyCursor.Y = (y < y1 || y > y2) ? MyCursor.Y : y;
	
		/* check if new position is inside the output window
		 * This might not be true after the window has been resized.
		 * In this case we just set it to the center of the window or
		 * with respect to the grid (if possible)
		 */
	if (MyCursor.X < x1 || MyCursor.X > x2)
	{
		if (x2 -x1 +1 >= PCB->Grid)
				/* there must be a point that matches the grid 
				 * so we just have to look for it with some integer
				 * calculations
				 */
			MyCursor.X = ((x1 +PCB->Grid -PCB->GridOffsetX) /PCB->Grid) *PCB->Grid +PCB->GridOffsetX;
		else
			MyCursor.X = (x1+x2)/2;
	}
	else
			/* check if the new position matches the grid */
		MyCursor.X = ((MyCursor.X -PCB->GridOffsetX) /PCB->Grid) *PCB->Grid +PCB->GridOffsetX;

	if (MyCursor.Y < y1 || MyCursor.Y > y2)
	{
		if (y2 -y1 +1 >= PCB->Grid)
			MyCursor.Y = ((y1 +PCB->Grid -PCB->GridOffsetY) /PCB->Grid) *PCB->Grid +PCB->GridOffsetY;
		else
			MyCursor.Y = (y1+y2)/2;
	}
	else
		MyCursor.Y = ((MyCursor.Y -PCB->GridOffsetY) /PCB->Grid) *PCB->Grid +PCB->GridOffsetY;
}

/* ---------------------------------------------------------------------------
 * move cursor relative (has to be switched off)
 */
void MoveCursorRelative(Position DeltaX, Position DeltaY)
{
	ClipCursor(MyCursor.X +DeltaX, MyCursor.Y +DeltaY);
}

/* ---------------------------------------------------------------------------
 * move cursor absolute (has to be switched off)
 */
void MoveCursorAbsolute(Position x, Position y)
{
	ClipCursor(x, y);
}

/* ---------------------------------------------------------------------------
 * initializes cursor stuff
 */
void InitCursor(void)
{
	static	char	markeddashes[] = {1, 1};

		/* clear struct */
	memset(&MyCursor, 0, sizeof(CursorType));

	if (!(MyCursor.GC = XCreateGC(Dpy, Output.OutputWindow, 0, NULL)) ||
	    !(MyCursor.MarkedGC = XCreateGC(Dpy, Output.OutputWindow, 0, NULL)))
		MyFatal("can't create default cursor GC\n");

	XSetState(Dpy, MyCursor.GC, Settings.CursorColor, Settings.bgColor,
		GXxor, AllPlanes);

		/* change dashes for drawing of marked stuff */
	XCopyGC(Dpy, MyCursor.GC, -1, MyCursor.MarkedGC);
	XSetLineAttributes(Dpy, MyCursor.MarkedGC, 0, LineOnOffDash, CapButt, JoinMiter);
	XSetDashes(Dpy, MyCursor.MarkedGC, 0, markeddashes, 2);
}

/* ---------------------------------------------------------------------------
 * exits cursor routines, release memory
 */
void ExitCursor(void)
{
	CursorOff();
	XFreeGC(Dpy, MyCursor.GC);
	XFreeGC(Dpy, MyCursor.MarkedGC);
}

