/* 
**  mod_text2html.c -- Apache sample text2html module
**  Brian Aker
**  brian@tangent.org
*/ 

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_log.h"
#include "ap_config.h"
#include <sys/mman.h>

#define WATCHPOINT printf("WATCHPOINT %s %d\n", __FILE__, __LINE__);

/* This character list comes from Ian F. Darwin file application */
#define F 0   /* character never appears in text */
#define T 1   /* character appears in plain ASCII text */
#define I 2   /* character appears in ISO-8859 text */
#define X 3   /* character appears in non-ISO extended ASCII (Mac, IBM PC) */

static char text_chars[256] = {
	/*                  BEL BS HT LF    FF CR    */
	F, F, F, F, F, F, F, T, T, T, T, F, T, T, F, F,  /* 0x0X */
        /*                              ESC          */
	F, F, F, F, F, F, F, F, F, F, F, T, F, F, F, F,  /* 0x1X */
	T, T, T, T, T, T, T, T, T, T, T, T, T, T, T, T,  /* 0x2X */
	T, T, T, T, T, T, T, T, T, T, T, T, T, T, T, T,  /* 0x3X */
	T, T, T, T, T, T, T, T, T, T, T, T, T, T, T, T,  /* 0x4X */
	T, T, T, T, T, T, T, T, T, T, T, T, T, T, T, T,  /* 0x5X */
	T, T, T, T, T, T, T, T, T, T, T, T, T, T, T, T,  /* 0x6X */
	T, T, T, T, T, T, T, T, T, T, T, T, T, T, T, F,  /* 0x7X */
	/*            NEL                            */
	X, X, X, X, X, T, X, X, X, X, X, X, X, X, X, X,  /* 0x8X */
	X, X, X, X, X, X, X, X, X, X, X, X, X, X, X, X,  /* 0x9X */
	I, I, I, I, I, I, I, I, I, I, I, I, I, I, I, I,  /* 0xaX */
	I, I, I, I, I, I, I, I, I, I, I, I, I, I, I, I,  /* 0xbX */
	I, I, I, I, I, I, I, I, I, I, I, I, I, I, I, I,  /* 0xcX */
	I, I, I, I, I, I, I, I, I, I, I, I, I, I, I, I,  /* 0xdX */
	I, I, I, I, I, I, I, I, I, I, I, I, I, I, I, I,  /* 0xeX */
	I, I, I, I, I, I, I, I, I, I, I, I, I, I, I, I   /* 0xfX */
};

static int looks_printable(char const *buf, off_t size) {
	int x;

	for (x = 0; x < size; x++) {
		if (text_chars[buf[x]] != T)
			return 0;
	}

	return 1;
}

/* The sample content handler */
static int text2html_handler(request_rec *r) {
	int fd = -1;
	int x = 0;
	int k = 0;
	int end = 0;
	caddr_t text = NULL;
	caddr_t sub = NULL;

	if (r->header_only) {
		ap_send_http_header(r);

		return OK;
	}

	/* Lets look like Apache's core handler */
	if (r->finfo.st_mode == 0 || (r->path_info && *r->path_info)) {
			ap_log_rerror(APLOG_MARK, APLOG_ERR|APLOG_NOERRNO, r,
			"File does not exist: %s",r->path_info ?
			ap_pstrcat(r->pool, r->filename, r->path_info, NULL)
			: r->filename);
		return HTTP_NOT_FOUND;
	}
	if (r->method_number != M_GET) {
		return METHOD_NOT_ALLOWED;
	}

	if ((fd = ap_popenf(r->pool, r->filename,O_RDONLY,S_IRWXU)) < 0) {
		ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, r,
				"mod_text2html: couldn't open a file descriptor for : %s", r->filename);
		return FORBIDDEN;
	}
	ap_update_mtime(r, r->finfo.st_mtime);
	ap_set_last_modified(r);
	ap_set_etag(r);

	text = mmap(NULL,  r->finfo.st_size, PROT_READ, MAP_PRIVATE,fd, 0);

	if(looks_printable(text, r->finfo.st_size)) {
		r->content_type = "text/html";      
		ap_send_http_header(r);

		ap_rputs("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">", r);
		ap_rputs("<HTML>\n\t<HEAD>\n", r);
		ap_rprintf(r, "\t\t<TITLE>%s</TITLE>\n", r->uri);
		ap_rputs("\t<HEAD>\n", r);
		ap_rputs("\t<BODY>\n", r);

		for(x=0; x <  r->finfo.st_size; x++) {
			if( ((x + 7) < r->finfo.st_size) && (text[x] == 'h' || text[x] == 'H')) {
				if(!strncasecmp(&text[x], "http://", 7)) {
					sub = text + x;
					end = 7;
					if(x > 1 && text[x-1] == '"') {
						for(; end < (r->finfo.st_size - x); end++) {
							if(isspace(sub[end])) break;
							if(sub[end] == '"') break;
						}
							/* The below is for catching bad html */
					} else if(x > 1 && text[x-1] == '=') {
						for(; end < (r->finfo.st_size - x); end++) {
							if(isspace(sub[end])) break;
							if(sub[end] == '>') break;
						}
					} else {
						while(!(isspace(sub[end]))) end++;
					}
					ap_rputs("<A HREF=\"", r);
					for(k = 0; k < end; k++) {
						ap_rputc(text[x + k], r);
					}
					ap_rputs("\">", r);
					for(k = 0; k < end; k++) {
						ap_rputc(text[x + k], r);
					}
					ap_rputs("</A>", r);
					x += end;
				} else {
					ap_rputc(text[x], r);
				}
			} else if(text[x] == '\n') {
				ap_rputs("<BR>", r);
				ap_rputc(text[x], r);
			} else if(text[x] == '>') {
				ap_rputs("&gt;", r);
			} else if(text[x] == '<') {
				ap_rputs("&lt;", r);
			} else if(text[x] == '&') {
				ap_rputs("&amp;", r);
			} else {
				ap_rputc(text[x], r);
			}
		}
		ap_rputs("\n\t</BODY>\n</HTML>\n", r);
	} else {
		r->content_type = "application/octet-stream";      
		ap_send_http_header(r);
		ap_send_mmap(text, r, 0, r->finfo.st_size);
	}
	munmap(text, r->finfo.st_size);

	return OK;
}

/* Dispatch list of content handlers */
static const handler_rec text2html_handlers[] = { 
	{ "text/plain", text2html_handler }, 
	{ NULL, NULL }
};

/* Dispatch list for API hooks */
module MODULE_VAR_EXPORT text2html_module = {
	STANDARD_MODULE_STUFF, 
	NULL,                  /* module initializer                  */
	NULL,                  /* create per-dir    config structures */
	NULL,                  /* merge  per-dir    config structures */
	NULL,                  /* create per-server config structures */
	NULL,                  /* merge  per-server config structures */
	NULL,                  /* table of config file commands       */
	text2html_handlers,    /* [#8] MIME-typed-dispatched handlers */
	NULL,                  /* [#1] URI to filename translation    */
	NULL,                  /* [#4] validate user id from request  */
	NULL,                  /* [#5] check if the user is ok _here_ */
	NULL,                  /* [#3] check access by host address   */
	NULL,                  /* [#6] determine MIME type            */
	NULL,                  /* [#7] pre-run fixups                 */
	NULL,                  /* [#9] log a transaction              */
	NULL,                  /* [#2] header parser                  */
	NULL,                  /* child_init                          */
	NULL,                  /* child_exit                          */
	NULL                   /* [#0] post read-request              */
};

