// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "tmatrix.h"
#include <cmath>
#ifdef VISUAL_ALLOW_PYTHON_PRIMITIVES
#include <boost/python.hpp>
#endif

#include GL_INCLUDE

namespace visual {

	
void 
frustum( tmatrix& T, tmatrix& I, double l, double r, double b, double t, double n, double f ) throw()
{
	T.x_column(2*n/(r-l),0,0);
	T.y_column(0,2*n/(t-b),0);
	T.z_column((r+l)/(r-l),(t+b)/(t-b),(f+n)/(n-f));
	T.w_column(0,0,2*f*n/(n-f));
	T.w_row(0,0,-1,0);

	I.x_column((r-l)/(2*n), 0, 0);
	I.y_column(0,(t-b)/(2*n),0);
	I.z_column(0,0,0);
	I.w_column((r+l)/(2*n),(t+b)/(2*n),-1);
	I.w_row(0,0,(n-f)/(2*f*n),(f+n)/(2*f*n));
}

void 
rotation( tmatrix& T, double angle, const vector& a) throw()
{
	double c = std::cos(angle), s = std::sin(angle);
	double ic = 1.0-c;
	double icxx = ic*a.x*a.x;
	double icxy = ic*a.x*a.y;
	double icxz = ic*a.x*a.z;
	double icyy = ic*a.y*a.y;
	double icyz = ic*a.y*a.z;
	double iczz = ic*a.z*a.z;

	T.x_column( icxx  +     c, icxy + a.z*s, icxz - a.y*s );
	T.y_column( icxy  - a.z*s, icyy +     c, icyz + a.x*s );
	T.z_column( icxz  + a.y*s, icyz - a.x*s, iczz +     c );
	T.w_column();
	T.w_row();
}

double 
norm_dot( const vector& a, const vector& b) throw()
{
	double m = std::sqrt( a.mag2() * b.mag2() );
	return (a.x*b.x + a.y*b.y + a.z*b.z) / m;
}

void
py_rotation( tmatrix& R, double angle, vector axis, vector origin) throw()
{
	rotation( R, angle, axis.norm());
	
	vector T = origin - R * origin;
	R.w_column( T.x, T.y, T.z);	
}

void 
tmatrix::concat( const tmatrix& A, const tmatrix& B) throw()
{
	// m.concat(a,b) -> m*v == b*(a*v)
	for( int c=0; c<4; ++c) {
		M[0][c] = B[0][0]*A[0][c] + B[0][1]*A[1][c] + B[0][2]*A[2][c] + B[0][3]*A[3][c];
		M[1][c] = B[1][0]*A[0][c] + B[1][1]*A[1][c] + B[1][2]*A[2][c] + B[1][3]*A[3][c];
		M[2][c] = B[2][0]*A[0][c] + B[2][1]*A[1][c] + B[2][2]*A[2][c] + B[2][3]*A[3][c];
		M[3][c] = B[3][0]*A[0][c] + B[3][1]*A[1][c] + B[3][2]*A[2][c] + B[3][3]*A[3][c];
	}
}

void 
tmatrix::invert_ortho(const tmatrix& A) throw()
{
	// Precondition: w = Mv = Rv + t  (R orthogonal)
	// Therefore: (M^-1)w = v = (R^T)Rv 
	//                        = (R^T)(Rv+t - t) 
	//                        = (R^T)(w-t) 
	//                        = (R^T)w - (R^T)t

	x_column(A[0][0], A[0][1], A[0][2]);  // row 0
	y_column(A[1][0], A[1][1], A[1][2]);  // row 1
	z_column(A[2][0], A[2][1], A[2][2]);  // row 2
	w_column(-A[0][0]*A[0][3] - A[1][0]*A[1][3] - A[2][0]*A[2][3]
			 , -A[0][1]*A[0][3] - A[1][1]*A[1][3] - A[2][1]*A[2][3]
			 , -A[0][2]*A[0][3] - A[1][2]*A[1][3] - A[2][2]*A[2][3]
			 );
	w_row();
}

vector 
tmatrix::times_inv( const vector& v, double w) const throw()
{
	double x = v.x - M[0][3]*w;
	double y = v.y - M[1][3]*w;
	double z = v.z - M[2][3]*w;
	return vector(
		M[0][0]*x + M[1][0]*y + M[2][0]*z,
		M[0][1]*x + M[1][1]*y + M[2][1]*z,
		M[0][2]*x + M[1][2]*y + M[2][2]*z
		);
}

vector 
tmatrix::times_v( const vector& v) const throw()
{
	return vector(
		M[0][0]*v.x + M[0][1]*v.y + M[0][2]*v.z,
		M[1][0]*v.x + M[1][1]*v.y + M[1][2]*v.z,
		M[2][0]*v.x + M[2][1]*v.y + M[2][2]*v.z
		);
}

vector 
tmatrix::operator*( const vector& v) const throw()
{
	return vector(
		M[0][0]*v.x + M[0][1]*v.y + M[0][2]*v.z + M[0][3],
		M[1][0]*v.x + M[1][1]*v.y + M[1][2]*v.z + M[1][3],
		M[2][0]*v.x + M[2][1]*v.y + M[2][2]*v.z + M[2][3]
		);
}

void
tmatrix::project( const vector v, vertex& o) const throw()
{
	o.x = M[0][0]*v.x + M[0][1]*v.y + M[0][2]*v.z + M[0][3];
	o.y = M[1][0]*v.x + M[1][1]*v.y + M[1][2]*v.z + M[1][3];
	o.z = M[2][0]*v.x + M[2][1]*v.y + M[2][2]*v.z + M[2][3];
	o.w = M[3][0]*v.x + M[3][1]*v.y + M[3][2]*v.z + M[3][3];
}

void
tmatrix::gl_load(void)
{
	// Translate from row-major to column-major format.
	double tmp[] = { 
		M[0][0], M[1][0], M[2][0], M[3][0], 
		M[0][1], M[1][1], M[2][1], M[3][1],
		M[0][2], M[1][2], M[2][2], M[3][2],
		M[0][3], M[1][3], M[2][3], M[3][3],
		};
	glLoadMatrixd(tmp);	
}
	
void 
tmatrix_init_type()
{
#ifdef VISUAL_ALLOW_PYTHON_PRIMITIVES
// There isn't any point in exporting this class until we have more infrastructure
//   in place for producing new primitives in python.
	using namespace boost::python;

	class_<vertex>( "vertex", init<double, double, double, double>())
		.def_readwrite( "x", &vertex::x)
		.def_readwrite( "y", &vertex::y)
		.def_readwrite( "z", &vertex::z)
		.def_readwrite( "w", &vertex::w)
		;

	void (tmatrix::*x_column_vector)( const vector&) = &tmatrix::x_column;
	void (tmatrix::*y_column_vector)( const vector&) = &tmatrix::y_column;
	void (tmatrix::*z_column_vector)( const vector&) = &tmatrix::z_column;
	void (tmatrix::*w_column_vector)( const vector&) = &tmatrix::w_column;
	void (tmatrix::*w_row_doubles)( double, double, double, double) = &tmatrix::w_row;
	
	class_<tmatrix>( "tmatrix", init<>())
		.def( init<tmatrix>())
		.def( init<tmatrix, tmatrix>())
		.def( "project", &tmatrix::project)
		.def( "identity", &tmatrix::identity, "Returns the identity matrix")
		.staticmethod( "identity")
		.def( "x_column", x_column_vector, "Set the x column to the specified vector")
		.def( "y_column", y_column_vector, "Set the y column to the specified vector")
		.def( "z_column", z_column_vector, "Set the z column to the specifed vector")
		.def( "w_column", w_column_vector, "Set the w column to the specified vector")
		.def( "w_row", w_row_doubles, "Set the w row to the specified four values")
		.def( "concat", &tmatrix::concat, "Concatenate two tmatrixes")
		.def( "invert_ortho", &tmatrix::invert_ortho)
		.def( "times_inv", &tmatrix::times_inv)
		.def( "times_v", &tmatrix::times_v, "multiplication by a vector [x y z 0]")
		.def( self * other<visual::vector>()) // "multiplication by a point [x y z 1]"
		.def( "x", &tmatrix::x)
		.def( "y", &tmatrix::y)
		.def( "z", &tmatrix::z)
		.def( "w", &tmatrix::w)
		;
#endif // !VISUAL_ALLOW_PYTHON_PRIMITIVES
}

} // !namespace visual
