/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.remote.soap;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Map;
import java.util.HashMap;

import javax.management.MBeanServer;
import javax.management.remote.JMXServiceURL;
import javax.management.remote.JMXConnectorServerFactory;

import mx4j.remote.ConnectionResolver;
import mx4j.tools.remote.AbstractJMXConnectorServer;
import mx4j.tools.remote.soap.web.axis.SimpleAxisWebContainer;
import mx4j.tools.remote.soap.web.WebContainer;
import mx4j.tools.remote.soap.web.jetty.JettyWebContainer;

/**
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.11 $
 */
public class SOAPConnectorServer extends AbstractJMXConnectorServer
{
   public static final String USE_EXTERNAL_WEB_CONTAINER = "jmx.remote.x.soap.use.external.web.container";
   public static final String EMBEDDED_WEB_CONTAINER_CLASS = "jmx.remote.x.soap.embedded.web.container.class";
   public static final String AXIS_DEPLOY_URL = "jmx.remote.x.axis.deploy.url";
   // TODO: maybe worth to make this private ?
   public static final String DEFAULT_AXIS_DEPLOY_URL = "/axis/services/AdminService";
   private static Map instances = new HashMap();

   public static SOAPConnectionManager find(JMXServiceURL address)
   {
      synchronized (SOAPConnectorServer.class)
      {
         return (SOAPConnectionManager)instances.get(address);
      }
   }

   private MBeanServer mbeanServer;
   private WebContainer webContainer;
   private SOAPConnectionManager connectionManager;

   public SOAPConnectorServer(JMXServiceURL url, Map environment, MBeanServer server)
   {
      super(url, environment, server);
   }

   public MBeanServer getMBeanServer()
   {
      return mbeanServer;
   }

   protected void doStart() throws IOException, IllegalStateException
   {
      JMXServiceURL address = getAddress();
      String protocol = address.getProtocol();
      Map environment = getEnvironment();
      ConnectionResolver resolver = ConnectionResolver.newConnectionResolver(protocol, environment);
      if (resolver == null) throw new MalformedURLException("Unsupported protocol: " + protocol);

      MBeanServer realServer = null;
      MBeanServer server = super.getMBeanServer();

      MBeanServer resolvedServer = (MBeanServer)resolver.createServer(address, environment);
      if (resolvedServer == null)
      {
         if (server == null) throw new IllegalStateException("This SOAPConnectorServer is not attached to an MBeanServer");
         realServer = server;
      }
      else
      {
         if (server == null)
         {
            realServer = resolvedServer;
         }
         else
         {
            if (server != resolvedServer) throw new IllegalStateException("This SOAPConnectorServer cannot be attached to 2 MBeanServers");
            realServer = server;
         }
      }
      this.mbeanServer = realServer;

      boolean useExternalWebContainer = environment == null ? false : Boolean.valueOf(String.valueOf(environment.get(USE_EXTERNAL_WEB_CONTAINER))).booleanValue();
      if (!useExternalWebContainer)
      {
         // Create and start an embedded web container
         String webContainerClassName = environment == null ? null : (String)environment.get(EMBEDDED_WEB_CONTAINER_CLASS);
         // Not present, by default use Jetty
         if (webContainerClassName == null || webContainerClassName.length() == 0) webContainerClassName = JettyWebContainer.class.getName();
         webContainer = createWebContainer(webContainerClassName, environment);
         // Jetty not present, use AXIS
         if (webContainer == null) webContainer = createWebContainer(SimpleAxisWebContainer.class.getName(), environment);
         // Nothing present, give up
         if (webContainer == null) throw new IOException("Could not start embedded web container");
         try
         {
            webContainer.start(address, environment);
         }
         catch (RuntimeException x)
         {
            throw x;
         }
         catch (Exception x)
         {
            throw new IOException(x.toString());
         }
      }

      connectionManager = new SOAPConnectionManager(this, environment);

      setAddress(resolver.bindServer(realServer, address, environment));

      // Here is where we give to clients the possibility to access us
      register(getAddress(), connectionManager);
   }

   private WebContainer createWebContainer(String webContainerClassName, Map environment)
   {
      ClassLoader loader = Thread.currentThread().getContextClassLoader();
      if (environment != null)
      {
         Object cl = environment.get(JMXConnectorServerFactory.PROTOCOL_PROVIDER_CLASS_LOADER);
         if (cl instanceof ClassLoader) loader = (ClassLoader)cl;
      }

      try
      {
         return (WebContainer)loader.loadClass(webContainerClassName).newInstance();
      }
      catch (Exception x)
      {
      }
      return null;
   }

   private void register(JMXServiceURL url, SOAPConnectionManager manager) throws IOException
   {
      synchronized (SOAPConnectorServer.class)
      {
         if (instances.get(url) != null) throw new IOException("A SOAPConnectorServer is already serving at address " + url);
         instances.put(url, manager);
      }
   }

   protected void doStop() throws IOException
   {
      if (connectionManager != null)
      {
         connectionManager.close();
         connectionManager = null;
      }

      JMXServiceURL address = getAddress();

      unregister(address);

      String protocol = address.getProtocol();
      Map environment = getEnvironment();
      ConnectionResolver resolver = ConnectionResolver.newConnectionResolver(protocol, environment);
      if (resolver == null) throw new MalformedURLException("Unsupported protocol: " + protocol);
      resolver.unbindServer(getMBeanServer(), address, environment);

      try
      {
         if (webContainer != null) webContainer.stop();
         webContainer = null;
      }
      catch (Exception x)
      {
         throw new IOException(x.toString());
      }
   }

   private void unregister(JMXServiceURL url) throws IOException
   {
      synchronized (SOAPConnectorServer.class)
      {
         Object removed = instances.remove(url);
         if (removed == null) throw new IOException("No SOAPConnectorServer is present for address " + url);
      }
   }
}
