/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management.monitor;

import javax.management.MBeanNotificationInfo;

import java.util.HashMap;

import javax.management.ObjectName;

/**
 *
 *
 * @author <a href="mailto:tibu@users.sourceforge.net">Carlos Quiroz</a>
 * @version $Revision: 1.9 $
 */
public class GaugeMonitor extends Monitor implements MonitorMBean, GaugeMonitorMBean
{
   private static final Integer NULLINTEGER = new Integer(0);
   private static final Class NONE = null;
   private static final Class INT = Integer.class;
   private static final Class LONG = Long.class;
   private static final Class BYTE = Byte.class;
   private static final Class SHORT = Short.class;
   private static final Class FLOAT = Float.class;
   private static final Class DOUBLE = Double.class;

   private Number highThreshold = NULLINTEGER;
   private Number lowThreshold = NULLINTEGER;
   private boolean notifyHigh = false, notifyLow = false;
   private boolean differenceMode = false;
   private transient boolean errorNotified = false;

   private transient Class type = NONE;
   private transient boolean isLong = false;

   // hold info on last values/timestamps for all observed objects
   private HashMap infos = new HashMap();

   private static final MBeanNotificationInfo[] notificationInfos = {
      new MBeanNotificationInfo(new String[]{
         MonitorNotification.RUNTIME_ERROR,
         MonitorNotification.OBSERVED_OBJECT_ERROR,
         MonitorNotification.OBSERVED_ATTRIBUTE_ERROR,
         MonitorNotification.OBSERVED_ATTRIBUTE_TYPE_ERROR,
         MonitorNotification.THRESHOLD_ERROR,
         MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED,
         MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED
      }
              , "javax.management.monitor.MonitorNotification", "Notifications sent by the GaugeMonitor MBean")
   };

   public synchronized void start()
   {
      doStart();
   }

   public synchronized void stop()
   {
      doStop();
   }

   void executeMonitor(ObjectName objectName, Object attributeValue)
   {
      GaugeInfo gi = (GaugeInfo)infos.get(objectName);
      if (highThreshold == null || highThreshold == NULLINTEGER)
      {
         if (!errorNotified)
         {
            getLogger().info(new StringBuffer("Monitor ").append(this).append(" threshold value is null or zero"));
            notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
            errorNotified = true;
            return;
         }
      }
      if (lowThreshold == null || lowThreshold == NULLINTEGER)
      {
         if (!errorNotified)
         {
            getLogger().info(new StringBuffer("Monitor ").append(this).append(" threshold value is null or zero"));
            notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
            errorNotified = true;
            return;
         }
      }
      // need to be refined
      if (!(attributeValue instanceof Number))
      {
         if (!errorNotified)
         {
            getLogger().info(new StringBuffer("Monitor ").append(this).append(" attribute is not a Number"));
            notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
            errorNotified = true;
            return;
         }
      }
      determineType(attributeValue);
      if (type == NONE)
      {
         if (!errorNotified)
         {
            getLogger().info(new StringBuffer("Monitor ").append(this).append(" attribute, threshold, offset and modules types don't match"));
            notifyListeners(MonitorNotification.THRESHOLD_ERROR, objectName, attribute);
            errorNotified = true;
            return;
         }
      }
      calculateDerivedGauge(gi, (Number)attributeValue);
      if (isLong)
      {
         if (((Number)attributeValue).longValue() >= highThreshold.longValue())
         {
            if (!gi.isHighNotified() && notifyHigh)
            {
               getLogger().info(new StringBuffer("Monitor ").append(this).append(" counter over the threshold"));
               notifyListeners(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, objectName, attribute);
               gi.setHighNotified(true);
               // hystersis
               gi.setLowNotified(false);
            }
            if (!notifyHigh)
            {
               gi.setLowNotified(false);
            }
         }
         if (((Number)attributeValue).longValue() <= lowThreshold.longValue())
         {
            if (!gi.isLowNotified() && notifyLow)
            {
               getLogger().info(new StringBuffer("Monitor ").append(this).append(" counter over the threshold"));
               notifyListeners(MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED, objectName, attribute);
               gi.setLowNotified(true);
               // hystersis
               gi.setHighNotified(false);
            }
            if (!notifyLow)
            {
               gi.setHighNotified(false);
            }
         }
      }
      else
      {
         if (((Number)attributeValue).doubleValue() >= highThreshold.doubleValue())
         {
            if (!gi.isHighNotified() && notifyHigh)
            {
               getLogger().info(new StringBuffer("Monitor ").append(this).append(" counter over the threshold"));
               notifyListeners(MonitorNotification.THRESHOLD_HIGH_VALUE_EXCEEDED, objectName, attribute);
               gi.setHighNotified(true);
               // hystersis
               gi.setLowNotified(false);
            }
            if (!notifyHigh)
            {
               gi.setLowNotified(false);
            }
         }
         if (((Number)attributeValue).doubleValue() <= lowThreshold.doubleValue())
         {
            if (!gi.isLowNotified() && notifyLow)
            {
               getLogger().info(new StringBuffer("Monitor ").append(this).append(" counter over the threshold"));
               notifyListeners(MonitorNotification.THRESHOLD_LOW_VALUE_EXCEEDED, objectName, attribute);
               gi.setLowNotified(true);
               // hystersis
               gi.setHighNotified(false);
            }
            if (!notifyLow)
            {
               gi.setHighNotified(false);
            }
         }
      }
   }

   void determineType(Object attributeValue)
   {
      Class targetClass = attributeValue.getClass();
      if (highThreshold != null && lowThreshold != null)
      {
         if (highThreshold.getClass().equals(lowThreshold.getClass()) && highThreshold.getClass().equals(targetClass))
         {
            boolean match = true;
            if (targetClass != INT && targetClass != LONG && targetClass != BYTE
                    && targetClass != SHORT && targetClass != FLOAT && targetClass != DOUBLE)
            {
               match = false;
            }
            if (match)
            {
               type = targetClass;
               if (targetClass.equals(FLOAT) || targetClass.equals(DOUBLE))
               {
                  isLong = false;
               }
               else
               {
                  isLong = true;
               }
            }
         }
         else
         {
            type = NONE;
         }
      }
      else
      {
         type = NONE;
      }
   }

   void calculateDerivedGauge(GaugeInfo gi, Number attributeValue)
   {
      gi.setLastDerivatedGaugeTimestamp(System.currentTimeMillis());
      if (differenceMode)
      {
         if (gi.getLastValue() != null)
         {
            if (isLong)
            {
               long difference = attributeValue.longValue() - gi.getLastValue().longValue();
               gi.setLastDerivatedGauge(createNumber(difference));
            }
            else
            {
               double difference = attributeValue.doubleValue() - gi.getLastValue().doubleValue();
               gi.setLastDerivatedGauge(createNumber(difference));
            }
         }
      }
      else
      {
         gi.setLastDerivatedGauge(attributeValue);
      }
      gi.setLastValue(attributeValue);
   }

   Number createNumber(long value)
   {
      Number result = null;
      if (type == INT)
      {
         result = new Integer((int)value);
      }
      else if (type == LONG)
      {
         result = new Long(value);
      }
      else if (type == SHORT)
      {
         result = new Short((short)value);
      }
      else if (type == BYTE)
      {
         result = new Byte((byte)value);
      }
      else
      {
         getLogger().error("Invalid type " + type + " in createNumber(long)");
      }
      return result;
   }

   Number createNumber(double value)
   {
      Number result = null;
      if (type == FLOAT)
      {
         result = new Float((float)value);
      }
      else if (type == DOUBLE)
      {
         result = new Double(value);
      }
      else
      {
         getLogger().error("Invalid type " + type + " in createNumber(double)");
      }
      return result;
   }

   public synchronized Number getDerivedGauge()
   {
      if (objectNames.size() == 0) return null;
      return getDerivedGauge((ObjectName)objectNames.get(0));
   }

   public Number getDerivedGauge(ObjectName objectName)
   {
      return ((GaugeInfo)infos.get(objectName)).getLastDerivatedGauge();
   }


   public synchronized long getDerivedGaugeTimeStamp()
   {
      if (objectNames.size() == 0) return 0;
      return getDerivedGaugeTimeStamp((ObjectName)objectNames.get(0));
   }

   public long getDerivedGaugeTimeStamp(ObjectName objectName)
   {
      return ((GaugeInfo)infos.get(objectName)).getLastDerivatedGaugeTimestamp();
   }

   public Number getHighThreshold()
   {
      return highThreshold;
   }

   public Number getLowThreshold()
   {
      return lowThreshold;
   }

   public void setThresholds(Number highValue, Number lowValue) throws java.lang.IllegalArgumentException
   {
      if (highValue == null || lowValue == null)
      {
         throw new IllegalArgumentException("Threshold values cannot be null");
      }
      if (!highValue.getClass().equals(lowValue.getClass()))
      {
         throw new IllegalArgumentException("Threshold need to be of the same class");
      }
      if (highValue.doubleValue() < lowValue.doubleValue())
      {
         throw new IllegalArgumentException("High threshold value must be greater than low threshold value");
      }
      highThreshold = highValue;
      lowThreshold = lowValue;
   }

   public boolean getNotifyHigh()
   {
      return notifyHigh;
   }

   public void setNotifyHigh(boolean value)
   {
      this.notifyHigh = value;
   }

   public boolean getNotifyLow()
   {
      return notifyLow;
   }

   public void setNotifyLow(boolean value)
   {
      this.notifyLow = value;
   }

   public boolean getDifferenceMode()
   {
      return differenceMode;
   }

   public void setDifferenceMode(boolean value)
   {
      this.differenceMode = value;
   }

   public MBeanNotificationInfo[] getNotificationInfo()
   {
      return notificationInfos;
   }

   public String toString()
   {
      return new StringBuffer("GaugeMonitor on ").append(super.toString()).toString();
   }

   public synchronized void addObservedObject(ObjectName objectName) throws java.lang.IllegalArgumentException
   {
      super.addObservedObject(objectName);
      infos.put(objectName, new GaugeInfo());
   }

   public void removeObservedObject(ObjectName objectName)
   {
      super.removeObservedObject(objectName);
      infos.remove(objectName);
   }

   class GaugeInfo
   {
      Number lastDerivatedGauge = new Integer(0);
      long lastDerivatedGaugeTimestamp = 0;

      boolean highNotified = false;
      boolean lowNotified = false;
      Number lastValue = null;

      public GaugeInfo()
      {
      }

      /** Getter for property lastDerivatedGauge.
       * @return Value of property lastDerivatedGauge.
       *
       */
      public java.lang.Number getLastDerivatedGauge()
      {
         return lastDerivatedGauge;
      }

      /** Setter for property lastDerivatedGauge.
       * @param lastDerivatedGauge New value of property lastDerivatedGauge.
       *
       */
      public void setLastDerivatedGauge(java.lang.Number lastDerivatedGauge)
      {
         this.lastDerivatedGauge = lastDerivatedGauge;
      }

      /** Getter for property lastDerivatedGaugeTimestamp.
       * @return Value of property lastDerivatedGaugeTimestamp.
       *
       */
      public long getLastDerivatedGaugeTimestamp()
      {
         return lastDerivatedGaugeTimestamp;
      }

      /** Setter for property lastDerivatedGaugeTimestamp.
       * @param lastDerivatedGaugeTimestamp New value of property lastDerivatedGaugeTimestamp.
       *
       */
      public void setLastDerivatedGaugeTimestamp(long lastDerivatedGaugeTimestamp)
      {
         this.lastDerivatedGaugeTimestamp = lastDerivatedGaugeTimestamp;
      }

      /** Getter for property lastValue.
       * @return Value of property lastValue.
       *
       */
      public java.lang.Number getLastValue()
      {
         return lastValue;
      }

      /** Setter for property lastValue.
       * @param lastValue New value of property lastValue.
       *
       */
      public void setLastValue(java.lang.Number lastValue)
      {
         this.lastValue = lastValue;
      }

      /** Getter for property highNotified.
       * @return Value of property highNotified.
       *
       */
      public boolean isHighNotified()
      {
         return highNotified;
      }

      /** Setter for property highNotified.
       * @param highNotified New value of property highNotified.
       *
       */
      public void setHighNotified(boolean highNotified)
      {
         this.highNotified = highNotified;
      }

      /** Getter for property lowNotified.
       * @return Value of property lowNotified.
       *
       */
      public boolean isLowNotified()
      {
         return lowNotified;
      }

      /** Setter for property lowNotified.
       * @param lowNotified New value of property lowNotified.
       *
       */
      public void setLowNotified(boolean lowNotified)
      {
         this.lowNotified = lowNotified;
      }

   }


}
