/* $NetBSD: $ */

/*-
 * Copyright (c) 2017 Jared McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "opt_soc.h"
#include "opt_multiprocessor.h"
#include "opt_console.h"
#include "opt_fdt_arm.h"

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/bus.h>
#include <sys/cpu.h>
#include <sys/device.h>
#include <sys/termios.h>

#include <dev/fdt/fdtvar.h>
#include <arm/fdt/arm_fdtvar.h>

#include <uvm/uvm_extern.h>

#include <machine/bootconfig.h>
#include <arm/cpufunc.h>

#include <arm/cortex/gtmr_var.h>
#include <arm/cortex/gic_reg.h>

#include <arm/arm/psci.h>
#include <arm/fdt/psci_fdtvar.h>

#include <arm/meson/meson_platform.h>
#include <arm/meson/meson_comreg.h>

#include <libfdt.h>

#define	MESON_REF_FREQ	24000000

#define	SUN50I_H6_WDT_BASE	0x01c20ca0
#define	SUN50I_H6_WDT_SIZE	0x20
#define	SUN50I_H6_WDT_CFG	0x14
#define	 SUN50I_H6_WDT_CFG_SYS	__BIT(0)
#define	SUN50I_H6_WDT_MODE	0x18
#define	 SUN50I_H6_WDT_MODE_EN	__BIT(0)

extern struct arm32_bus_dma_tag arm_generic_dma_tag;
extern struct bus_space arm_generic_bs_tag;
extern struct bus_space arm_generic_a4x_bs_tag;

#define	meson_dma_tag		arm_generic_dma_tag
#define	meson_bs_tag		arm_generic_bs_tag
#define	meson_a4x_bs_tag	arm_generic_a4x_bs_tag

static const struct pmap_devmap *
meson_platform_devmap(void)
{
	static const struct pmap_devmap devmap[] = {
		DEVMAP_ENTRY(MESON_CORE_VBASE,
			     MESON_CORE_PBASE,
			     MESON_CORE_SIZE),
		DEVMAP_ENTRY(MESON_APB_VBASE,
			     MESON_APB_PBASE,
			     MESON_APB_SIZE),
		DEVMAP_ENTRY_END
	};

	return devmap;
}

static void
meson_platform_init_attach_args(struct fdt_attach_args *faa)
{
	faa->faa_bst = &meson_bs_tag;
	faa->faa_a4x_bst = &meson_a4x_bs_tag;
	faa->faa_dmat = &meson_dma_tag;
}

void meson_platform_early_putchar(char);

void
meson_platform_early_putchar(char c)
{
	int timo = 150000;
#ifdef CONSADDR
#define CONSADDR_VA	((CONSADDR - MESON_CORE_PBASE) + MESON_CORE_VBASE)
	volatile uint32_t *uartaddr = cpu_earlydevice_va_p() ?
	    (volatile uint32_t *)CONSADDR_VA :
	    (volatile uint32_t *)CONSADDR;

	while ((le32toh(uartaddr[UART_STATUS_REG/4]) & UART_STATUS_TX_EMPTY) == 0) {
		if (--timo == 0)
			break;
	}

	uartaddr[UART_WFIFO_REG] = htole32(c);

	while ((le32toh(uartaddr[UART_STATUS_REG/4]) & UART_STATUS_TX_EMPTY) == 0) {
		if (--timo == 0)
			break;
	}
#endif
}

static void
meson_platform_device_register(device_t self, void *aux)
{
	prop_dictionary_t prop = device_properties(self);

	if (device_is_a(self, "rgephy")) {
		/* Pine64+ and NanoPi NEO Plus2 gigabit ethernet workaround */
		const char * compat[] = {
			"pine64,pine64-plus",
			"friendlyarm,nanopi-neo-plus2",
			NULL
		};
		if (of_match_compatible(OF_finddevice("/"), compat)) {
			prop_dictionary_set_bool(prop, "no-rx-delay", true);
		}
	}
}

static u_int
meson_platform_uart_freq(void)
{
	return MESON_REF_FREQ;
}

static void
meson_platform_bootstrap(void)
{
	void *fdt_data = __UNCONST(fdtbus_get_data());
	const int chosen_off = fdt_path_offset(fdt_data, "/chosen");
	if (chosen_off < 0)
		return;

	if (match_bootconf_option(boot_args, "console", "fb")) {
		const int framebuffer_off =
		    fdt_path_offset(fdt_data, "/chosen/framebuffer");
		if (framebuffer_off >= 0) {
			const char *status = fdt_getprop(fdt_data,
			    framebuffer_off, "status", NULL);
			if (status == NULL || strncmp(status, "ok", 2) == 0) {
				fdt_setprop_string(fdt_data, chosen_off,
				    "stdout-path", "/chosen/framebuffer");
			}
		}
	} else if (match_bootconf_option(boot_args, "console", "serial")) {
		fdt_setprop_string(fdt_data, chosen_off,
		    "stdout-path", "serial0:115200n8");
	}
}

static void
meson_platform_psci_bootstrap(void)
{
	arm_fdt_cpu_bootstrap();
	meson_platform_bootstrap();
}

static void
meson_s905_platform_reset(void)
{
	bus_space_tag_t bst = &meson_bs_tag;
	bus_space_handle_t bsh;

	bus_space_map(bst, SUN50I_H6_WDT_BASE, SUN50I_H6_WDT_SIZE, 0, &bsh);

	bus_space_write_4(bst, bsh, SUN50I_H6_WDT_CFG, SUN50I_H6_WDT_CFG_SYS);
	bus_space_write_4(bst, bsh, SUN50I_H6_WDT_MODE, SUN50I_H6_WDT_MODE_EN);
}

static const struct arm_platform meson_s905_platform = {
	.ap_devmap = meson_platform_devmap,
	.ap_bootstrap = meson_platform_psci_bootstrap,
	.ap_init_attach_args = meson_platform_init_attach_args,
	.ap_device_register = meson_platform_device_register,
	.ap_reset = meson_s905_platform_reset,
	.ap_delay = gtmr_delay,
	.ap_uart_freq = meson_platform_uart_freq,
};

ARM_PLATFORM(meson_s905, "hardkernel,odroid-c2", &meson_s905_platform);
