/* $NetBSD: $ */

/*-
 * Copyright (c) 2018 Jared McNeill <jmcneill@invisible.ca>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/bus.h>
#include <sys/device.h>
#include <sys/intr.h>
#include <sys/systm.h>
#include <sys/mutex.h>
#include <sys/kmem.h>
#include <sys/lwp.h>

#include <dev/fdt/fdtvar.h>

struct meson_iomux {
	bus_size_t		base;
	u_int			type;
#define	MESON_IOMUX_TYPE_3BIT	0x01
};

struct meson_iomux_bank {
	struct meson_iomux		iomux[4];
};


struct meson_iomux_config {
	const struct meson_iomux_bank *banks;
	u_int nbanks;
};

#if 0
static const struct meson_iomux_config rk3328_iomux_config = {
	.banks = rk3328_iomux_banks,
	.nbanks = __arraycount(rk3328_iomux_banks),
};
#endif

static const struct of_compat_data compat_data[] = {
	{ "amlogic,meson-gxbb-aobus-pinctrl",	NULL },
	{ NULL }
};

struct meson_iomux_softc {
	device_t sc_dev;
	bus_space_tag_t sc_bst;
	bus_space_handle_t sc_bsh;

	const struct meson_iomux_config *sc_conf;
};

#define RD4(sc, reg) 		\
    bus_space_read_4((sc)->sc_bst, (sc)->sc_bsh, (reg))
#define WR4(sc, reg, val) 	\
    bus_space_write_4((sc)->sc_bst, (sc)->sc_bsh, (reg), (val))

static int	meson_iomux_match(device_t, cfdata_t, void *);
static void	meson_iomux_attach(device_t, device_t, void *);

CFATTACH_DECL_NEW(meson_iomux, sizeof(struct meson_iomux_softc),
	meson_iomux_match, meson_iomux_attach, NULL, NULL);

#if 0
static void
meson_iomux_calc_iomux_reg(struct meson_iomux_softc *sc, u_int bank, u_int pin, bus_size_t *reg, uint32_t *mask)
{
	const struct meson_iomux_bank *banks = sc->sc_conf->banks;

	KASSERT(bank < sc->sc_conf->nbanks);

	*reg = banks[bank].iomux[pin / 8].base;
	if (banks[bank].iomux[pin / 8].type & MESON_IOMUX_TYPE_3BIT) {
		if ((pin % 8) >= 5)
			*reg += 0x04;
		const u_int bit = (pin % 8 % 5) * 3;
		*mask = 7 << bit;
	} else {
		const u_int bit = (pin % 8) * 2;
		*mask = 3 << bit;
	}
}

static void
meson_iomux_set_bias(struct meson_iomux_softc *sc, u_int bank, u_int idx, u_int bias)
{
	WR4(sc, GRF_GPIO_P_REG(bank, idx),
	    __SHIFTIN(GRF_GPIO_P_CTL_MASK, GRF_GPIO_P_WRITE_EN(idx)) |
	    __SHIFTIN(bias, GRF_GPIO_P_CTL(idx)));
}

static void
meson_iomux_set_drive_strength(struct meson_iomux_softc *sc, u_int bank, u_int idx, u_int drv)
{
	WR4(sc, GRF_GPIO_E_REG(bank, idx),
	    __SHIFTIN(GRF_GPIO_E_CTL_MASK, GRF_GPIO_E_WRITE_EN(idx)) |
	    __SHIFTIN(drv, GRF_GPIO_E_CTL(idx)));
}

static void
meson_iomux_set_mux(struct meson_iomux_softc *sc, u_int bank, u_int idx, u_int mux)
{
	bus_size_t reg;
	uint32_t mask;

	meson_iomux_calc_iomux_reg(sc, bank, idx, &reg, &mask);

	WR4(sc, reg, (mask << 16) | __SHIFTIN(mux, mask));
}
#endif

#if 0
static int
meson_iomux_config(struct meson_iomux_softc *sc, const int phandle, u_int bank, u_int idx, u_int mux)
{
	u_int drv;

	if (of_hasprop(phandle, "bias-disable"))
		meson_iomux_set_bias(sc, bank, idx, GRF_GPIO_P_CTL_Z);
	else if (of_hasprop(phandle, "bias-pull-up"))
		meson_iomux_set_bias(sc, bank, idx, GRF_GPIO_P_CTL_PULLUP);
	else if (of_hasprop(phandle, "bias-pull-down"))
		meson_iomux_set_bias(sc, bank, idx, GRF_GPIO_P_CTL_PULLDOWN);

	if (of_getprop_uint32(phandle, "drive-strength", &drv) == 0) {
		switch (drv) {
		case 2:
			meson_iomux_set_drive_strength(sc, bank, idx, GRF_GPIO_E_CTL_2MA);
			break;
		case 4:
			meson_iomux_set_drive_strength(sc, bank, idx, GRF_GPIO_E_CTL_4MA);
			break;
		case 8:
			meson_iomux_set_drive_strength(sc, bank, idx, GRF_GPIO_E_CTL_8MA);
			break;
		case 12:
			meson_iomux_set_drive_strength(sc, bank, idx, GRF_GPIO_E_CTL_12MA);
			break;
		default:
			aprint_error_dev(sc->sc_dev, "unsupported drive-strength %u\n", drv);
			return EINVAL;
		}
	}

#if notyet
	if (of_hasprop(phandle, "input-enable"))
		meson_iomux_set_direction(sc, bank, idx, GPIO_PIN_INPUT, -1);
	else if (of_hasprop(phandle, "output-high"))
		meson_iomux_set_direction(sc, bank, idx, GPIO_PIN_OUTPUT, GPIO_PIN_HIGH);
	else if (of_hasprop(phandle, "output-low"))
		meson_iomux_set_direction(sc, bank, idx, GPIO_PIN_OUTPUT, GPIO_PIN_LOW);
#endif

	meson_iomux_set_mux(sc, bank, idx, mux);

	return 0;
}
#endif

static int
meson_iomux_pinctrl_set_config(device_t dev, const void *data, size_t len)
{
#if 0
	struct meson_iomux_softc * const sc = device_private(dev);
	int pins_len;

	if (len != 4)
		return -1;

	const int phandle = fdtbus_get_phandle_from_native(be32dec(data));
	const u_int *pins = fdtbus_get_prop(phandle, "groups", &pins_len);

	while (pins_len >= 16) {
		const u_int bank = be32toh(pins[0]);
		const u_int idx = be32toh(pins[1]);
		const u_int mux = be32toh(pins[2]);
		const int cfg = fdtbus_get_phandle_from_native(be32toh(pins[3]));

		meson_iomux_config(sc, cfg, bank, idx, mux);

		pins_len -= 16;
		pins += 4;
	}
#endif
	return 0;
}

static struct fdtbus_pinctrl_controller_func meson_iomux_pinctrl_funcs = {
	.set_config = meson_iomux_pinctrl_set_config,
};

static int
meson_iomux_match(device_t parent, cfdata_t cf, void *aux)
{
	struct fdt_attach_args * const faa = aux;

	return of_match_compat_data(faa->faa_phandle, compat_data);
}

static void
meson_iomux_attach(device_t parent, device_t self, void *aux)
{
	struct meson_iomux_softc * const sc = device_private(self);
	struct fdt_attach_args * const faa = aux;
	const int phandle = faa->faa_phandle;
#if 0
	bus_addr_t addr;
	bus_size_t size;
#endif
	int child, sub;

	sc->sc_dev = self;
	sc->sc_bst = faa->faa_bst;
#if 0
	if (bus_space_map(sc->sc_bst, addr, size, 0, &sc->sc_bsh) != 0) {
		aprint_error(": couldn't map registers\n");
		return;
	}
#endif
	sc->sc_conf = (void *)of_search_compatible(phandle, compat_data)->data;

	aprint_naive("\n");
	aprint_normal(": IOMUX control\n");

	for (child = OF_child(phandle); child; child = OF_peer(child)) {
		for (sub = OF_child(child); sub; sub = OF_peer(sub)) {
			if (!of_hasprop(sub, "function"))
				continue;
			fdtbus_register_pinctrl_config(self, sub, &meson_iomux_pinctrl_funcs);
		}
	}

	fdtbus_pinctrl_configure();

	for (child = OF_child(phandle); child; child = OF_peer(child)) {
		struct fdt_attach_args cfaa = *faa;
		cfaa.faa_phandle = child;
		cfaa.faa_name = fdtbus_get_string(child, "name");
		cfaa.faa_quiet = false;

		config_found(self, &cfaa, NULL);
	}
}
