/* $NetBSD: $ */

/*-
 * Copyright (c) 2013, 2014, 2015 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Matt Thomas of 3am Software Foundry and Martin Husemann.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#include <sys/cdefs.h>

__KERNEL_RCSID(1, "$NetBSD: $");

#include <sys/param.h>
#include <sys/bus.h>
#include <sys/device.h>
#include <sys/intr.h>
#include <sys/systm.h>

#include <net/if.h>
#include <net/if_ether.h>
#include <net/if_media.h>

#include <dev/mii/miivar.h>

#include <dev/ic/dwc_gmac_var.h>
#include <dev/ic/dwc_gmac_reg.h>

#include <dev/fdt/fdtvar.h>
#include <dev/fdt/syscon.h>

#define MESON_GMAC_TXDLY_DEFAULT	10
#define MESON_GMAC_RGMII_MASK	__BITS()
#define MESON_GMAC_PHY_MODE(x)	__SHIFTIN((x), __BIT(0)

static int meson_gmac_match(device_t, cfdata_t, void *);
static void meson_gmac_attach(device_t, device_t, void *);
static int meson_gmac_intr(void*);

static const char * const compatible[] = {
	"amlogic,meson8b-dwmac",
	"amlogic,meson-gx-dwmac",
	"amlogic,meson-gxbb-dwmac",
	"snps,dwmac-3.70a",
	"snps,dwmac",
	NULL
};

struct meson_gmac_softc {
	struct dwc_gmac_softc sc_core;
	bus_space_handle_t sc_regs;
	void *sc_ih;
};

CFATTACH_DECL_NEW(meson_gmac, sizeof(struct meson_gmac_softc),
	meson_gmac_match, meson_gmac_attach, NULL, NULL);

static void
meson_gmac_set_mode_rgmii(struct meson_gmac_softc *sc, u_int tx_delay)
{
#if 0
	struct rk_gmac_softc * const rk_sc = (struct rk_gmac_softc *)sc;
	uint32_t write_mask, write_val;

	syscon_lock(rk_sc->sc_syscon);

	write_mask = (RK3328_GRF_MAC_CON1_MODE | RK3328_GRF_MAC_CON1_SEL) << 16;
	write_val = __SHIFTIN(RK3328_GRF_MAC_CON1_SEL_RGMII, RK3328_GRF_MAC_CON1_SEL);
	syscon_write_4(rk_sc->sc_syscon, RK3328_GRF_MAC_CON1, write_mask | write_val);

#if notyet
	write_mask = (RK3328_GRF_MAC_CON0_TXDLY | RK3328_GRF_MAC_CON0_RXDLY) << 16;
	write_val = __SHIFTIN(tx_delay, RK3328_GRF_MAC_CON0_TXDLY) |
		    __SHIFTIN(rx_delay, RK3328_GRF_MAC_CON0_RXDLY);
	syscon_write_4(rk_sc->sc_syscon, RK3328_GRF_MAC_CON0, write_mask | write_val);

	write_mask = (RK3328_GRF_MAC_CON1_RXDLY_EN | RK3328_GRF_MAC_CON1_TXDLY_EN) << 16;
	write_val = RK3328_GRF_MAC_CON1_RXDLY_EN | RK3328_GRF_MAC_CON1_TXDLY_EN;
	syscon_write_4(rk_sc->sc_syscon, RK3328_GRF_MAC_CON1, write_mask | write_val);
#endif

	syscon_unlock(rk_sc->sc_syscon);
#endif
}

static void
meson_gmac_set_speed_rgmii(struct dwc_gmac_softc *sc, int speed)
{
#if 0
	struct rk_gmac_softc * const rk_sc = (struct rk_gmac_softc *)sc;
	u_int clksel;

	switch (speed) {
	case IFM_10_T:
		clksel = RK3328_GRF_MAC_CON1_CLKSEL_2_5M;
		break;
	case IFM_100_TX:
		clksel = RK3328_GRF_MAC_CON1_CLKSEL_25M;
		break;
	default:
		clksel = RK3328_GRF_MAC_CON1_CLKSEL_125M;
		break;
	}

	syscon_lock(rk_sc->sc_syscon);
	syscon_write_4(rk_sc->sc_syscon, RK3328_GRF_MAC_CON1,
	    (RK3328_GRF_MAC_CON1_CLKSEL << 16) |
	    __SHIFTIN(RK3328_GRF_MAC_CON1_CLKSEL_125M, RK3328_GRF_MAC_CON1_CLKSEL));
	syscon_unlock(rk_sc->sc_syscon);
#endif
}

static int
meson_gmac_match(device_t parent, cfdata_t cf, void *aux)
{
	struct fdt_attach_args * const faa = aux;

	return of_match_compatible(faa->faa_phandle, compatible);
}

static void
meson_gmac_attach(device_t parent, device_t self, void *aux)
{
	struct meson_gmac_softc * const sc = device_private(self);
	struct fdt_attach_args * const faa = aux;
	const int phandle = faa->faa_phandle;
	const char *phy_mode;
	bus_addr_t addr, size;
	char intrstr[128];
	u_int tx_delay;

	sc->sc_core.sc_dev = self;
	sc->sc_core.sc_bst = faa->faa_bst;
	sc->sc_core.sc_dmat = faa->faa_dmat;
	if (fdtbus_get_reg(phandle, 0, &addr, &size) != 0)
		return;
	if(bus_space_map(sc->sc_core.sc_bst, addr, size, 0, &sc->sc_core.sc_bsh) != 0)
		return;

	if (fdtbus_get_reg(phandle, 1, &addr, &size) != 0)
		return;
	if(bus_space_map(sc->sc_core.sc_bst, addr, size, 0, &sc->sc_regs) != 0)
		return;

	if (!fdtbus_intr_str(phandle, 0, intrstr, sizeof(intrstr))) {
		aprint_error(": failed to decode interrupt\n");
		return;
	}

	if (of_getprop_uint32(phandle, "amlogic,tx_delay-ns", &tx_delay) != 0)
		tx_delay = MESON_GMAC_TXDLY_DEFAULT;

	phy_mode = fdtbus_get_string(phandle, "phy-mode");
	if (phy_mode == NULL) {
		aprint_error(": missing 'phy-mode' property\n");
		return;
	}

	if (strcmp(phy_mode, "rgmii") == 0) {
		meson_gmac_set_mode_rgmii(sc, tx_delay);

		sc->sc_core.sc_set_speed = meson_gmac_set_speed_rgmii;
	} else {
		aprint_error(": unsupported phy-mode '%s'\n", phy_mode);
		return;
	}

	aprint_naive("\n");
	aprint_normal(": Gigabit Ethernet Controller\n");

#if 0
	meson_eth_init();
#endif

	/*
	 * Interrupt handler
	 */
	sc->sc_ih = fdtbus_intr_establish(phandle, 0, IPL_NET,
	    0, meson_gmac_intr, sc);
	if (sc->sc_ih == NULL) {
		aprint_error_dev(self, "failed to establish interrupt %s\n",
		     intrstr);
		return;
	}
	aprint_normal_dev(self, "interrupting on irq %s\n", intrstr);

	dwc_gmac_attach(&sc->sc_core, GMAC_MII_CLK_100_150M_DIV62);
}

#if 0
static int
meson_gmac_reset(const int phandle)
{
	struct fdtbus_gpio_pin *pin_reset;
	const u_int *reset_delay_us;
	bool reset_active_low;
	int len, val;

	pin_reset = fdtbus_gpio_acquire(phandle, "snps,reset-gpio", GPIO_PIN_OUTPUT);
	if (pin_reset == NULL)
		return 0;

	reset_delay_us = fdtbus_get_prop(phandle, "snps,reset-delays-us", &len);
	if (reset_delay_us == NULL || len != 12)
		return ENXIO;

	reset_active_low = of_hasprop(phandle, "snps,reset-active-low");

	val = reset_active_low ? 1 : 0;

	fdtbus_gpio_write(pin_reset, val);
	delay(be32toh(reset_delay_us[0]));
	fdtbus_gpio_write(pin_reset, !val);
	delay(be32toh(reset_delay_us[1]));
	fdtbus_gpio_write(pin_reset, val);
	delay(be32toh(reset_delay_us[2]));

	return 0;
}
#endif

static int
meson_gmac_intr(void *arg)
{
	struct meson_gmac_softc *sc = arg;

	return dwc_gmac_intr(&sc->sc_core);
}
