/*	$NetBSD: $	*/

/*-
 * Copyright (c) 2013 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Nick Hudson
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/device.h>
#include <sys/mutex.h>
#include <sys/bus.h>
#include <sys/workqueue.h>

#include <dev/fdt/fdtvar.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include <dev/usb/usbdivar.h>
#include <dev/usb/usb_mem.h>

#include <dwc2/dwc2var.h>

#include <dwc2/dwc2.h>
#include "dwc2_core.h"

static const char * const compatible[] = {
	"amlogic,meson8b-usb",
	"amlogic,meson-gxbb-usb",
	"snps,dwc2",
	NULL
};

struct meson_dwctwo_softc {
	struct dwc2_softc	sc_dwc2;

	void			*sc_ih;
};

static struct dwc2_core_params meson_dwctwo_params = {
	.otg_cap			= 2,	/* 0 - HNP/SRP capable */
	.otg_ver			= 1,	/* 1 - 2.0 */
	.dma_enable			= -1,	/* 1 - DMA (default, if available) */
	.dma_desc_enable		= 0,	/* 1 - Descriptor DMA (default, if available) */
	.speed				= -1,	/* 0 - High Speed */
	.enable_dynamic_fifo		= 1,	/* 1 - Allow dynamic FIFO sizing (default, if available) */
	.en_multiple_tx_fifo		= -1,	/* Specifies whether dedicated per-endpoint transmit FIFOs are enabled */
	.host_rx_fifo_size		= 512,	/* 512 DWORDs */
	.host_nperio_tx_fifo_size	= 500,	/* 500 DWORDs */
	.host_perio_tx_fifo_size	= -1,	/* 256 DWORDs */
	.max_transfer_size		= -1,   /* 2047 to 65,535 */
	.max_packet_count		= -1,   /* 15 to 511 */
	.host_channels			= -1,	/* 1 to 16 */
	.phy_type			= 1, 	/* 1- UTMI+ Phy */
	.phy_utmi_width			= -1,	/* 8 bits */
	.phy_ulpi_ddr			= -1,	/* Single */
	.phy_ulpi_ext_vbus		= -1,
	.i2c_enable			= -1,
	.ulpi_fs_ls			= -1,	/* 0 - No (default) */
	.host_support_fs_ls_low_power	= -1,	/* 0 - Don't support low power mode (default) */
	.host_ls_low_power_phy_clk	= -1,	/* 1 - 48 MHz  default when phy_type is UTMI+ or ULPI*/
	.ts_dline			= -1,	/* 0 - No (default) */
	.reload_ctl			= -1,	/* 0 - No (default for core < 2.92a) */
	.ahbcfg				= 0x3,	/* INCR4 */
	.uframe_sched			= 1,	/* True to enable microframe scheduler */
	.external_id_pin_ctl		= -1,
	.hibernation			= -1,
};

static int meson_dwctwo_match(device_t, struct cfdata *, void *);
static void meson_dwctwo_attach(device_t, device_t, void *);
static void meson_dwctwo_deferred(device_t);

CFATTACH_DECL_NEW(meson_dwctwo, sizeof(struct meson_dwctwo_softc),
    meson_dwctwo_match, meson_dwctwo_attach, NULL, NULL);

static int
meson_dwctwo_match(device_t parent, struct cfdata *match, void *aux)
{
	struct fdt_attach_args * const faa = aux;

	return of_match_compatible(faa->faa_phandle, compatible);
}

static void
meson_dwctwo_attach(device_t parent, device_t self, void *aux)
{
	struct meson_dwctwo_softc * const sc = device_private(self);
	struct fdt_attach_args * const faa = aux;
	const int phandle = faa->faa_phandle;
	bus_addr_t addr, size;
	char intrstr[128];

	sc->sc_dwc2.sc_dev = self;
	sc->sc_dwc2.sc_iot = faa->faa_bst;

	if (fdtbus_get_reg(phandle, 0, &addr, &size) != 0)
		return;
	if(bus_space_map(sc->sc_dwc2.sc_iot, addr, size, 0, &sc->sc_dwc2.sc_ioh) != 0)
		return;
	if (!fdtbus_intr_str(phandle, 0, intrstr, sizeof(intrstr))) {
		aprint_error(": failed to decode interrupt\n");
		return;
	}

	sc->sc_dwc2.sc_bus.ub_dmatag = faa->faa_dmat;
	sc->sc_dwc2.sc_params = &meson_dwctwo_params;

	aprint_naive("\n");
	aprint_normal(": USB controller\n");

	sc->sc_ih = fdtbus_intr_establish(phandle, 0, IPL_VM,
	    0, dwc2_intr, &sc->sc_dwc2);
	if (sc->sc_ih == NULL) {
		aprint_error_dev(self, "failed to establish interrupt %s\n",
		    intrstr);
		return;
	}
	aprint_normal_dev(self, "interrupting on %s\n", intrstr);

	config_defer(self, meson_dwctwo_deferred);
}

static void
meson_dwctwo_deferred(device_t self)
{
	struct meson_dwctwo_softc *sc = device_private(self);
	int error;

	error = dwc2_init(&sc->sc_dwc2);
	if (error != 0) {
		aprint_error_dev(self, "couldn't initialize host, error=%d\n",
		    error);
		return;
	}
	sc->sc_dwc2.sc_child = config_found(sc->sc_dwc2.sc_dev,
	    &sc->sc_dwc2.sc_bus, usbctlprint);
}
