/* $NetBSD: $ */

/*-
 * Copyright (c) 2013 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Matt Thomas of 3am Software Foundry.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "opt_fdt_arm.h"

#include <sys/cdefs.h>

__KERNEL_RCSID(1, "$NetBSD: $");

#define cn_trap()			\
	do {				\
		console_debugger();	\
		cn_trapped = 1;		\
	} while (/* CONSTCOND */ 0)

#include <sys/param.h>
#include <sys/bus.h>
#include <sys/device.h>
#include <sys/conf.h>
#include <sys/intr.h>
#include <sys/systm.h>
#include <sys/time.h>
#include <sys/termios.h>
#include <sys/kauth.h>
#include <sys/lwp.h>
#include <sys/tty.h>

#include <dev/cons.h>
#include <dev/fdt/fdtvar.h>

#include <arm/meson/meson_comreg.h>

static int	meson_com_match(device_t, cfdata_t, void *);
static void	meson_com_attach(device_t, device_t, void *);

static int	meson_com_intr(void *);

static int	meson_com_cngetc(dev_t);
static void	meson_com_cnputc(dev_t, int);
static void	meson_com_cnpollc(dev_t, int);

static void	meson_com_start(struct tty *);
static int	meson_com_param(struct tty *, struct termios *);

static bool	meson_com_cnattach(bus_space_tag_t, bus_space_handle_t,
				    int, tcflag_t);
static bool	meson_com_is_console(bus_addr_t);

static const char * const compatible[] = {
	"amlogic,meson8b-uart",
	"amlogic,meson-gx-uart",
	"amlogic,meson-uart",
	NULL
};

extern struct cfdriver mesoncom_cd;

struct meson_com_softc {
	device_t sc_dev;
	bus_space_tag_t	sc_bst;
	bus_space_handle_t sc_bsh;
	void *sc_ih;

	struct tty *sc_tty;

	int sc_ospeed;
	tcflag_t sc_cflag;

	u_char sc_buf[1024];
};

static struct meson_com_softc meson_com_cnsc;

static struct cnm_state meson_com_cnm_state;

struct consdev meson_com_consdev = {
	.cn_getc = meson_com_cngetc,
	.cn_putc = meson_com_cnputc,
	.cn_pollc = meson_com_cnpollc,
	.cn_dev = NODEV,
	.cn_pri = CN_NORMAL,
};

static dev_type_open(meson_com_open);
static dev_type_open(meson_com_close);
static dev_type_read(meson_com_read);
static dev_type_write(meson_com_write);
static dev_type_ioctl(meson_com_ioctl);
static dev_type_tty(meson_com_tty);
static dev_type_poll(meson_com_poll);
static dev_type_stop(meson_com_stop);

const struct cdevsw mesoncom_cdevsw = {
	.d_open = meson_com_open,
	.d_close = meson_com_close,
	.d_read = meson_com_read,
	.d_write = meson_com_write,
	.d_ioctl = meson_com_ioctl,
	.d_stop = meson_com_stop,
	.d_tty = meson_com_tty,
	.d_poll = meson_com_poll,
	.d_mmap = nommap,
	.d_kqfilter = ttykqfilter,
	.d_discard = nodiscard,
	.d_flag = D_TTY
};

static int meson_com_cmajor = -1;

CFATTACH_DECL_NEW(meson_com, sizeof(struct meson_com_softc),
	meson_com_match, meson_com_attach, NULL, NULL);

static int
meson_com_match(device_t parent, cfdata_t cf, void *aux)
{
	struct fdt_attach_args * const faa = aux;

	return of_match_compatible(faa->faa_phandle, compatible);
}

static void
meson_com_attach(device_t parent, device_t self, void *aux)
{
	struct meson_com_softc * const sc = device_private(self);
	struct fdt_attach_args * const faa = aux;
	const int phandle = faa->faa_phandle;
	bus_addr_t addr, size;
	struct tty *tp;
	int major, minor;
	uint32_t misc, control;
	char intrstr[128];

	sc->sc_dev = self;
	sc->sc_bst = faa->faa_bst;
	if (fdtbus_get_reg(phandle, 0, &addr, &size) != 0)
		return;
	if(bus_space_map(sc->sc_bst, addr, size, 0, &sc->sc_bsh) != 0)
		return;
	if (!fdtbus_intr_str(phandle, 0, intrstr, sizeof(intrstr))) {
		aprint_error(": failed to decode interrupt\n");
		return;
	}
	sc->sc_ih = fdtbus_intr_establish(phandle, 0, IPL_SERIAL,
	    0, meson_com_intr, sc);
	if (sc->sc_ih == NULL) {
		aprint_error_dev(self, "failed to establish interrupt %s\n",
		    intrstr);
		return;
	}

	if (meson_com_cmajor == -1) {
		/* allocate a major number */
		int bmajor = -1, cmajor = -1;
		int error = devsw_attach("mesoncom", NULL, &bmajor,
		    &mesoncom_cdevsw, &cmajor);
		if (error) {
			aprint_error(": couldn't allocate major number\n");
			return;
		}
		meson_com_cmajor = cmajor;
	}

	major = cdevsw_lookup_major(&mesoncom_cdevsw);
	minor = device_unit(self);

	tp = sc->sc_tty = tty_alloc();
	tp->t_oproc = meson_com_start;
	tp->t_param = meson_com_param;
	tp->t_dev = makedev(major, minor);
	tp->t_sc = sc;
	tty_attach(tp);

	aprint_naive("\n");
	if (meson_com_is_console(addr)) {
		cn_tab->cn_dev = tp->t_dev;
		aprint_normal(": console");
	}
	aprint_normal("\n");

	aprint_normal_dev(self, "interrupting on %s\n", intrstr);

	misc = bus_space_read_4(sc->sc_bst, sc->sc_bsh, UART_MISC_REG);
	misc &= ~UART_MISC_TX_IRQ_CNT;
	misc |= __SHIFTIN(0, UART_MISC_TX_IRQ_CNT);
	misc &= ~UART_MISC_RX_IRQ_CNT;
	misc |= __SHIFTIN(1, UART_MISC_RX_IRQ_CNT);
	bus_space_write_4(sc->sc_bst, sc->sc_bsh, UART_MISC_REG, misc);

	control = bus_space_read_4(sc->sc_bst, sc->sc_bsh, UART_CONTROL_REG);
	control &= ~UART_CONTROL_TX_INT_EN;
	control |= UART_CONTROL_RX_INT_EN;
	bus_space_write_4(sc->sc_bst, sc->sc_bsh, UART_CONTROL_REG, control);
}

static int
meson_com_cngetc(dev_t dev)
{
	bus_space_tag_t bst = meson_com_cnsc.sc_bst;
	bus_space_handle_t bsh = meson_com_cnsc.sc_bsh;
	uint32_t status;
	int s, c;

	s = splserial();

	status = bus_space_read_4(bst, bsh, UART_STATUS_REG);
	if (status & UART_STATUS_RX_EMPTY) {
		splx(s);
		return -1;
	}

	c = bus_space_read_4(bst, bsh, UART_RFIFO_REG);
#if defined(DDB)
	extern int db_active;
	if (!db_active)
#endif
	{
		int cn_trapped __unused = 0;
		cn_check_magic(dev, c, meson_com_cnm_state);
	}

	splx(s);

	return c & 0xff;
}

static void
meson_com_cnputc(dev_t dev, int c)
{
	bus_space_tag_t bst = meson_com_cnsc.sc_bst;
	bus_space_handle_t bsh = meson_com_cnsc.sc_bsh;
	int timo = 150000;
	int s;

	s = splserial();

	while ((bus_space_read_4(bst, bsh, UART_STATUS_REG) & UART_STATUS_TX_FULL) != 0) {
		if (--timo == 0)
			break;
	}

	bus_space_write_4(bst, bsh, UART_WFIFO_REG, c);

	while ((bus_space_read_4(bst, bsh, UART_STATUS_REG) & UART_STATUS_TX_EMPTY) == 0) {
		if (--timo == 0)
			break;
	}
	splx(s);
}
	

static void
meson_com_cnpollc(dev_t dev, int on)
{
}

static bool
meson_com_cnattach(bus_space_tag_t bst, bus_space_handle_t bsh,
    int ospeed, tcflag_t cflag)
{
	struct meson_com_softc *sc = &meson_com_cnsc;

	cn_tab = &meson_com_consdev;
	cn_init_magic(&meson_com_cnm_state);
	cn_set_magic("\047\001");

	sc->sc_bst = bst;
	sc->sc_bsh = bsh;
	sc->sc_ospeed = ospeed;
	sc->sc_cflag = cflag;

	return true;
}

static bool
meson_com_is_console(bus_addr_t iobase)
{
	return iobase == CONSADDR;
}

static int
meson_com_open(dev_t dev, int flag, int mode, lwp_t *l)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));
	struct tty *tp = sc->sc_tty;

	if (kauth_authorize_device_tty(l->l_cred,
	    KAUTH_DEVICE_TTY_OPEN, tp) != 0) {
		return EBUSY;
	}

	if ((tp->t_state & TS_ISOPEN) == 0 && tp->t_wopen == 0) {
		tp->t_dev = dev;
		ttychars(tp);
		tp->t_iflag = TTYDEF_IFLAG;
		tp->t_oflag = TTYDEF_OFLAG;
		tp->t_cflag = TTYDEF_CFLAG;
		tp->t_lflag = TTYDEF_LFLAG;
		tp->t_ispeed = tp->t_ospeed = TTYDEF_SPEED;
		ttsetwater(tp);
	}
	tp->t_state |= TS_CARR_ON;

	return tp->t_linesw->l_open(dev, tp);
}

static int
meson_com_close(dev_t dev, int flag, int mode, lwp_t *l)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));
	struct tty *tp = sc->sc_tty;

	tp->t_linesw->l_close(tp, flag);
	ttyclose(tp);

	return 0;
}

static int
meson_com_read(dev_t dev, struct uio *uio, int flag)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));
	struct tty *tp = sc->sc_tty;

	return tp->t_linesw->l_read(tp, uio, flag);
}

static int
meson_com_write(dev_t dev, struct uio *uio, int flag)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));
	struct tty *tp = sc->sc_tty;

	return tp->t_linesw->l_write(tp, uio, flag);
}

static int
meson_com_poll(dev_t dev, int events, lwp_t *l)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));
	struct tty *tp = sc->sc_tty;

	return tp->t_linesw->l_poll(tp, events, l);
}

static int
meson_com_ioctl(dev_t dev, u_long cmd, void *data, int flag, lwp_t *l)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));
	struct tty *tp = sc->sc_tty;
	int error;

	error = tp->t_linesw->l_ioctl(tp, cmd, data, flag, l);
	if (error != EPASSTHROUGH)
		return error;

	return ttioctl(tp, cmd, data, flag, l);
}

static struct tty *
meson_com_tty(dev_t dev)
{
	struct meson_com_softc *sc =
	    device_lookup_private(&mesoncom_cd, minor(dev));

	return sc->sc_tty;
}

static void
meson_com_stop(struct tty *tp, int flag)
{
}

static void
meson_com_start(struct tty *tp)
{
	struct meson_com_softc *sc = tp->t_sc;
	u_char *p = sc->sc_buf;
	int s, brem;

	s = spltty();

	if (tp->t_state & (TS_TTSTOP | TS_BUSY | TS_TIMEOUT)) {
		splx(s);
		return;
	}
	tp->t_state |= TS_BUSY;

	splx(s);

	for (brem = q_to_b(&tp->t_outq, sc->sc_buf, sizeof(sc->sc_buf));
	     brem > 0;
	     brem--, p++) {
		while ((bus_space_read_4(sc->sc_bst, sc->sc_bsh,
		    UART_STATUS_REG) & UART_STATUS_TX_FULL) != 0)
			;

		bus_space_write_4(sc->sc_bst, sc->sc_bsh,
		    UART_WFIFO_REG, *p);
	}

	s = spltty();
	tp->t_state &= ~TS_BUSY;
	if (ttypull(tp)) {
		tp->t_state |= TS_TIMEOUT;
		callout_schedule(&tp->t_rstrt_ch, 1);
	}
	splx(s);
}

static int
meson_com_param(struct tty *tp, struct termios *t)
{

	tp->t_ispeed = t->c_ispeed;
	tp->t_ospeed = t->c_ospeed;
	tp->t_cflag = t->c_cflag;

	return 0;
}

static int
meson_com_intr(void *priv)
{
	struct meson_com_softc *sc = priv;
	struct tty *tp = sc->sc_tty;
	uint32_t status, c;

	for (;;) {
		int cn_trapped = 0;
		status = bus_space_read_4(sc->sc_bst, sc->sc_bsh,
		    UART_STATUS_REG);
		if (status & UART_STATUS_RX_EMPTY) {
			break;
		}
		if (status & UART_STATUS_BREAK) {
			cn_check_magic(tp->t_dev, CNC_BREAK,
			    meson_com_cnm_state);
			if (cn_trapped)
				continue;
		}

		c = bus_space_read_4(sc->sc_bst, sc->sc_bsh, UART_RFIFO_REG);
		cn_check_magic(tp->t_dev, c & 0xff, meson_com_cnm_state);
		if (cn_trapped)
			continue;
		tp->t_linesw->l_rint(c & 0xff, tp);
	}

	return 0;
}

/*
 * Console support
 */

static int
meson_com_console_match(int phandle)
{
	return of_match_compatible(phandle, compatible);
}

static void
meson_com_console_consinit(struct fdt_attach_args *faa, u_int uart_freq)
{
	const int phandle = faa->faa_phandle;
	bus_space_tag_t bst = faa->faa_a4x_bst;
	bus_addr_t addr;
	tcflag_t flags;
	int speed;

	fdtbus_get_reg(phandle, 0, &addr, NULL);
	speed = fdtbus_get_stdout_speed();
	if (speed < 0)
		speed = 115200;	/* default */
	flags = fdtbus_get_stdout_flags();

	if (!meson_com_cnattach(bst, addr, speed, flags))
		panic("Cannot initialize meson com console");
}

static const struct fdt_console meson_com_console = {
	.match = meson_com_console_match,
	.consinit = meson_com_console_consinit,
};

FDT_CONSOLE(meson_com, &meson_com_console);
