#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <netdb.h>

#include <arpa/inet.h>

#include <net/pfkeyv2.h>

#include <netinet/in.h>
#include <netinet/udp.h>

#include <netipsec/ipsec.h>

#include "libpfkey.h"

char keymat[] = {0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08};
char udp_payload[] = {
	0x01, 0x02, 0x03, 0x04, 0x5, 0x06, 0x07, 0x08,
	0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
	0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
	0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
};

const char *sa2_mode[] = {
	"IPSEC_MODE_ANY",	/* 0 */
	"IPSEC_MODE_TRANSPORT",	/* 1 */
	"IPSEC_MODE_TUNNEL",	/* 2 */
	"IPSEC_MODE_TCPMD5",	/* 3 */
};

void
help(void)
{
	printf("pfkey_test - generate IPsec NAT-T packet\n");
	printf("\n");
	printf("usage:\n");
	printf("\tpfkey_test <local_ipv4> <remote_ipv4>\n");
	printf("\n");
	printf("This program send IPsec NAT-T pakcet with udp port 4500.\n");
}

struct sadb_msg *
get_response(int so, int type, caddr_t *mhp)
{
	struct sadb_msg *msg;
	pid_t pid;
	int err;

	pid = getpid();
	while ( (msg = pfkey_recv(so)) != NULL) {
		if (msg->sadb_msg_pid != pid) {
			free(msg);
			continue;
		}
		if (msg->sadb_msg_type == type)
			break;
	}

	err = pfkey_align(msg, mhp);
	if (err) {
		fprintf(stderr,
		    "pfkey_align() failed: %s\n", ipsec_strerror());
		free(msg);
		return NULL;
	}
	err = pfkey_check(mhp); 
	if (err) {
		fprintf(stderr, 
		    "pfkey_check() failed: %s\n", ipsec_strerror());
		free(msg);
		return NULL;
	}

	return msg;
}

uint32_t
get_spi(int so, struct sockaddr *src, struct sockaddr *dst)
{
	struct sadb_msg *msg;
	struct sadb_sa *sa;
	caddr_t mhp[SADB_EXT_MAX + 1];
	unsigned int spi;
	static int seq = 1;
	int err;

	err = pfkey_send_getspi_nat(
	    so,
	    SADB_SATYPE_ESP,
	    IPSEC_MODE_TRANSPORT,
	    src,
	    dst,
	    UDP_ENCAP_ESPINUDP,
	    4500, /* src port */
	    4500, /* dst port */
	    0, /* minspi */
	    0, /* maxspi */
	    0, /* reqid */
	    seq);
	if (err < 0) {
		fprintf(stderr,
		    "pfkey_send_getspi_nat() failed: %s\n",
		    ipsec_strerror());
		return 0;
	}

	msg = get_response(so, SADB_GETSPI, mhp);
	if (msg == NULL)
		return 0;
	printf("got reponse(error=%d)\n", msg->sadb_msg_errno);

	if (mhp[SADB_X_EXT_NAT_T_SPORT] == NULL) {
		fprintf(stderr, "GETSPI: no SADB_X_EXT_NAT_T_SPORT\n");
	}
	if (mhp[SADB_X_EXT_NAT_T_DPORT] == NULL) {
		fprintf(stderr, "GETSPI: no SADB_X_EXT_NAT_T_DPORT\n");
	}
	sa = (struct sadb_sa *)mhp[SADB_EXT_SA];
	if (sa == NULL) {
		fprintf(stderr, "GETSPI: no SADB_EXT_SA\n");
		free(msg);
		return 0;
	}
	spi = sa->sadb_sa_spi;
	printf("GETSPI: SPI=0x%x\n", ntohl(spi));
	printf("GETSPI: NAT-T: %d\n",
	    PFKEY_ADDR_X_NATTYPE(mhp[SADB_X_EXT_NAT_T_TYPE]));

	free(msg);
	return spi;
}

int
update_sa(int so, struct sockaddr *src, struct sockaddr *dst, int spi)
{
	struct sadb_msg *msg;
	struct sadb_sa *sa;
	struct sadb_x_sa2 *sa2;
	caddr_t mhp[SADB_EXT_MAX + 1];
	static int seq = 1;
	int err;

	err = pfkey_send_update_nat(
	    so,
	    SADB_SATYPE_ESP,
	    IPSEC_MODE_TRANSPORT,
	    src,
	    dst,
	    spi,
	    0, /* reqid */
	    4, /* wsize */
	    keymat,
	    SADB_EALG_NULL, /* enc type */
	    8, /* enc keylen [bytes] */
	    SADB_AALG_NONE,
	    0, /* auth keylen */
	    0, /* flags */
	    0, /* l_alloc */
	    0, /* l_bytes */
	    time(NULL), /* l_addtime */
	    0, /* l_usetime */
	    seq,
	    UDP_ENCAP_ESPINUDP,
	    4500, /* src port */
	    4500, /* dst port */
	    NULL, /* NAT-OA */
	    0 /* NAT-T Frag */);
	if (err < 0) {
		fprintf(stderr,
		    "pfkey_send_update_nat() failed: %s\n",
		    ipsec_strerror());
		return -1;
	}

	msg = get_response(so, SADB_UPDATE, mhp);
	if (msg == NULL)
		return -1;
	printf("got reponse(error=%d)\n", msg->sadb_msg_errno);

	sa = (struct sadb_sa *)mhp[SADB_EXT_SA];
	if (sa == NULL) {
		fprintf(stderr, "UPDATE: no SADB_EXT_SA\n");
	}
	sa2 = (struct sadb_x_sa2 *)mhp[SADB_X_EXT_SA2];
	if (sa2 == NULL) {
		fprintf(stderr, "UPDATE: no SADB_X_EXT_SA2\n");
	}

	printf("got reponse(error=%d)\n", msg->sadb_msg_errno);
	printf("UPDATE: msg->sadb_msg_satype: %d\n", msg->sadb_msg_satype);
	printf("UPDATE: msg->sadb_sa_spi: 0x%x\n", sa->sadb_sa_spi);
	if (sa2) {
		printf("UPDATE: mode: %s(%d)\n",
		    sa2_mode[sa2->sadb_x_sa2_mode], sa2->sadb_x_sa2_mode);
	}
	printf("UPDATE: NAT-T: %d\n",
	    PFKEY_ADDR_X_NATTYPE(mhp[SADB_X_EXT_NAT_T_TYPE]));

	return 0;
}

int
send_udp(struct sockaddr *src0, struct sockaddr *dst0)
{
	struct sockaddr_in sin_src, sin_dst;
	struct sockaddr *src, *dst;
	int so;
	int err;

	memcpy(&sin_src, src0, src0->sa_len);
	memcpy(&sin_dst, dst0, dst0->sa_len);
	sin_src.sin_port = htons(1701);
	sin_dst.sin_port = htons(1701);
	src = (struct sockaddr *)&sin_src;
	dst = (struct sockaddr *)&sin_dst;
	so = socket(PF_INET, SOCK_DGRAM, 0);
	if (so < 0) {
		fprintf(stderr, "socket() failed: %s\n", strerror(errno));
		return -1;
	}
	err = bind(so, src, src->sa_len);
	if (err < 0) {
		fprintf(stderr, "bind() failed: %s\n", strerror(errno));
		close(so);
		return -1;
	}
	err = connect(so, dst, dst->sa_len);
	if (err < 0) {
		fprintf(stderr, "connect() failed: %s\n", strerror(errno));
		close(so);
		return -1;
	}
	err = send(so, udp_payload, sizeof(udp_payload), 0);
	if (err < 0) {
		fprintf(stderr, "connect() failed: %s\n", strerror(errno));
		close(so);
		return -1;
	}
	printf("UDP Message sent\n");
	close(so);
	return 0;
}

int
main(int argc, char *argv[])
{
	int so;
	struct sockaddr_in src;
	struct sockaddr_in dst;
	const char *local_address;
	const char *remote_address;
	int spi;
	int err;

	if (argc < 3) {
		help();
		return EXIT_FAILURE;
	}
	local_address = argv[1];
	remote_address = argv[2];

	memset(&src, 0, sizeof(src));
	src.sin_len = sizeof(src);
	src.sin_family = AF_INET;
	err = inet_pton(AF_INET, local_address, &src.sin_addr);
	if (err < 0) {
		fprintf(stderr,
		    "inet_pton(%s) failed: %s\n",
		    local_address, strerror(errno));
		return EXIT_FAILURE;
	}

	memset(&dst, 0, sizeof(dst));
	dst.sin_len = sizeof(dst);
	dst.sin_family = AF_INET;
	err = inet_pton(AF_INET, remote_address, &dst.sin_addr);
	if (err < 0) {
		fprintf(stderr,
		    "inet_pton(%s) failed: %s\n",
		   remote_address,  strerror(errno));
		return EXIT_FAILURE;
	}

	so = pfkey_open();
	if (so < 0) {
		fprintf(stderr,
		    "pfkey_open() failed: %s\n", strerror(errno));
		return EXIT_FAILURE;
	}

	spi = get_spi(so, (struct sockaddr *)&src, (struct sockaddr *)&dst);
	if (spi == 0) {
		fprintf(stderr, "get_spi() failed.\n");
		return EXIT_FAILURE;
	}
	err = update_sa(so,
	    (struct sockaddr *)&src, (struct sockaddr *)&dst, spi);
	if (err < 0) {
		fprintf(stderr, "update_sa() failed.\n");
		return EXIT_FAILURE;
	}
	err = send_udp((struct sockaddr *)&src, (struct sockaddr *)&dst);
	if (err < 0) {
		fprintf(stderr, "send_udp() failed.\n");
		return EXIT_FAILURE;
	}

	close(so);
	return EXIT_SUCCESS;
}
