# frozen_string_literal: true

require 'time'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      module ClickHouse
        class Table
          LOG_PREFIX = "[CodeCoverage]"

          def initialize(url:, database:, username: nil, password: nil, logger: nil)
            @url = url
            @database = database
            @username = username
            @password = password
            @logger = logger || Runtime::Logger.logger
          end

          # @param data [Array<Hash>] Code coverage related data to be pushed to ClickHouse
          # @return [nil]
          def push(data) # rubocop:disable Metrics/AbcSize
            return logger.warn("#{LOG_PREFIX} No data found, skipping ClickHouse export!") if data.empty?

            logger.debug("#{LOG_PREFIX} Starting data export to ClickHouse")
            sanitized_data = sanitize(data)

            return logger.warn("#{LOG_PREFIX} No valid data found after sanitization, skipping ClickHouse export!") if sanitized_data.empty?

            client.insert_json_data(table_name, sanitized_data)
            logger.info("#{LOG_PREFIX} Successfully pushed #{sanitized_data.size} records to #{full_table_name}!")
          rescue StandardError => e
            logger.error("#{LOG_PREFIX} Error occurred while pushing data to #{full_table_name}: #{e.message}")
            raise
          end

          private

          attr_reader :url, :database, :username, :password, :logger

          def sanitize(data)
            data.filter_map { |record| sanitized_data_record(record) if valid_record?(record) }
          end

          def sanitized_data_record(record)
            record
          end

          def full_table_name
            "#{database}.#{table_name}"
          end

          def table_name
            self.class::TABLE_NAME
          rescue NameError
            raise NotImplementedError, "#{self.class} must define the TABLE_NAME constant"
          end

          def valid_record?(_record)
            raise NotImplementedError, "#{self.class}##{__method__} method must be implemented in a subclass"
          end

          # @return [Time] Common timestamp for all records, memoized
          def time
            @time ||= parse_ci_timestamp
          end

          def parse_ci_timestamp
            ci_created_at = ENV.fetch('CI_PIPELINE_CREATED_AT', nil)
            return Time.now.utc unless ci_created_at

            Time.strptime(ci_created_at, '%Y-%m-%dT%H:%M:%S%z')
          rescue ArgumentError
            logger.warn("#{LOG_PREFIX} Invalid CI_PIPELINE_CREATED_AT format: #{ci_created_at}, using current time")
            Time.now.utc
          end

          # @return [GitlabQuality::TestTooling::ClickHouse::Client]
          def client
            @client ||= GitlabQuality::TestTooling::ClickHouse::Client.new(
              url: url,
              database: database,
              username: username,
              password: password,
              logger: logger
            )
          end
        end
      end
    end
  end
end
