# {%= name %} {%= badge('npm') %} {%= badge('downloads') %} {%= badge('travis') %}

> {%= description %}

Micromatch supports all of the same matching features as [minimatch][] and [multimatch][]. 

* [mm()](#usage) is the same as [multimatch()][multimatch]
* [mm.match()](#match) is the same as [minimatch.match()][minimatch]
* use [mm.isMatch()](#ismatch) instead of [minimatch()][minimatch]

## Install
{%= include("install-npm", {save: true}) %}

## Start matching!

```js
var mm = require('{%= name %}');
console.log(mm(['']))
```

## Table of contents

<!-- toc -->

***


### Features

+ [Drop-in replacement][switch] for [minimatch][] and [multimatch][]
+ Built-in support for multiple glob patterns, like `['foo/*.js', '!bar.js']`
+ [Brace Expansion][braces] (`foo/bar-{1..5}.md`, `one/{two,three}/four.md`)
+ Typical glob patterns, like `**/*`, `a/b/*.js`, or `['foo/*.js', '!bar.js']`
+ Methods like `.isMatch()`, `.contains()` and `.any()` 

**Extended globbing features:**

+ Logical `OR` (`foo/bar/(abc|xyz).js`)
+ Regex character classes (`foo/bar/baz-[1-5].js`)
+ POSIX [bracket expressions][expand-brackets] (`**/[[:alpha:][:digit:]]/`)
+ [extglobs][extglob] (`**/+(x|y)`, `!(a|b)`, etc). 

You can combine these to create whatever matching patterns you need.

**Example**

```js
// double-negation!
mm(['fa', 'fb', 'f', 'fo'], '!(f!(o))');
//=> ['fo']
```

## Why switch to micromatch?

- Native support for multiple glob patterns, no need for wrappers like [multimatch][]
- [10-55x faster](#benchmarks) and more performant than [minimatch][] and [multimatch][]. This is achieved through a combination of caching and regex optimization strategies, a fundamentally different approach than minimatch.
- More extensive support for the Bash 4.3 specification
- More complete extglob support
- Extensive [unit tests](./test) (approx. 1,300 tests). Minimatch fails many of the tests.


### Switch from minimatch

Use `mm.isMatch()` instead of `minimatch()`:

```js
mm.isMatch('foo', 'b*');
//=> false
```

Use `mm.match()` instead of `minimatch.match()`:

```js
mm.match(['foo', 'bar'], 'b*');
//=> 'bar'
```

### Switch from multimatch

Same signature:

```js
mm(['foo', 'bar', 'baz'], ['f*', '*z']);
//=> ['foo', 'baz']
```


***


## Usage

Add micromatch to your node.js project:

```js
var mm = require('{%= name %}');
```

**Signature**

```js
mm(array_of_strings, glob_patterns[, options]);
```

**Example**


```js
mm(['foo', 'bar', 'baz'], 'b*');
//=> ['bar', 'baz']
```


### Usage examples

**Brace expansion**

Match files with `.js` or `.txt` extensions.

```js
mm(['a.js', 'b.md', 'c.txt'], '*.{js,txt}');
//=> ['a.js', 'c.txt']
```

**Extglobs**

Match anything except for files with the `.md` extension.

```js
mm(files, '**/*.!(md)');

//=> ['a.js', 'c.txt']
```

**Multiple patterns**

Match using an array of patterns.

```js
mm(['a.md', 'b.js', 'c.txt', 'd.json'], ['*.md', '*.txt']);
//=> ['a.md', 'c.txt']
```

**Negation patterns:**

Behavior is designed to be what users would expect, based on conventions that are already well-established.

- [minimatch][] behavior is used when the pattern is a string, so patterns are **inclusive by default**. 
- [multimatch][] behavior is used when an array of patterns is passed, so patterns are **exclusive by default**.

```js
mm(['a.js', 'b.md', 'c.txt'], '!*.{js,txt}');
//=> ['b.md']

mm(['a.md', 'b.js', 'c.txt', 'd.json'], ['*.*', '!*.{js,txt}']);
//=> ['a.md', 'd.json']
```

***


## API methods

```js
var mm = require('micromatch');
```

### .match

```js
mm.match(array, globString);
```


Return an array of files that match the given glob pattern. Useful if you only need to use a single glob pattern.

**Example**

```js
mm.match(['ab', 'a/b', 'bb', 'b/c'], '?b');
//=> ['ab', 'bb']

mm.match(['ab', 'a/b', 'bb', 'b/c'], '*/b');
//=> ['a/b']
```

### .isMatch

```js
mm.isMatch(filepath, globString);
```

Returns true if a file path matches the given glob pattern.


**Example**

```js
mm.isMatch('.verb.md', '*.md');
//=> false

mm.isMatch('.verb.md', '*.md', {dot: true});
//=> true
```

### .contains

Returns true if any part of a file path matches the given glob pattern. Think of this is "has path" versus "is path".

**Example**

`.isMatch()` would return false for both of the following:

```js
mm.contains('a/b/c', 'a/b');
//=> true

mm.contains('a/b/c', 'a/*');
//=> true
```

### .matcher

Returns a function for matching using the supplied pattern. e.g. create your own "matcher". The advantage of this method is that the pattern can be compiled outside of a loop.

**Pattern**

Can be any of the following:

- `glob/string`
- `regex`
- `function`

**Example**

```js
var isMatch = mm.matcher('*.md');
var files = [];

['a.md', 'b.txt', 'c.md'].forEach(function(fp) {
  if (isMatch(fp)) {
    files.push(fp);
  }
});
```

### .filter

Returns a function that can be passed to `Array#filter()`. 

**Params**

- `patterns` **{String|Array}**: 

**Examples**

Single glob:

```js
var fn = mm.filter('*.md');
['a.js', 'b.txt', 'c.md'].filter(fn);
//=> ['c.md']

var fn = mm.filter('[a-c]');
['a', 'b', 'c', 'd', 'e'].filter(fn);
//=> ['a', 'b', 'c']
```

Array of glob patterns:

```js
var arr = [1,2,3,4,5,6,7,8,9,10,11,12,13,14,15];

var fn = mm.filter(['{1..10}', '![7-9]', '!{3..4}']);
arr.filter(fn);
//=> [1, 2, 5, 6, 10]
```

_(Internally this function generates the matching function by using the [matcher](#matcher) method. You can use the [matcher](#matcher) method directly to create your own filter function)_


### .any

Returns true if a file path matches any of the given patterns.

```js
mm.any(filepath, patterns, options);
```

**Params**

- filepath `{String}`: The file path to test.
- patterns `{String|Array}`: One or more glob patterns
- options: `{Object}`: options to pass to the `.matcher()` method.


**Example**

```js
mm.any('abc', ['!*z']);
//=> true
mm.any('abc', ['a*', 'z*']);
//=> true
mm.any('abc', 'a*');
//=> true
mm.any('abc', ['z*']);
//=> false
```


### .expand

Returns an object with a regex-compatible string and tokens.

```js
mm.expand('*.js');

// when `track` is enabled (for debugging), the `history` array is used
// to record each mutation to the glob pattern as it's converted to regex
{ options: { track: false, dot: undefined, makeRe: true, negated: false },
  pattern: '(.*\\/|^)bar\\/(?:(?!(?:^|\\/)\\.).)*?',
  history: [],
  tokens:
   { path:
      { whole: '**/bar/**',
        dirname: '**/bar/',
        filename: '**',
        basename: '**',
        extname: '',
        ext: '' },
     is:
      { glob: true,
        negated: false,
        globstar: true,
        dotfile: false,
        dotdir: false },
     match: {},
     original: '**/bar/**',
     pattern: '**/bar/**',
     base: '' } }
```

### .makeRe

Create a regular expression for matching file paths based on the given pattern:

```js
mm.makeRe('*.js');
//=> /^(?:(?!\.)(?=.)[^/]*?\.js)$/
```

## Options

### options.unixify

Normalize slashes in file paths and glob patterns to forward slashes.

Type: `{Boolean}`

Default: `undefined` on non-windows, `true` on windows.


### options.dot

Match dotfiles. Same behavior as [minimatch].

Type: `{Boolean}`

Default: `false`


### options.unescape

Unescape slashes in glob patterns. Use cautiously, especially on windows.

Type: `{Boolean}`

Default: `undefined`

**Example**

```js
mm.isMatch('abc', '\\a\\b\\c', {unescape: true});
//=> true
```

### options.nodupes

Remove duplicate elements from the result array.

Type: `{Boolean}`

Default: `undefined`

**Example**

Example of using the `unescape` and `nodupes` options together:

```js
mm.match(['abc', '\\a\\b\\c'], '\\a\\b\\c', {unescape: true});
//=> ['abc', 'abc']

mm.match(['abc', '\\a\\b\\c'], '\\a\\b\\c', {unescape: true, nodupes: true});
//=> ['abc']
```

### options.matchBase

Allow glob patterns without slashes to match a file path based on its basename. . Same behavior as [minimatch].

Type: `{Boolean}`

Default: `false`

**Example**

```js
mm(['a/b.js', 'a/c.md'], '*.js');
//=> []

mm(['a/b.js', 'a/c.md'], '*.js', {matchBase: true});
//=> ['a/b.js']
```

### options.nobraces

Don't expand braces in glob patterns. Same behavior as [minimatch][] `nobrace`.

Type: `{Boolean}`

Default: `undefined`

See [braces][] for more information about extended brace expansion.


### options.nobrackets

Don't expand POSIX bracket expressions. 

Type: `{Boolean}`

Default: `undefined`

See [expand-brackets][] for more information about extended bracket expressions.


### options.noextglob

Don't expand extended globs.

Type: `{Boolean}`

Default: `undefined`

See [extglob][] for more information about extended globs.


### options.nocase

Use a case-insensitive regex for matching files. Same behavior as [minimatch][].

Type: `{Boolean}`

Default: `false`


### options.nonegate

Disallow negation (`!`) patterns.

Type: `{Boolean}`

Default: `false`


### options.nonull

If `true`, when no matches are found the actual (array-ified) glob pattern is returned instead of an empty array. Same behavior as [minimatch][].

Type: `{Boolean}`

Default: `false`


### options.cache

Cache the platform (e.g. `win32`) to prevent this from being looked up for every filepath.

Type: `{Boolean}`

Default: `true`


***

## Other features

Micromatch also supports the following.

### Extended globbing

#### extglobs

Extended globbing, as described by the bash man page:

| **pattern** | **regex equivalent** | **description** |
| --- | --- | --- |
| `?(pattern-list)` | `(...|...)?` |  Matches zero or one occurrence of the given patterns |
| `*(pattern-list)` | `(...|...)*` |  Matches zero or more occurrences of the given patterns |
| `+(pattern-list)` | `(...|...)+` |  Matches one or more occurrences of the given patterns |
| `@(pattern-list)` | `(...|...)` <sup>*</sup> |  Matches one of the given patterns |
| `!(pattern-list)` | N/A |  Matches anything except one of the given patterns |

<sup><strong>*</strong></sup> `@` isn't a RegEx character.

Powered by [extglob][]. Visit that library for the full range of options or to report extglob related issues.

See [extglob][] for more information about extended globs.

#### brace expansion

In simple cases, brace expansion appears to work the same way as the logical `OR` operator. For example, `(a|b)` will achieve the same result as `{a,b}`.

Here are some powerful features unique to brace expansion (versus character classes):

 - range expansion: `a{1..3}b/*.js` expands to: `['a1b/*.js', 'a2b/*.js', 'a3b/*.js']`
 - nesting: `a{c,{d,e}}b/*.js` expands to: `['acb/*.js', 'adb/*.js', 'aeb/*.js']`


Visit [braces][braces] to ask questions and create an issue related to brace-expansion, or to see the full range of features and options related to brace expansion.


#### regex character classes

With the exception of brace expansion (`{a,b}`, `{1..5}`, etc), most of the special characters convert directly to regex, so you can expect them to follow the same rules and produce the same results as regex.

For example, given the list: `['a.js', 'b.js', 'c.js', 'd.js', 'E.js']`:

 - `[ac].js`: matches both `a` and `c`, returning `['a.js', 'c.js']`
 - `[b-d].js`: matches from `b` to `d`, returning `['b.js', 'c.js', 'd.js']`
 - `[b-d].js`: matches from `b` to `d`, returning `['b.js', 'c.js', 'd.js']`
 - `a/[A-Z].js`: matches and uppercase letter, returning `['a/E.md']`

Learn about [regex character classes][character-classes].

#### regex groups

Given `['a.js', 'b.js', 'c.js', 'd.js', 'E.js']`:

 - `(a|c).js`: would match either `a` or `c`, returning `['a.js', 'c.js']`
 - `(b|d).js`: would match either `b` or `d`, returning `['b.js', 'd.js']`
 - `(b|[A-Z]).js`: would match either `b` or an uppercase letter, returning `['b.js', 'E.js']`

As with regex, parenthese can be nested, so patterns like `((a|b)|c)/b` will work. But it might be easier to achieve your goal using brace expansion.

#### POSIX bracket expressions

**Example**

```js
mm.isMatch('a1', '[[:alpha:][:digit:]]');
//=> true
```

See [expand-brackets][] for more information about extended bracket expressions.


***

## Notes

Whenever possible parsing behavior for patterns is based on globbing specifications in Bash 4.3. Patterns that aren't described by Bash follow wildmatch spec (used by git).


## Benchmarks

Run the [benchmarks](./benchmark):

```bash
node benchmark
```

As of {%= date() %}:

```bash
{%= docs("benchmark/last.md") %}
```

## Tests

### Running tests
{%= include("tests") %}

### Coverage
As of {%= date() %}:

```sh
Statements : 100% (441/441)
Branches   : 100% (270/270)
Functions  : 100% (54/54)
Lines      : 100% (429/429)
```

## Contributing
{%= include("contributing") %}

Please be sure to run the benchmarks before/after any code changes to judge the impact before you do a PR. thanks!

## Related 
{%= related(verb.related.list) %}

## Contributing
{%= include("contributing") %}

## Building docs
{%= include("build-docs") %}

## Running tests
{%= include("tests") %}

## Author
{%= include("author") %}

## License
{%= copyright({linkify: true}) %}
{%= license %}

***

{%= include("footer") %}

{%= reflinks(verb.reflinks) %}


[switch]: #switch-from-minimatch
[expand]: #expand
[character-classes]: http://www.regular-expressions.info/charclass.html
[extended]: http://mywiki.wooledge.org/BashGuide/Patterns#Extended_Globs
