=begin
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
=end

# $Id: european_option.rb,v 1.4 2002/01/16 15:17:06 nando Exp $

require 'QuantLib'
require 'runit/testcase'
require 'runit/testsuite'
require 'runit/cui/testrunner'

def relErr(x1,x2,reference)
    if reference != 0.0
        (x1-x2).abs/reference
    else
        1.0e+10
    end
end

def pricer(type,underlying,strike,divCurve,rfCurve,exDate,volatility)
    QuantLib::PlainOption.new(type,underlying,strike,divCurve,rfCurve,
        exDate,volatility,QuantLib::EuropeanEngine.new)
end

def quote(value)
    h = QuantLib::MarketElementHandle.new
    h.linkTo QuantLib::SimpleMarketElement.new(value)
    return h
end

def flatCurve(forward)
    h = QuantLib::TermStructureHandle.new
    h.linkTo QuantLib::FlatForward.new('EUR',
                                       QuantLib::DayCounter.new('act/360'),
                                       QuantLib::Date.new(12,10,2001),
                                       QuantLib::Calendar.new('TARGET'), 2,
                                       forward)
    return h
end


class EuropeanOptionTest < RUNIT::TestCase
    include QuantLib
    def name
        "Testing European option pricer..."
    end
    def test
        # errors allowed for Greeks
        error = { 'delta' => 5.0e-5,
                  'gamma' => 5.0e-5,
                  'theta' => 6.0e-5,
                  'rho'   => 5.0e-5,
                  'dividendRho' => 5.0e-5,
                  'vega'  => 5.0e-5
                }

        [100].each { |under|
            dS = under/10000.0
            underlying  = quote(under)
            underlyingP = quote(under+dS)
            underlyingM = quote(under-dS)

        [0.04, 0.05, 0.06].each { |qRate|
            dQ = qRate/10000.0
            divCurve  = flatCurve(qRate)
            divCurveP = flatCurve(qRate+dQ)
            divCurveM = flatCurve(qRate-dQ)

        [Date.new(16,10,2002)].each { |exDate|
            yf = DayCounter.new('act/360').yearFraction(exDate-1,exDate+1)

        [0.01, 0.05, 0.15].each     { |rRate|
            dR = rRate/10000.0
            rfCurve  = flatCurve(rRate)
            rfCurveP = flatCurve(rRate+dR)
            rfCurveM = flatCurve(rRate-dR)

        [0.11, 0.5, 1.2].each { |vol|
            dVol = vol/10000.0
            volatility  = quote(vol)
            volatilityP = quote(vol+dVol)
            volatilityM = quote(vol-dVol)

        [50, 99.5, 100, 100.5, 150].each { |strike|
        ['Call','Put','Straddle'].each   { |type|

            #Check Greeks
            dT = 1
            opt = pricer(type,underlying,strike,
                         divCurve,rfCurve,exDate,volatility)
            opt_val = opt.NPV
            if opt_val > 1.0e-5*under
                optPs = pricer(type, underlyingP, strike, divCurve,
                               rfCurve,  exDate ,   volatility)
                optMs = pricer(type, underlyingM, strike, divCurve,
                               rfCurve,  exDate ,   volatility)
                optPt = pricer(type, underlying , strike, divCurve,
                               rfCurve,  exDate+dT,  volatility)
                optMt = pricer(type, underlying , strike, divCurve,
                               rfCurve,  exDate-dT,  volatility)
                optPr = pricer(type, underlying , strike, divCurve,
                               rfCurveP, exDate   , volatility)
                optMr = pricer(type, underlying , strike, divCurve,
                               rfCurveM, exDate   , volatility)
                optPq = pricer(type, underlying , strike, divCurveP,
                               rfCurve,  exDate   , volatility)
                optMq = pricer(type, underlying , strike, divCurveM,
                               rfCurve,  exDate   , volatility)
                optPv = pricer(type, underlying , strike, divCurve,
                               rfCurve,  exDate   , volatilityP)
                optMv = pricer(type, underlying , strike, divCurve,
                               rfCurve,  exDate   , volatilityM)

                # numeric values
                results = {
                    'delta' =>  (optPs.NPV - optMs.NPV)/(2*dS),
                    'gamma' =>  (optPs.delta - optMs.delta)/(2*dS),
                    'theta' => -(optPt.NPV - optMt.NPV)/yf,
                    'rho'   =>  (optPr.NPV - optMr.NPV)/(2*dR),
                    'dividendRho' => (optPq.NPV - optMq.NPV)/(2*dQ),
                    'vega' =>   (optPv.NPV - optMv.NPV)/(2*dVol)
                }

                ['delta','gamma','theta','rho','dividendRho','vega'].each { |greek|
                    unless relErr(opt.send(greek),results[greek],under) <= error[greek]
                        assert_fail(<<-MESSAGE

    Option details: #{type} #{under} #{strike} #{qRate} #{rRate} #{exDate} #{vol}
        value  = #{opt_val}
        #{greek} = #{opt.send(greek)}, #{greek}Num = #{results[greek]}

                            MESSAGE
                        )
                    end
                }
            end
        }}}}}}}
    end
end

if $0 == __FILE__
    RUNIT::CUI::TestRunner.run(EuropeanOptionTest.suite)
end

