"""
A persistent 4RDF model driver using more recent (development) rdflib store API

see:

- http://svn.rdflib.net/trunk/
- http://rdflib.net/store/
- http://rdflib.net/store/#ConjunctiveGraph
- http://rdflib.net/store/#Formula
- http://rdflib.net/Graph/

This is the first step in facilitating the move from 4rdf to rdlib:

see: http://lists.fourthought.com/pipermail/4suite-dev/2005-September/002021.html

For the most part, this is an interface mapping from Ft.Rdf.Model to the rdflib store API

Finally, it's also worth noting that rdflib does not support reification (the persistence of statement identity for reference).
The store interface provides a framework for persisting 'quoted'/'hypothetical' sets of statements (N3 formulae) which are somewhat analagous.

see: http://chatlogs.planetrdf.com/swig/2005-12-13.html#T16-40-05

Copyright 2005 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import os, re,sys
from Ft.Rdf.Drivers import PROPERTIES
from Ft.Lib.DbUtil import EscapeQuotes
from Ft.Lib.Set import Unique
from Ft.Rdf import Statement, Model, OBJECT_TYPE_UNKNOWN, OBJECT_TYPE_LITERAL, BNODE_BASE
from Ft.Rdf.Drivers import DataBaseExceptions
from Ft import GetConfigVar

VERSION = "1.0"

REGEX_FLAGS = [Model.IGNORE_CASE,Model.REGEX,Model.IGNORE_CASE|Model.REGEX]

def InitializeModule():
    """
    Post-import hook to initialize module's runtime variables that are not
    required at import time, but will be needed before the module-level
    functions are called.
    """
    global Namespace,plugin, Variable, Literal,URIRef,BNode,QuotedGraph,ConjunctiveGraph,AuditableStorage,REGEXMatching, REGEXTerm,rdflib,Graph,Store,DUMMY_MODEL,mapRdfLibQuad,mapFtRdfStatements,mapFtRdfTerm,REGEX_FLAGS,NATIVE_REGEX, PYTHON_REGEX,BNODE_PATTERN
    DUMMY_MODEL = Model.Model(None)
    BNODE_PATTERN = re.compile(BNODE_BASE)

    try:
        from rdflib import BNode, Literal, URIRef, plugin, Namespace, Variable
        from rdflib.Graph import Graph, QuotedGraph, ConjunctiveGraph
        from rdflib.store import Store
        from rdflib.store.REGEXMatching import REGEXTerm, REGEXMatching, NATIVE_REGEX, PYTHON_REGEX
        from rdflib.store.AuditableStorage import AuditableStorage
    except ImportError:
        raise ImportError("Unable to import rdflib.  Is it installed? (http://rdflib.net)")

def mapRdfLibObject(obj):
    if isinstance(obj,Literal):
        return obj,OBJECT_TYPE_LITERAL
    else:
        return obj,OBJECT_TYPE_UNKNOWN

def mapRdfLibQuad((subject,predicate,object_,context)):
    if isinstance(object_,Literal):
        objType=Model.OBJECT_TYPE_LITERAL
    else:
        objType=Model.OBJECT_TYPE_RESOURCE
    context = (isinstance(context,Graph) or isinstance(context,QuotedGraph)) and context.identifier or context
    return subject,predicate,object_,u'',context,objType

def mapFtRdfTerm(ftTerm,notBNode = False):
    if ftTerm is None or (isinstance(ftTerm,str) or isinstance(ftTerm,unicode)) and not ftTerm:
        return None
    elif not notBNode and isinstance(ftTerm,BNode) or BNODE_PATTERN.match(ftTerm):# DUMMY_MODEL.isBnodeLabel(ftTerm):
        return BNode(ftTerm)
    else:
        return URIRef(ftTerm)

def mapFtRdfStatements((subject,predicate,object_,statementUri,scope,otype),g,flags=None):
    regex_needed = not hasattr(g.store,'regex_matching')
    
    flags=flags and flags or {}

    #Convert subject string / flags to rdflib term
    if flags and flags.get('subjectFlags') in REGEX_FLAGS:
        if flags.get('subjectFlags') in [Model.REGEX|Model.IGNORE_CASE,Model.IGNORE_CASE] and (regex_needed or g.store.regex_matching == PYTHON_REGEX):
            subj = REGEXTerm('(?i)'+subject)
        else:
            subj = REGEXTerm(subject)        
    else:
        subj = mapFtRdfTerm(subject)

    #Convert predicate string / flags to rdflib term        
    if flags and flags.get('predicateFlags') in REGEX_FLAGS:
        if flags.get('predicateFlags') in [Model.REGEX|Model.IGNORE_CASE,Model.IGNORE_CASE] and (regex_needed or g.store.regex_matching == PYTHON_REGEX):
            pred = REGEXTerm('(?i)'+predicate)
        else:
            pred = REGEXTerm(predicate)
    else:
        pred=mapFtRdfTerm(predicate,notBNode=True)

    #Convert object string/type/flags to rdflib term
    if otype == Model.OBJECT_TYPE_LITERAL:
        obj = Literal(object_)
    elif flags and flags.get('objectFlags') in REGEX_FLAGS:
        if flags.get('objectFlags') in [Model.REGEX|Model.IGNORE_CASE,Model.IGNORE_CASE] and (regex_needed or g.store.regex_matching == PYTHON_REGEX):
            obj = REGEXTerm('(?i)'+object_)
        else:
            obj = REGEXTerm(object_)
    else:
        obj=mapFtRdfTerm(object_)

    #Convert scope/flags to rdflib term
    if flags.get('scopeFlags') in REGEX_FLAGS:
        if flags.get('scopeFlags') in [Model.REGEX|Model.IGNORE_CASE,Model.IGNORE_CASE] and (regex_needed or g.store.regex_matching == PYTHON_REGEX):
            context = REGEXTerm('(?i)'+scope)
        else:
            context = REGEXTerm(scope)
    else:
        context = mapFtRdfTerm(scope,notBNode=True)

    return subj,pred,obj,context

def _initializeStore(store,closedWorld):
    return plugin.get(store,Store)(identifier=mapFtRdfTerm(closedWorld))

def readRdflibDbParameters(db_param):
    #db parameter format:
    #two parameters, seperated by '#':
    # - database path / database configuration string
    # - store implementation:
    #   MySQL,IOMemory,Memory,SleepyCat,Redland,...
    try:
        config,store=db_param.split('#')
        return config,store
    except:
        raise Exception("Malformed rdflib database parameter string: %s"%db_param)

def GetDb(rdflib_params,closedWorld=None):
    connectionString,store = readRdflibDbParameters(rdflib_params)
    store = _initializeStore(store,closedWorld)
    if not store.transaction_aware:
        store = AuditableStorage(store)
    g=ConjunctiveGraph(store)
    g.open(connectionString,create=False)
    return DbAdapter(g)

def CreateDb(rdflib_params,closedWorld=None):
    connectionString,store = readRdflibDbParameters(rdflib_params)
    store = _initializeStore(store,closedWorld)
    if not store.transaction_aware:
        store = AuditableStorage(store)
    g=ConjunctiveGraph(store)
    g.open(connectionString)
    return DbAdapter(g)

def ExistsDb(rdflib_params,closedWorld=None):
    connectionString,store = readRdflibDbParameters(rdflib_params)
    exists=-1
    try:
        store = _initializeStore(store,closedWorld)
        g=Graph(store)
        exists = g.store.open(connectionString,create=False)
        #Accounts for open implementations that give more information about the state of a store
        if isinstance(exists,type(None)):
            exists = 0
            len(g)
            exists = 1
        g.close()
    except Exception, e:
        sys.stderr.write("Closed World (%s) doesn't exist: %s\n" % (closedWorld,e));
    return exists

def DestroyDb(rdflib_params,closedWorld=None):
    connectionString,store = readRdflibDbParameters(rdflib_params)
    store = _initializeStore(store,closedWorld)    
    g=Graph(store)
    try:
        g.open(connectionString,create=False)
        g.destroy(connectionString)
        g.close()        
    except Exception,e:
        sys.stderr.write("Unable to destroy rdflib store %s (%s)\n%s\n" % (closedWorld,connectionString,e));

class RdflibAdapter:
    def depthFirstGraphPurge(self,graph):
        """
        Recursively removes all formulae (QuotedGraphs) referenced within the given scope (named graph).
        This is particulary useful for clearing a top level graph which expresses a rule set -
        in which case, the antecendants and consequents are formulae which will contain quoted
        / hypothetical statements that must be erased recursively downward.
        """
        #print "Purging context/formula %s"%graph.identifier
        for s,p,o in graph:
            #print "processing triple %s|%s|%s"%(s,p,o)
            if isinstance(s,QuotedGraph):
                self.depthFirstGraphPurge(s)
            if isinstance(s,QuotedGraph):
                self.depthFirstGraphPurge(o)
            #print "removing triple %s|%s|%s"%(s,p,o)
            graph.remove((s,p,o))
    
    def hypothesize(self,scope):
        """
        This function 'converts' a set of quoted (hypothesized) N3 statements, reified via http://www.w3.org/2000/10/swap/reify#
        vocabulary, into a rdflib.Graph.QuotedGraph / Formula (http://rdflib.net/store/#Formula).  It essentially ensures that hypothetical statements
        are distinct from asserted statements (in rdflib store implementations that support formulae
        partitioning).  It also creates universally quantified rdflib.Variable instances for resources whose URIs begin with ? - ala
        Notation 3 syntax:
        
        http://lists.w3.org/Archives/Public/public-cwm-talk/2005OctDec/0007.html
        
        Note, the 4Suite RDF N3 / n3p parser reifies all quoted statements it comes across.    
        """
        assert self._g.store.formula_aware
        NR3 = Namespace("http://www.w3.org/2000/10/swap/reify#")
        graph = self.getGraph(scope)
        formulae = {}
        for formulaId,pred,quotedStmt in graph.triples((None,NR3['statement'],None)):
            formulae[formulaId]=None
            triple = []
            for stmtPart in [NR3['subject'],NR3['predicate'],NR3['object']]:
                for quotedStmt,stmtPart,value in graph.triples((quotedStmt,stmtPart,None)):                    
                    if value[0] == '?':
                        value = Variable(value)                        
                        
                    triple.append(value)
                    #print "removing %s from %s"%(((quotedStmt,stmtPart,value)),graph)
                    graph.remove((quotedStmt,stmtPart,value))
            #print "removing %s from %s"%(((formulaId,pred,quotedStmt)),graph)
            graph.remove((formulaId,pred,quotedStmt))            
            triple = tuple(triple)                        
            formula = QuotedGraph(self._g.store,mapFtRdfTerm(formulaId))
            formula.add(triple)
            #print "adding %s to %s"%(triple,formula)

        formulaTriples = []
        for formulaId in formulae.keys():            
            for formulaId, predicate, object_ in graph.triples((formulaId,None,None)):
                formulaTriples.append((formulaId, predicate, object_))
                graph.remove((formulaId, predicate, object_))
                #print "removing %s from %s"%(((formulaId, predicate, object_)),graph)
            for subject, predicate, formulaId in graph.triples((None,None,formulaId)):
                formulaTriples.append((subject, predicate, formulaId))
                graph.remove((subject, predicate, formulaId))
                #print "removing %s from %s"%(((subject, predicate, formulaId)),graph)
        #from pprint import pprint;pprint(formulaTriples)
        for subject, predicate, object_ in formulaTriples:
            #print subject, formulae.keys()
            subject = subject in formulae.keys() and QuotedGraph(self._g.store,mapFtRdfTerm(subject)) or subject
            object_ = object_ in formulae.keys() and QuotedGraph(self._g.store,mapFtRdfTerm(object_)) or object_
            graph.add((subject,predicate,object_))
            #print "adding %s to %s"%((subject,predicate,object_),graph)

    def getGraph(self,scope=None,store=None):
        if scope:
            return Graph(store is not None and store or self._g.store,identifier=isinstance(scope,REGEXTerm) and scope or mapFtRdfTerm(scope))
        elif store is not None:
            return ConjunctiveGraph(store)
        else:
            return self._g    

class DbAdapter(RdflibAdapter):
    def __init__(self, g):
        self._g = g
        self.props = {PROPERTIES.OBJECT_TYPE_SUPPORTED: 1}
        
    def begin(self):
        pass

    def commit(self):
        self._g.commit()
        self._g.close()

    def rollback(self):
        self._g.rollback()        
        self._g.close()

    def add(self, statements):
        # Takes a list of tuples [(s, p, o, stmtUri, srcUri, otype), ...]
        for stmt in statements:
            subj,predicate,object_,context = mapFtRdfStatements(stmt,self._g)
            self.getGraph(context).add((subj,predicate,object_)) 

    def remove(self, statements):
        for s in statements:
            self.removePattern(s[0], s[1], s[2], s[3], s[4], {})

    def removePattern(self, subject, predicate, object_, statementUri, scope, flags):
        subject,predicate,object_,context = mapFtRdfStatements((subject,predicate,object_,statementUri,scope,None),self._g,flags)
        regex_needed = (isinstance(subject,REGEXTerm) or \
           isinstance(predicate,REGEXTerm) or \
           isinstance(object_,REGEXTerm) or \
           isinstance(context,REGEXTerm)) and not hasattr(self._g.store,'regex_matching')        
        targetStore = regex_needed and REGEXMatching(self._g.store) or self._g.store        
        self.getGraph(context,store=targetStore).remove((subject,predicate,object_))
        
    def complete(self, subject, predicate, object_, statementUri, scope,flags):
        subject,predicate,object_,context=mapFtRdfStatements((subject,predicate,object_,statementUri,scope,None),self._g,flags)
        rt = []
        regex_needed = (isinstance(subject,REGEXTerm) or \
           isinstance(predicate,REGEXTerm) or \
           isinstance(object_,REGEXTerm) or \
           isinstance(context,REGEXTerm)) and not hasattr(self._g.store,'regex_matching')
        targetStore = regex_needed and REGEXMatching(self._g.store) or self._g.store
        if context is None:
            for (s,p,o),cg in ConjunctiveGraph(targetStore).triples((subject,predicate,object_)):
                for ctx in cg:
                    rt.append(mapRdfLibQuad((s,p,o,ctx)))
        elif isinstance(context,REGEXTerm):
            #Special case, we want capture the various graphs whose identifiers match the REGEX expression.
            #We don't want rdflib to confuse this as a query within a graph named by the REGEX expression
            #But a conjunctive query made with the REGEX expression
            for (s,p,o),cg in targetStore.triples((subject,predicate,object_),Graph(targetStore,context)):
                for ctx in cg:
                    rt.append(mapRdfLibQuad((s,p,o,ctx.identifier)))
        else:
            for s,p,o in Graph(targetStore,identifier=context).triples((subject,predicate,object_)):
                rt.append(mapRdfLibQuad((s,p,o,context)))
        return rt

    def size(self, scope):
        return len(self.getGraph(scope))

    def contains(self, subject, predicate, object_, statementUri, scope, flags):
        subj,pred,obj,context=mapFtRdfStatements((subject,predicate,object_,statementUri,scope,None),self._g,flags)
        regex_needed = False
        if isinstance(subj,REGEXTerm) or \
           isinstance(pred,REGEXTerm) or \
           isinstance(obj,REGEXTerm) or \
           isinstance(context,REGEXTerm):
            regex_needed = not hasattr(self._g.store,'regex_matching')
        targetStore = regex_needed and REGEXMatching(self._g.store) or self._g.store        
        return (subj,pred,obj) in self.getGraph(context,store=targetStore)

    def bind(self, object_, name, scope):
        raise NotImplementedError('Not Implemented')

    def unbind(self, name, scope):
        raise NotImplementedError('Not Implemented')

    def lookup(self, name, scope):
        raise NotImplementedError('Not Implemented')

    def keys(self, scope):
        raise NotImplementedError('Not Implemented')

    def has_key(self, name, scope):
        raise NotImplementedError('Not Implemented')

    ### Optimized interfaces are implemented as redirects to rdflib.Graph.triples
    ### http://rdflib.net/2005/10/10/rdflib-2.2.3/html/public/rdflib.Graph.Graph-class.html#triples
    ### This may be inefficient depending on the backend/store implementation

    def subjectsFromPredsAndObj(self, predicates, object_, scope):
        graph = self.getGraph(scope)
        predicates = predicates or []
        if scope:
            return Unique([s for s,p,o in graph.triples_choices((None,[mapFtRdfTerm(p) for p in predicates],mapFtRdfTerm(object_)))])
        else:
            return Unique([s for (s,p,o),cg in graph.triples_choices((None,[mapFtRdfTerm(p) for p in predicates],mapFtRdfTerm(object_)))])
    
    def subjectsFromPredAndObjs(self, predicate, objects, scope):
        """
        Get a list of resources with the given predicate and object
        """
        graph = self.getGraph(scope)
        objects = objects or []
        if scope:                
            return Unique([s for s,p,o in graph.triples_choices((None,mapFtRdfTerm(predicate),[mapFtRdfTerm(obj) for obj in objects]))])
        else:
            return Unique([s for (s,p,o),cg in graph.triples_choices((None,mapFtRdfTerm(predicate),[mapFtRdfTerm(obj) for obj in objects]))])
    
    def objectsFromSubsAndPredNonDistinct(self, subjects, predicate, scope):
        """
        Get a list of *non-distinct* objects with the given predicate and subjects
        """
        graph = self.getGraph(scope)
        subjects = subjects or []
        if scope:
            return [mapRdfLibObject(o) for s,p,o in graph.triples_choices(([mapFtRdfTerm(subj) for subj in subjects],mapFtRdfTerm(predicate),None))]
        else:
            return [mapRdfLibObject(o) for (s,p,o),cg in graph.triples_choices(([mapFtRdfTerm(subj) for subj in subjects],mapFtRdfTerm(predicate),None))]

    def objectsFromSubsAndPred(self, subjects, predicate, scope):
        """
        Get a list of obejcts with the given predicate and subjects
        """
        graph = self.getGraph(scope)
        subjects = subjects or []
        if scope:
            return Unique([mapRdfLibObject(o) for s,p,o in graph.triples_choices(([mapFtRdfTerm(subj) for subj in subjects],mapFtRdfTerm(predicate),None))])
        else:
            return Unique([mapRdfLibObject(o) for (s,p,o),cg in graph.triples_choices(([mapFtRdfTerm(subj) for subj in subjects],mapFtRdfTerm(predicate),None))])
    
    def objectsFromSubAndPreds(self, subject, predicates, scope):
        """
        Get a list of objects with the given predicates and subject
        """
        graph = self.getGraph(scope)
        predicates = predicates or []
        if scope:
            return Unique([mapRdfLibObject(o) for s,p,o in graph.triples_choices((mapFtRdfTerm(subject),[mapFtRdfTerm(p) for p in predicates],None))])
        else:
            return Unique([mapRdfLibObject(o) for (s,p,o),cg in graph.triples_choices((mapFtRdfTerm(subject),[mapFtRdfTerm(p) for p in predicates],None))])

    def isResource(self, res):
        return self.contains(res,None,None,None,None,{})

    def resources(self, scope):
        return Unique(self.subjectsFromPredAndObjs(None,None,scope))
